/*
 * uicode_tool - Manipulates Intel(R) IA32/x86_64 processor microcode bundles
 *
 * Copyright (c) 2010-2013 Henrique de Moraes Holschuh <hmh@hmh.eng.br>
 *               2000 Simon Trimmer, Tigran Aivazian
 *
 * Some code copied from microcode.ctl v1.17.
 * Some code based on the Linux kernel microcode_intel driver.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 */

#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <argp.h>
#include <dirent.h>
#include <time.h>

#include "intel_microcode.h"
#include "iucode_tool_config.h"

#define PROGNAME "iucode_tool"

/*
 * For micro-optimization on the hotter paths
 */
#define likely(x)       __builtin_expect(!!(x), 1)
#define unlikely(x)     __builtin_expect(!!(x), 0)

/* cpio archive constants */
#define LINUX_CPIO_UCODE_NAME "kernel/x86/microcode/GenuineIntel.bin"
#define LINUX_CPIO_FILE_MODE 0100644
#define LINUX_CPIO_DIR_MODE  0040755

/* Command-line processing and UI */

enum {
	EXIT_USAGE = 1,		/* Exit due to command line errors */
	EXIT_SWFAILURE = 2,	/* Exit due to software failure (ENOMEM, etc) */
};

static enum { /* actions */
	IUCODE_DO_UPLOADUC = 0x0001,
	IUCODE_DO_WRITEUC  = 0x0002,
	IUCODE_DO_WRITEFW  = 0x0004,
	IUCODE_DO_SELPROC  = 0x0008,
	IUCODE_DO_LOADFILE = 0x0010,
	IUCODE_DO_WRITEFWN = 0x0020,
	IUCODE_DO_WRITEFWE = 0x0040,

	IUCODE_F_UCSELECT  = 0x1000,

	IUCODE_DOMASK_NEEDSUC = IUCODE_DO_UPLOADUC
				| IUCODE_DO_WRITEUC
				| IUCODE_DO_WRITEFW
				| IUCODE_DO_WRITEFWN
				| IUCODE_DO_WRITEFWE,
} command_line_actions;

static char *progname;
static int verbosity = 1; /* 0 = errors, 1 = normal... */
static int strict_checks = 1;
static int ignore_bad_ucode = 0;
static int allow_downgrade = 0;
static int list_all_microcodes = 0;
static int list_sel_microcodes = 0;
static int unlink_files = 0;
static char *upload_microcode = NULL;
static char *write_microcode = NULL;
static char *write_early_firmware = NULL;
static char *write_firmware = NULL;
static char *write_named = NULL;

typedef enum { /* File type */
	INTEL_UC_FT_UNKNOWN = 0,
	INTEL_UC_FT_DAT,
	INTEL_UC_FT_BIN,
} intel_ucode_file_type_t;
static intel_ucode_file_type_t ucfiletype = INTEL_UC_FT_UNKNOWN;

/* linked list of filenames */
struct filename_list {
	intel_ucode_file_type_t type;
	struct filename_list *next;
	char path[];
};
static struct filename_list *input_files = NULL;

/* Intel Microcode data structures */

struct microcode_bundle {
	struct microcode_bundle *next;
	const char *filename;	/* source file name */
	unsigned long int id;   /* bundle id */
	unsigned long int size; /* bytes */
	const void *data;       /* binary file data */
};
static struct microcode_bundle *microcode_bundles = NULL;
static unsigned long int last_bundle_id = 0;

/* intel_uclist_entry signature flag constants */
enum {
    INTEL_UCLE_EXTSIG = 0x0001,	/* Sig came from extended table */
    INTEL_UCLE_SELID  = 0x0010,	/* Sig is candidate for loose datefilter */
    INTEL_UCLE_SELECT = 0x0020,	/* Sig selected */
};

/* signature single-linked list */
struct intel_uclist_entry {
	unsigned long int id;	/* microcode id within group */
	unsigned long int gid;	/* microcode group id */
	uint32_t flags;		/* INTEL_UCLE_ flags */
	uint32_t cpuid;		/* sorting key */
	uint32_t pf_mask;
	int32_t  uc_rev;	/* duplicated from header */
	const void *uc;
	uint32_t uc_size;
	struct intel_uclist_entry *next;
};
static struct intel_uclist_entry *all_microcodes = NULL;
static struct intel_uclist_entry *microcodes = NULL;

/* UI helpers */
#define UCODE_ID_MAX_LEN 20
struct microcode_interator_data {
	const struct microcode_bundle *current_bundle;
	unsigned long int total_signature_count;
	unsigned long int total_unique_sig_count;
	unsigned long int total_entry_count;
	unsigned long int current_uc;
	char current_uc_id_str[UCODE_ID_MAX_LEN];
};
static struct microcode_interator_data microcode_interator_data;

/* Filter masks */
struct microcode_filter_entry {
	uint32_t cpuid;		/* exact match */
	uint32_t pf_mask;	/* common bits set match */
	int invert;
	struct microcode_filter_entry *next;
};
static struct microcode_filter_entry *uc_filter_list = NULL;
static int filter_list_allow = 1;

static uint32_t datefilter_max = 0xffffffffU;  /* select dates before this */
static uint32_t datefilter_min = 0;	       /* select dates after this */
static int datefilter_loose = 0;

static int inline filter_list_active(void)
{
	return !!uc_filter_list ||
		!!(command_line_actions & IUCODE_F_UCSELECT);
}

/* Helpers */

#define print_msg_u(format, arg...) \
	do { fprintf(stderr, "%s: " format "\n", progname, ## arg); } while (0)

#define print_msg(level, format, arg...) \
	do { if (verbosity >= level) \
		fprintf(stderr, "%s: " format "\n", progname, ## arg); \
	} while (0)

#define print_err(format, arg...) \
	do { fprintf(stderr, "%s: " format "\n", progname, ## arg); } while (0)

static inline int is_dash(const char * const fn)
{
	return fn && *fn == '-' && !*(fn+1);
}

/* UI microcode id */

static void str_append_ucode_id(unsigned long int id,
				unsigned long int gid,
				char *str, unsigned int len)
{
	snprintf(str, len, "%02lu/%03lu", gid, id);
}

/* Signature linked list */

/**
 * free_uclist() - frees chain of struct intel_uclist_entry
 * @p:		pointer to the first struct intel_uclist_entry
 *
 * If p != NULL, frees all members of the struct intel_uclist_entry
 * linked list.
 */
static void free_uclist(struct intel_uclist_entry *p)
{
	struct intel_uclist_entry *e;

	while (p) {
		e = p;
		p = e->next;
		free(e);
	}
}

/*
 * remove entries that were superseeded
 */
static void __uclist_remove_shadowed(const unsigned long int gid,
				     const uint32_t cpuid,
				     const uint32_t pf_mask,
				     const int32_t uc_rev,
				     const int downgrade,
				     struct intel_uclist_entry **pnext)
{
	while (*pnext && (*pnext)->cpuid == cpuid) {
		struct intel_uclist_entry * const e = *pnext;

		if (((pf_mask & e->pf_mask) == e->pf_mask) &&
		    (uc_rev >= e->uc_rev || (downgrade && gid > e->gid))) {
			/* entry is a subset of the one we just added */
			/* delete entry */
			*pnext = e->next;
			free(e);
		} else {
			pnext = &(e->next);
		}
	}
}

/**
 * uclist_add_signature() - add signature to ucode sig list
 *
 * @id:		microcode id
 * @gid:	microcode group id
 * @flags:	INTEL_UCLE_* flags
 * @cpuid:	signature of the signature entry being added
 * @pf_mask:	pf_mask of the signature entry being added
 * @uc_rev:	revision of the microcode entry being added
 * @uc:		microcode data (including headers)
 * @uc_size:	microcode data size (total, including headers)
 * @strict:	validate microcode opaque data against duplicates
 * @uclist:	pointer to the first entry on the list
 *
 * Adds a microcode signature entry to the list.  Does not filter
 * out duplicates, nor superseed entries.
 *
 * If @strict is true, it will compare the opaque data with one
 * of the previously inserted duplicates (and return EBADF if it
 * doesn't match).
 *
 * returns: ENOMEM: could not allocate memory,
 *          EINVAL: bad @cpuid, @uc pointer or @uclist pointer
 *          EEXIST: entry already in list (will be inserted)
 *          EBADF : entry already in list, with different opaque data
 *          0:      entry not yet in list (will be inserted)
 */
static int uclist_add_signature(const unsigned long int id,
			const unsigned long int gid,
			const uint32_t flags,
			const uint32_t cpuid,
			const uint32_t pf_mask,
			const int32_t uc_rev,
			const void * const uc,
			const uint32_t uc_size,
			const int strict,
			struct intel_uclist_entry ** const uclist)
{
	struct intel_uclist_entry **pnext, *n;
	int res = 0;

	if (unlikely(!cpuid || !uc || !uclist))
		return EINVAL;

	pnext = uclist;
	n = NULL;

	/* search linked list for first insertion point */
	while (likely(*pnext && (*pnext)->cpuid < cpuid))
		pnext = &((*pnext)->next);

	/* look for duplicates */
	while (likely(*pnext) && (*pnext)->cpuid == cpuid) {
		if ((*pnext)->pf_mask == pf_mask && (*pnext)->uc_rev == uc_rev) {
			res = EEXIST;
			n = *pnext;
		}
		pnext = &((*pnext)->next);
	}

	if (strict && res == EEXIST) {
	    	int dup = intel_ucode_compare(uc, n->uc);
		if (dup == EINVAL || dup == EBADF)
			return dup;
	}

	n = malloc(sizeof(struct intel_uclist_entry));
	if (unlikely(!n))
		return ENOMEM;

	memset(n, 0, sizeof(struct intel_uclist_entry));
	n->id = id;
	n->gid = gid;
	n->cpuid = cpuid;
	n->pf_mask = pf_mask;
	n->uc_rev = uc_rev;
	n->uc = uc;
	n->uc_size = uc_size;
	n->flags = flags;

	/* prepend */
	n->next = *pnext;
	*pnext = n;

	return res;
}

/**
 * uclist_merge_signature() - merge signature into ucode sig list
 * @id:		microcode id
 * @gid:	microcode group id
 * @cpuid:	signature of the signature entry being added
 * @pf_mask:	pf_mask of the signature entry being added
 * @uc_rev:	revision of the microcode entry being added
 * @uc:		microcode data (including headers)
 * @uc_size:	microcode data size (total, including headers)
 * @downgrade:	version downgrades betwen groups allowed if NZ
 * @uclist:	pointer to the first entry on the list
 *
 * Adds a microcode signature entry to the list.  When @downgrade
 * is zero, it will replace any suitable entry that has a lesser
 * revision.  When @downgrade is non-zero, it will replace any
 * suitable entry that has a lesser revision or which is from
 * a smaller (earlier) microcode group.
 *
 * "suitable" means microcodes with the same cpuid and with a
 * pf_mask that is either equal or a strict subset of the one from
 * the new microcode.
 *
 * Note that it IS still possible that two signatures in the list
 * could apply to the same processor, if they have different revisions
 * and different pf masks, and the cpu is present in both pf masks.
 *
 * returns: ENOMEM: could not allocate memory,
 *          EEXIST: entry already in list (but may have been
 *                  inserted if @downgrade is non-zero)
 *          EINVAL: bad @cpuid, @uc pointer or @uclist pointer
 */
static int uclist_merge_signature(const unsigned long int id,
			const unsigned long int gid,
			const uint32_t cpuid,
			const uint32_t pf_mask,
			const int32_t uc_rev,
			const void * const uc,
			const uint32_t uc_size,
			const int downgrade,
			struct intel_uclist_entry ** const uclist)
{
	struct intel_uclist_entry **pnext, *n;

	if (unlikely(!cpuid || !uc || !uclist))
		return EINVAL;

	pnext = uclist;

	/* note: pf_mask *can* be zero on old processors */

	/* search linked list for matching CPUID */
	while (*pnext && (*pnext)->cpuid < cpuid)
		pnext = &((*pnext)->next);

	while (*pnext && (*pnext)->cpuid == cpuid) {
		struct intel_uclist_entry * const e = *pnext;

		if (likely(pf_mask && (e->pf_mask & pf_mask) == 0)) {
			/* disjoint sets, continue search */
			pnext = &(e->next);
		} else if (!downgrade || gid <= e->gid) { /* do not downgrade */
			if ((e->pf_mask & pf_mask) == pf_mask) {
				/* new set same or contained in the old set */
				if (e->uc_rev >= uc_rev) {
					/* old set as good, or newer */
					return EEXIST;
				} else if (pf_mask == e->pf_mask) {
					/* the sets are identical, and the new
					 * one is newer microcode: replace */
					e->id = id;
					e->gid = gid;
					/* e->cpuid = cpuid; */
					e->pf_mask = pf_mask;
					e->uc_rev = uc_rev;
					e->uc = uc;
					e->uc_size = uc_size;
					return 0;
				}
				pnext = &(e->next);
			} else if ((pf_mask & e->pf_mask) == e->pf_mask) {
				/* new entry a superset of the old */
				if (uc_rev >= e->uc_rev) {
					/* we can replace the old entry */
					e->id = id;
					e->gid = gid;
					/* e->cpuid = cpuid; */
					e->pf_mask = pf_mask;
					e->uc_rev = uc_rev;
					e->uc = uc;
					e->uc_size = uc_size;

					__uclist_remove_shadowed(gid, cpuid,
						     pf_mask, uc_rev,
						     downgrade, &(e->next));
					return 0;
				}
				pnext = &(e->next);
			}
		} else { /* downgrade */
			if (e->pf_mask == pf_mask) {
				int rc = (e->uc_rev == uc_rev) ? EEXIST : 0;
				/* same set, replace */
				e->id = id;
				e->gid = gid;
				/* e->cpuid = cpuid; */
				e->pf_mask = pf_mask;
				e->uc_rev = uc_rev;
				e->uc = uc;
				e->uc_size = uc_size;
				return rc;
			} else {
				/* sets are not disjoint: we have to insert it here */
				break;
			}
		}
	}

	n = malloc(sizeof(struct intel_uclist_entry));
	if (unlikely(!n))
		return ENOMEM;

	memset(n, 0, sizeof(struct intel_uclist_entry));
	n->id = id;
	n->gid = gid;
	n->cpuid = cpuid;
	n->pf_mask = pf_mask;
	n->uc_rev = uc_rev;
	n->uc = uc;
	n->uc_size = uc_size;

	/* prepend */
	n->next = *pnext;
	*pnext = n;

	__uclist_remove_shadowed(gid, cpuid, pf_mask, uc_rev, downgrade, &(n->next));

	return 0;
}

/* Microcode bundle handling */

/**
 * add_intel_microcode_bundle() - add a microcode bundle to set
 * @mcb_filename:	filename (metadata)
 * @mcb_id:		bundle id (metadata)
 * @mcb:		pointer to the microcode bundle data
 * @mcb_size:		size of the microcode bundle data
 *
 * Note: the memory pointed to by @mcb will be freed when
 * the bundles are freed by free_intel_microcode_bundles().
 *
 * Returns 0 if the bundled was added, or ENOMEM
 */
static int add_intel_microcode_bundle(const char * const mcb_filename,
				      unsigned long int mcb_id,
				      const void * const mcb,
				      const size_t mcb_size)
{
	struct microcode_bundle *b, **p;

	assert(mcb && mcb_size >= 1024);
	assert(mcb_filename);

	b = malloc(sizeof(struct microcode_bundle) + mcb_size);
	if (!b)
		return ENOMEM;

	memset(b, 0, sizeof(struct microcode_bundle));
	b->id = mcb_id;
	b->filename = strdup(mcb_filename);
	b->size = mcb_size;
	b->data = mcb;

	/* add to end of list */
	p = &microcode_bundles;
	while (*p)
		p = &((*p)->next);
	*p = b;

	return 0;
}

static void free_intel_microcode_bundles(void)
{
	struct microcode_bundle *p, *t;

	p = microcode_bundles;
	while (p) {
		t = p;
		p = t->next;
		free((void *)(t->filename));
		free((void *)(t->data));
		free(t);
	}

	microcode_bundles = NULL;
}

static int __load_intel_microcode_dat(FILE *fp, void **mcb, size_t *mcb_length)
{
	const size_t line_buffer_size = 2048;
	const size_t buffer_size_granularity = 256*1024; /* 1MiB, dwords */
	char *line_buffer = NULL;
	char *lp;
	size_t mcb_buflen;
	uint32_t *mcb_buffer = NULL;
	uint32_t *rp;
	size_t length;
	size_t pos = 0;
	int err;

	assert(mcb);
	assert(mcb_length);

	mcb_buflen = buffer_size_granularity;
	mcb_buffer = malloc(buffer_size_granularity * sizeof(uint32_t));
	line_buffer = malloc(line_buffer_size);
	if (!mcb_buffer || !line_buffer) {
		err = ENOMEM;
		goto err_out;
	}

	err = EINVAL;
	while (likely(fgets(line_buffer, line_buffer_size, fp) != NULL)) {
		/*
		 * Data lines are of the form "%x, %x, %x, %x,"
		 *
		 * other valid lines start with / (comments), and we also
		 * accept empty lines, and ignore whitespace.
		 */
		lp = line_buffer;
		while (isspace(*lp))
			lp++;

		if (!*lp || *lp == '/')
			continue;

		while (*lp) {
			char *ep;

			errno = 0;
			mcb_buffer[pos] = strtoul(lp, &ep, 0);
			if (unlikely(errno))
				goto err_out;
			if (lp == ep) {
				while (isspace(*lp))
					lp++;
				/* last value might have a , after it and
				 * we will hit it here */
				if (*lp == ',')
					lp++;
				while (isspace(*lp))
					lp++;
				if (unlikely(*lp))
					goto err_out;
				break; /* EOL */
			}

			/* we did read a value */
			pos++;
			if (unlikely(mcb_buflen < pos)) {
				/* expand buffer */
				mcb_buflen += buffer_size_granularity;
				rp = realloc(mcb_buffer,
					     mcb_buflen * sizeof(uint32_t));
				if (unlikely(!rp)) {
					err = ENOMEM;
					goto err_out;
				}
				mcb_buffer = rp;
			}

			lp = ep;
			/* next value */
			while (isspace(*lp))
				lp++;
			if (*lp == ',') {
				lp++;
				continue;
			} else if (unlikely(*lp)) {
				goto err_out;
			}
		}
	}

	length = pos * sizeof(uint32_t);
	if (length < 1024) {
		err = EINVAL;
		goto err_out;
	}

	/* truncate buffer if too large */
	rp = realloc(mcb_buffer, length);
	if (!rp) {
		err = ENOMEM;
		goto err_out;
	}

	*mcb = rp;
	*mcb_length = length;

	err = 0;

err_out:
	if (err) {
		free(mcb_buffer);
		*mcb = NULL;
		*mcb_length = 0;
	}

	free(line_buffer);
	return err;
}

static int __load_intel_microcode_bin(int fd, void **mcb, size_t *mcb_length)
{
	const size_t buffer_size_granularity = 1024*1024;

	int file_size_known;
	struct stat stat;

	size_t mcb_size;
	size_t mcb_space_left;
	uint8_t *mcb_buffer = NULL;
	uint8_t *rp;

	size_t pos;
	int err;

	assert(mcb);
	assert(mcb_length);

	/* Try to get file size beforehand */
	if (!fstat(fd, &stat) && S_ISREG(stat.st_mode) && stat.st_size > 0) {
		mcb_size = stat.st_size;
		file_size_known = 1;
	} else {
		mcb_size = buffer_size_granularity;
		file_size_known = 0;
	}

	/* sanity check size */
	if (mcb_size < 1024) {
		err = EINVAL;
		goto err_out;
	}

	mcb_buffer = malloc(mcb_size);
	if (!mcb_buffer) {
		err = ENOMEM;
		goto err_out;
	}

	err = 0;
	pos = 0;
	mcb_space_left = mcb_size;
	do {
		ssize_t rc;

		rc = read(fd, mcb_buffer + pos, mcb_space_left);
		if (rc == -1) {
			if (errno == EINTR)
				continue;
			err = -errno; /* negative! */
			goto err_out;
		}

		pos += rc;
		mcb_space_left -= rc;

		if (rc == 0)
			break; /* EOF */
		if (file_size_known && !mcb_space_left)
			break; /* Read entire file */

		if (!mcb_space_left) {
			mcb_size += buffer_size_granularity;
			mcb_space_left += buffer_size_granularity;
			rp = realloc(mcb_buffer, mcb_size);
			if (!rp) {
				err = ENOMEM;
				goto err_out;
			}
			mcb_buffer = rp;
		}
	} while (1);

	mcb_size = pos;
	if (mcb_size < 1024) {
		err = EINVAL;
		goto err_out;
	}

	rp = realloc(mcb_buffer, mcb_size);
	if (!rp) {
		err = ENOMEM;
		goto err_out;
	}

	*mcb = rp;
	*mcb_length = mcb_size;

err_out:
	if (err) {
		free(mcb_buffer);
		*mcb = NULL;
		*mcb_length = 0;
	}

	return err;
}

static int __load_intel_microcode_file(int fd,
				       FILE *fp,
				       const char * const fn,
				       const intel_ucode_file_type_t ftype)
{
	int err = 0;

	void *mcb = NULL;
	size_t mcb_length = 0;

	switch (ftype) {
	case INTEL_UC_FT_BIN:
		err = __load_intel_microcode_bin(fd, &mcb, &mcb_length);
		break;

	case INTEL_UC_FT_DAT:
		if (!fp)
			fp = fdopen(fd, "r");

		if (fp) {
			err = __load_intel_microcode_dat(fp, &mcb, &mcb_length);
		} else {
			err = -errno;
		}
		break;

	default:
		err = ENOTSUP;
	}

	if (err < 0) { /* error from read() or stdio */
		print_err("%s: could not read: %s", fn, strerror(-err));
		goto err_out;
	}

	if (!err && mcb && mcb_length) {
		last_bundle_id++;
		err = add_intel_microcode_bundle(fn, last_bundle_id,
						 mcb, mcb_length);
		if (!err)
			mcb = NULL; /* someone else will free it */
	}

	switch (err) {
	case 0:
		print_msg(2, "microcode bundle %lu: %s (%zu bytes)",
			  last_bundle_id, fn, mcb_length);
		break;
	case ENOMEM:
		print_err("%s: could not allocate memory while loading", fn);
		break;
	case EINVAL:
		print_err("%s: invalid file format", fn);
		break;
	}

err_out:
	if (fp)
		fclose(fp); /* also closes fd */
	else if (fd != -1)
		close(fd);

	return err;
}

static int load_intel_microcode(const char * path,
				const intel_ucode_file_type_t baseftype)
{
	int fd = -1;

	DIR *dir;
	int err;

	char fnbuf[PATH_MAX];
	const char *fn;
	intel_ucode_file_type_t ftype;

	assert(path);

	/* Should we read from stdin ? */
	if (is_dash(path)) {
		ftype = baseftype;

		/* default to .dat mode for stdin */
		if (ftype == INTEL_UC_FT_UNKNOWN)
			ftype = INTEL_UC_FT_DAT;

		return __load_intel_microcode_file(0, stdin, "(stdin)", ftype);
	}

	dir = NULL;
	fn = path;

	do {
		struct stat st;
		struct dirent *dentry;

		err = 0;

		if (fd != -1)
			close(fd);

		if (dir) {
			errno = 0;
			dentry = readdir(dir);
			if (dentry) {
				int s;

				if (dentry->d_name[0] == '.')
					continue;

				s = snprintf(fnbuf, sizeof(fnbuf), "%s/%s",
					     path, dentry->d_name);
				if (unlikely(s < 1 || s >= sizeof(fnbuf))) {
					print_err("%s/%s: path too long",
						  path, dentry->d_name);
					err = -ENAMETOOLONG;
					continue;
				}
				fn = fnbuf;
			} else {
				err = errno;
				if (unlikely(err)) {
					print_err("%s: cannot walk directory: %s",
						  path, strerror(err));
					err = -err;
				}
				break; /* finish/abort walk */
			}
		}

		fd = open(fn, O_RDONLY);
		if (unlikely(fd == -1)) {
			err = errno;
			print_err("%s: cannot open: %s", fn, strerror(err));
			err = -err;
			continue;
		}
		if (unlikely(fstat(fd, &st) == -1)) {
			err = errno;
			print_err("%s: cannot stat inode: %s", fn,
				  strerror(err));
			err = -err;
			continue;
		}
		if (S_ISDIR(st.st_mode)) {
			if (!dir) {
				dir = fdopendir(fd);
				if (!dir) {
					err = errno;
					print_err("%s: cannot open directory: %s",
						  path, strerror(err));
					err = -err;
					continue;
				}
				fd = -1;
			} else {
				print_msg(1, "%s: skipping nested directory: %s",
					  path, fn);
			}
			continue;
		}

		if (S_ISREG(st.st_mode) && st.st_size == 0)
			continue;

		ftype = baseftype;
		if (ftype == INTEL_UC_FT_UNKNOWN && S_ISREG(st.st_mode)) {
			char *p;

			/* try to guess based on extension */
			p = strrchr(fn, '.');
			if (p) {
				if (!strcasecmp(p + 1, "dat"))
					ftype = INTEL_UC_FT_DAT;
				/*
				 * Unneeded due to default to bin mode
				 *
				else if (!strcasecmp(p + 1, "bin"))
					ftype = INTEL_UC_FT_BIN;
				else if (!strcasecmp(p + 1, "fw"))
					ftype = INTEL_UC_FT_BIN;
				*/
			}
		}

		/* default to bin mode */
		if (ftype == INTEL_UC_FT_UNKNOWN)
			ftype = INTEL_UC_FT_BIN;

		err = __load_intel_microcode_file(fd, NULL, fn, ftype);
	} while (dir && (!err || ignore_bad_ucode));

	if (dir)
		closedir(dir);
	if (fd != -1)
		close(fd);
	return err;
}

/* Microcode write (binary format) and kernel upload */

static ssize_t __write_data(const int fd,
			    const void * const data,
			    const uint32_t size)
{
	const void *p;
	ssize_t len, s;

	len = size;
	if (len < 0)
		return -EINVAL;

	p = data;
	while (len > 0) {
		s = write(fd, p, len);
		if (s == -1) {
			if (errno != EINTR)
				return -errno;
		} else {
			p += s;
			len -= s;
		}
	}

	return size;
}

static void log_microcode_action(const char * const action,
				const char * const devname,
				const struct intel_uclist_entry * const uce)
{
	char id_str[UCODE_ID_MAX_LEN];

	str_append_ucode_id(uce->id, uce->gid, id_str, sizeof(id_str));
	print_msg_u("%s: %s microcode %s (sig 0x%08x, pf_mask 0x%02x, rev 0x%04x)",
		    devname, action, id_str,
		    uce->cpuid, uce->pf_mask, uce->uc_rev);
}

/**
 * upload_intel_microcodes() - uploads microcodes to kernel device
 * @devname:		device path
 * @uc_write_list:	uclist with the microcodes to write
 *
 * returns 0 if successful, or errno() if a problem happens
 */
static int upload_intel_microcodes(const char * const devname,
				  struct intel_uclist_entry *uc_write_list)
{
	int fd;
	struct stat stat;
	int err = 0;
	size_t total_written = 0;
	unsigned int entries_written = 0;
	ssize_t rc;

	assert(uc_write_list);
	assert(devname);

	fd = open(devname, O_WRONLY);
	if (fd == -1) {
		err = errno;
		print_err("%s: cannot open for writing: %s",
			  devname, strerror(err));
		return err;
	}
	/* Is it a char device? */
	if (fstat(fd, &stat) == -1) {
		err = errno;
		print_err("%s: cannot stat: %s", devname, strerror(err));
		return err;
	}
	if (!S_ISCHR(stat.st_mode)) {
		print_err("%s: not a character device", devname);
		return EINVAL;
	}

	while (uc_write_list && uc_write_list->uc) {
		if (verbosity >= 3)
			log_microcode_action("uploading",
					     devname, uc_write_list);

		rc = __write_data(fd, uc_write_list->uc,
				  uc_write_list->uc_size);
		if (rc < 0) {
			err = -rc;
			print_err("%s: write error: %s",
				  devname, strerror(err));
			break;
		} else {
			total_written += rc;
			uc_write_list = uc_write_list->next;
			entries_written++;
		}
	}

	if (close(fd)) {
		err = errno;
		print_err("%s: error while closing device: %s",
			  devname, strerror(err));
	}

	if (!err)
		print_msg(2, "%s: %u microcode entries uploaded, %zu bytes",
			  devname, entries_written, total_written);

	return err;
}

#define CPIO_STATIC_HDRSIZE (6 + 13*8)
#define CPIO_MAX_HDRSIZE (CPIO_STATIC_HDRSIZE + sizeof(LINUX_CPIO_UCODE_NAME))

/* if size is zero, assume it is a directory */
static int __write_cpio_hdrentry(int fd, const char * const name,
				 const size_t size)
{
	static int ino = 100;

	char buf[CPIO_MAX_HDRSIZE + 4];
	int nsize;
	int bufsize;

	assert(name);
	nsize = strlen(name) + 1;

	ino++;

	/* pad to DWORD. Caller guarantees last write was also padded */
	bufsize = CPIO_STATIC_HDRSIZE + nsize;
	while (bufsize % 4)
		bufsize++;

	memset(buf, 0, sizeof(buf));
	snprintf(buf, sizeof(buf),
	  "070701" /* signature, new ASCII portable format, no CRC */
	  "%08X%08X%08X%08X%08X%08lX%08zX%08X%08X%08X%08X%08X%08X%s",
	  ino,							   /* inode */
	  size ? LINUX_CPIO_FILE_MODE : LINUX_CPIO_DIR_MODE,	    /* mode */
	  0, 0,							/* uid, gid */
	  size ? 1 : 2, (long int) time(NULL), size,  /* nlink, mtime, size */
	  3, 1, 0, 0, nsize, 0,                   /* devj, devm, nsize, CRC */
	  name);                                               /* name, pad */

	return __write_data(fd, buf, bufsize);
}

/**
 * __write_cpio_header() - writes cpio header for early microcode
 * @fd:			file to write to
 * @size:		size of the file being written, for the header
 *
 * returns the number of bytes written if successful, or -errno()
 * if a problem happens
 *
 * Write cpio header.  For Intel microcodes, the kernel ABI
 * defines that they should be in a single file, named
 * kernel/x86/microcode/GenuineIntel.bin.  We use the 070701
 * format ("newc" format for GNU cpio).
 */
static int __write_cpio_header(int fd, size_t size)
{
	char fn[] = LINUX_CPIO_UCODE_NAME;
	char *p1, *p2;
	int s = 0;
	int r;

	p1 = fn;
	do {
		p2 = strchr(p1, '/');
		if (p2) {
			*p2 = 0;
			r = __write_cpio_hdrentry(fd, fn, 0);
			if (r < 0)
				return r;
			s += r;
			*p2 = '/';
			p1 = ++p2;
		}
	} while (p2);

	r = __write_cpio_hdrentry(fd, fn, size);
	if (r < 0)
		return r;

	s += r;
	return s;
}

/**
 * __write_cpio_trailer() - write a cpio EOF trailer and pad
 * @fd:			file to write to
 * @size:		file position/current size
 *
 * returns the number of bytes written if successful, or -errno()
 * if a problem happens
 *
 * Write a cpio EOF trailer, padding with NULs to the nearest
 * 16-byte boundary.  @size is used to calculate the amount of
 * padding, instead of requiring a lseek().
 */
static int __write_cpio_trailer(int fd, size_t size)
{
	const char cpio_trailer[] = "070701"
		/* inode, mode,  uid,   gid */
		"00000000" "00000000" "00000000" "00000000" 
		/* nlink, mtime, size,  devj */
		"00000001" "00000000" "00000000" "00000000"
		/* devm,  rdevj, rdevm, name size */
		"00000000" "00000000" "00000000" "0000000B"
		/* checksum, name */
		"00000000" "TRAILER!!!"
		/* 16 extra NULs for padding */
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0";

	size_t s = sizeof(cpio_trailer) - 16;
	while ((size + s) % 16)
		s++;

	return __write_data(fd, cpio_trailer, s);
}


/**
 * write_intel_microcodes() - writes microcodes to bin file
 * @filename:		file to write to
 * @ft:			file type
 * @uc_write_list:	uclist with the microcodes to write
 *
 * returns 0 if successful, or errno() if a problem happens
 *
 * @ft should be zero for binary format, 1 for linux early
 * initramfs cpio format.
 *
 * binary format is the Intel-specified format for binary microcode,
 * used by the Linux firmware loader and a few other operating
 * systems.
 *
 * Linux early initramfs cpio format will write microcodes inside
 * a cpio 070701 (New portable ASCII format without CRC) wrapper,
 * suitable for early microcode loading for Linux kernels v3.9 and
 * newer.  This file should be prepended to the compressed initramfs
 * image.
 */
static int write_intel_microcodes(const char * const filename, int ft,
				  struct intel_uclist_entry *uc_write_list)
{
	int fd;
	int err = 0;
	size_t total_written;
	unsigned int entries_written = 0;
	ssize_t rc;
	struct intel_uclist_entry *e;

	assert(uc_write_list);
	assert(filename);

	/* Unlink first */
	if (unlink_files) {
		if (unlink(filename) == -1) {
			if (errno != ENOENT) {
				err = errno;
				print_err("%s: cannot unlink: %s", filename,
					  strerror(err));
				return err;
			}
		} else {
			print_msg(3, "unlinked %s", filename);
		}
	}
	fd = open(filename, O_CREAT | O_WRONLY | O_EXCL,
			    S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH);
	if (fd == -1) {
		err = errno;
		print_err("%s: cannot write to, or create file: %s",
			  filename, strerror(err));
		return err;
	}

	/*
	 * precalculate raw file size. It is less annoying than
	 * having to seek back to write it later in cpio mode
	 */
	e = uc_write_list;
	total_written = 0;
	while (e && e->uc) {
		total_written += e->uc_size;
		e = e->next;
	}

	if (ft == 1) {
		rc = __write_cpio_header(fd, total_written);
		if (rc < 0)
			goto error;
		total_written += rc;
	}

	while (uc_write_list && uc_write_list->uc) {
		if (verbosity >= 3)
			log_microcode_action("writing",
					     filename, uc_write_list);

		rc = __write_data(fd, uc_write_list->uc,
				  uc_write_list->uc_size);
		if (unlikely(rc < 0))
			goto error;

		uc_write_list = uc_write_list->next;
		entries_written++;
	}

	if (ft == 1) {
		rc = __write_cpio_trailer(fd, total_written);
		if (unlikely(rc < 0))
			goto error;
		total_written += rc;
	}

	if (close(fd)) {
		err = errno;
		print_err("%s: error while closing file: %s",
			  filename, strerror(err));
	}

	if (!err)
		print_msg(2, "%s: %u microcode entries written, %zu bytes",
			  filename, entries_written, total_written);

	return err;

error:
	err = -rc;
	print_err("%s: write error: %s", filename, strerror(err));
	return err;
}

/* Microcode filtering */

#define PFMASK_MATCH_ANY 0xffffffffU

static int is_in_date_range(const struct intel_ucode_metadata * const m)
{
	const uint32_t d = (m->date_year << 16) | (m->date_month << 8) |
			   (m->date_day);
	return !!((d > datefilter_min) && (d < datefilter_max));
}

static void free_filter_list(struct microcode_filter_entry *f)
{
	static struct microcode_filter_entry *p;

	while (f) {
		p = f;
		f = f->next;
		free(p);
	}
}

static void __merge_filter(struct microcode_filter_entry * const f,
			 const uint32_t pf_mask, const int invert)
{
	if (f->invert == invert) {
		f->pf_mask |= pf_mask;
	} else if (!(f->invert)) {
		f->pf_mask &= ~pf_mask;
	} else {
		f->invert = 0;
		f->pf_mask = pf_mask;
	}
}

static int add_filter_to_list(uint32_t cpuid, uint32_t pf_mask, int invert,
				struct microcode_filter_entry ** const base)
{
	struct microcode_filter_entry **pnext, *n;

	assert(base);

	if (!pf_mask)
		pf_mask = PFMASK_MATCH_ANY;

	pnext = base;
	/* search for matching cpuid */
	while (*pnext && (*pnext)->cpuid < cpuid)
		pnext = &((*pnext)->next);

	if (*pnext && (*pnext)->cpuid == cpuid) {
		if (((pf_mask & (*pnext)->pf_mask) == pf_mask) &&
		     ((*pnext)->invert == invert)) {
			/* found equivalent, report it */
			return EEXIST;
		}

		__merge_filter(*pnext, pf_mask, invert);
		return 0;
	}

	/* insert before */
	n = malloc(sizeof(struct microcode_filter_entry));
	if (!n)
		return ENOMEM;
	n->cpuid = cpuid;
	n->pf_mask = pf_mask;
	n->invert = invert;
	n->next = *pnext;
	*pnext = n;

	return 0;
}

/* After calling this, "entries" becomes invalid */
static void add_filter_list_to_list(
			struct microcode_filter_entry ** const base,
			struct microcode_filter_entry *entries)
{
	struct microcode_filter_entry **pp;

	assert(base);

	/* both lists are sorted, no need to rescan from the
	 * beginning at every iteration */
	pp = base;
	while (entries) {
		struct microcode_filter_entry *e;

		e = entries;
		entries = entries->next;

		while (*pp && (*pp)->cpuid < e->cpuid)
			pp = &((*pp)->next);

		if (*pp && (*pp)->cpuid == e->cpuid) {
			__merge_filter(*pp, e->pf_mask, e->invert);
		} else {
			/* insert before */
			e->next = *pp;
			*pp = e;
			e = NULL;
		}
		if (e)
			free(e);
	}
}

/* returns non-zero if selected */
static int is_selected(uint32_t cpuid, uint32_t pf_mask,
			const struct microcode_filter_entry *f)
{
	while (f && f->cpuid < cpuid)
		f = f->next;
	while (f && f->cpuid == cpuid) {
		if ((pf_mask & f->pf_mask) != 0 ||
		    f->pf_mask == PFMASK_MATCH_ANY)
			return !(f->invert);
		f = f->next;
	}

	return filter_list_allow;
}

/* Microcode processing */

static int __uclist_add_print_errors(int status)
{
	switch (status) {
	case EEXIST:
	case 0:
		return 0;
	case ENOMEM:
		print_err("Cannot add index entry: out of memory");
		return 1;
	case EINVAL:
		print_err("Internal error uclist_merge_signature() returned EINVAL");
		return 1;
	default:
		return 1;
	}
}

static int __datefilter_loose_inplaceinsert(struct intel_uclist_entry **p,
					     struct intel_uclist_entry **uclist)
{
	const uint32_t cpuid = (*p)->cpuid;
	const uint32_t pfm   = (*p)->pf_mask;
	struct intel_uclist_entry *e;
	int rc;

	e = *uclist;
	while (e && e->cpuid < cpuid)
		e = e->next;
	*uclist = e; /* speed up next search */

	while (e && e->cpuid == cpuid) {
		if (e->flags & INTEL_UCLE_SELID && e->pf_mask & pfm) {
			e->flags &= ~INTEL_UCLE_SELID;
			e->flags |= INTEL_UCLE_SELECT;
			rc = __uclist_add_print_errors(
			        uclist_merge_signature(e->id, e->gid,
				        e->cpuid, e->pf_mask,
					e->uc_rev, e->uc, e->uc_size,
					allow_downgrade, p));
			if (rc)
				return rc;
		}
		e = e->next;
	}

	return 0;
}

static int __process_ucode_signature(void *userdata,
				unsigned int const sig_count,
				uint32_t const cpuid,
				uint32_t const pf_mask,
				const void * const uc_data,
				unsigned int const uc_data_size,
				const void * const uc,
				unsigned int const uc_size)
{
	struct microcode_interator_data * const ctx = userdata;
	struct intel_ucode_metadata m;
	intel_ucode_status_t s;
	int add_status;
	uint32_t uce_flags;

	assert(ctx);

	ctx->total_signature_count++;

	s = intel_ucode_getmetadata(uc, &m);
	if (s != INTEL_UCODE_NOERROR) {
		print_err("Microcode entry %s: %s",
			  ctx->current_uc_id_str, intel_ucode_errstr(s));
		return 1;
	}

	uce_flags = (sig_count)? INTEL_UCLE_EXTSIG : 0;
	if (is_selected(cpuid, pf_mask, uc_filter_list))
		uce_flags |= (is_in_date_range(&m)) ?
				INTEL_UCLE_SELECT : INTEL_UCLE_SELID;

	if (list_all_microcodes) {
		if (!sig_count)
			printf("  %6s: sig 0x%08x, pf mask 0x%02x, "
				"%04x-%02x-%02x, rev 0x%04x, size %u\n",
				ctx->current_uc_id_str, cpuid, pf_mask,
				m.date_year, m.date_month, m.date_day,
				m.revision, uc_size);
		else
			printf("          sig 0x%08x, pf mask 0x%02x, "
				"%04x-%02x-%02x, rev 0x%04x\n",
				cpuid, pf_mask,
				m.date_year, m.date_month, m.date_day,
				m.revision);
	}

	add_status = uclist_add_signature(ctx->current_uc,
			ctx->current_bundle->id, uce_flags,
			cpuid, pf_mask, m.revision, uc, uc_size,
			strict_checks, &all_microcodes);

	switch (add_status) {
	case 0:
		ctx->total_unique_sig_count++;
		break;
	case EEXIST:
		break;
	case EBADF:
		print_err("WARNING: Microcode %s has the same revision "
			  "and signature as a previously loaded microcode, "
			  "but different contents", ctx->current_uc_id_str);
		return 1;
	default:
		print_err("Failed to add microcode entry %s: %s",
			  ctx->current_uc_id_str, strerror(add_status));
		return 1;
	}

	if (uce_flags & INTEL_UCLE_SELECT) {
		add_status = uclist_merge_signature(ctx->current_uc,
					ctx->current_bundle->id,
					cpuid, pf_mask, m.revision,
					uc, uc_size, allow_downgrade,
					&microcodes);
		if (add_status && add_status != EEXIST) {
			print_err("Failed to select microcode entry %s: %s",
				  ctx->current_uc_id_str, strerror(add_status));
			return 1;
		}
	}

	return __uclist_add_print_errors(add_status);
}

static int __process_ucode_entry(void *userdata,
			         unsigned int const uc_count,
			         const void * const uc)
{
	struct microcode_interator_data * const ctx = userdata;
	intel_ucode_status_t s;

	assert(ctx);

	ctx->current_uc = uc_count;
	str_append_ucode_id(ctx->current_uc, ctx->current_bundle->id,
			    ctx->current_uc_id_str,
			    sizeof(ctx->current_uc_id_str));

	s = intel_ucode_check_microcode(uc, strict_checks);
	if (s != INTEL_UCODE_NOERROR) {
		print_err("microcode %s: %s", ctx->current_uc_id_str,
			  intel_ucode_errstr(s));
		return (ignore_bad_ucode) ? 0 : 1;
	}

	ctx->total_entry_count++;

	s = intel_ucode_foreach_signature(uc, __process_ucode_signature,
					  userdata);
	if (s != INTEL_UCODE_NOERROR && !ignore_bad_ucode) {
		print_err("aborting microcode processing...");
		return 1;
	}

	return 0;
}

static int do_process_microcodes(void)
{
	intel_ucode_status_t s;
	struct microcode_bundle *mcb;

	memset(&microcode_interator_data, 0, sizeof(microcode_interator_data));
	mcb = microcode_bundles;

	while (mcb) {
		if (list_all_microcodes) {
			if (mcb->filename) {
				printf("microcode bundle %lu: %s\n",
					mcb->id, mcb->filename);
			} else {
				printf("microcode bundle %lu:\n",
					mcb->id);
			}
		}
		microcode_interator_data.current_bundle = mcb;
		s = intel_ucode_foreach_microcode(mcb->data, mcb->size,
						__process_ucode_entry,
						&microcode_interator_data);
		if (s != INTEL_UCODE_NOERROR) {
			if (s != INTEL_UCODE_CALLBACK_ERROR)
				print_err("microcode bundle %s: %s",
					mcb->filename ? mcb->filename : "(no filename)",
					intel_ucode_errstr(s));

			if (!ignore_bad_ucode)
				return 1;
		}

		mcb = mcb->next;
	}

	print_msg(2, "processed %lu valid microcode(s), %lu signature(s), %lu unique signature(s)",
		  microcode_interator_data.total_entry_count,
		  microcode_interator_data.total_signature_count,
		  microcode_interator_data.total_unique_sig_count);

	/*
	 * we iterate over all selected microcodes only once, either
	 * to list selected microcodes, or to gather some stats when
	 * the verbosity is high, or to implement the loose date
	 * filtering mode.
	 */
	if (list_sel_microcodes || datefilter_loose || verbosity >= 2) {
		struct intel_uclist_entry *uce = microcodes;
		struct intel_uclist_entry *ucl = all_microcodes;
		unsigned long uccount = 0;
		unsigned long sigcount = 0;

		if (list_sel_microcodes)
			printf("selected microcodes:\n");

		while (uce) {
			/*
			 * search any extra microcode for loose mode and
			 * insert it at this point
			 */
			if (unlikely(datefilter_loose &&
			    __datefilter_loose_inplaceinsert(&uce, &ucl)))
					return 1;

			if (likely(!(uce->flags & INTEL_UCLE_EXTSIG)))
			    uccount++;

			sigcount++;

			if (list_sel_microcodes) {
				struct intel_ucode_metadata m;

				intel_ucode_getmetadata(uce->uc, &m);
				printf("%03lu: sig 0x%08x, pf mask 0x%02x, "
					"%04x-%02x-%02x, rev 0x%04x, size %u\n",
					sigcount, uce->cpuid, uce->pf_mask,
					m.date_year, m.date_month, m.date_day,
					m.revision, uce->uc_size);
			}

			uce = uce->next;
		}
		print_msg(2, "selected %lu microcode(s), %lu signature(s)",
			  uccount, sigcount);
	}

	return 0;
}

static int do_write_microcode(const char * const filename, int ft)
{
	if (!microcodes)
		return 0;

	print_msg(1, "Writing selected microcodes to: %s", filename);
	return write_intel_microcodes(filename, ft, microcodes);
}

static int do_upload_microcode(const char * const filename)
{
	if (!microcodes)
		return 0;

	print_msg(1, "Uploading selected microcodes to: %s", filename);
	return upload_intel_microcodes(filename, microcodes);
}

static int do_write_named(const char * const dirname)
{
	char fn[PATH_MAX];
	struct stat st;

	struct intel_uclist_entry e, *p;
	unsigned long int count;
	int rc;

	if (!microcodes)
		return 0;

	/* were we given a directory ? */
	if (stat(dirname, &st) == -1) {
		int err = errno;
		print_err("%s: cannot stat inode: %s", dirname,
			  strerror(err));
		return err;
	}
	if (!S_ISDIR(st.st_mode)) {
		print_err("%s: is not a directory", dirname);
		return EINVAL;
	}

	print_msg(1, "Writing microcode file(s) into %s", dirname);

	p = microcodes;
	count = 0;
	rc = 0;

	while (p && !rc) {
		/* write to file in dirname/ */
		rc = snprintf(fn, sizeof(fn), "%s/s%08X_m%08X_r%08X.fw", dirname,
			     p->cpuid, p->pf_mask, p->uc_rev);
		if (rc < 1 || rc >= sizeof(fn)) {
			print_err("%s: directory name too long, cannot create files inside",
				  dirname);
			return EINVAL;
		}

		memcpy(&e, p, sizeof(e));
		e.next = NULL;
		rc = write_intel_microcodes(fn, 0, &e);
		if (!rc)
			count++;

		p = p->next;
	}
	if (count)
		print_msg(2, "%lu file(s) were written into %s", count, dirname);
	else
		print_msg(1, "no files were written into %s", dirname);

	return rc;
}

static int do_write_firmware(const char * const dirname)
{
	char fn[PATH_MAX];
	struct stat st;

	struct intel_uclist_entry *samecpuid_list, *p;
	unsigned long int count;
	int rc;

	if (!microcodes)
		return 0;

	/* were we given a directory ? */
	if (stat(dirname, &st) == -1) {
		int err = errno;
		print_err("%s: cannot stat inode: %s", dirname,
			  strerror(err));
		return err;
	}
	if (!S_ISDIR(st.st_mode)) {
		print_err("%s: is not a directory", dirname);
		return EINVAL;
	}

	print_msg(1, "Writing microcode firmware file(s) into %s", dirname);

	p = microcodes;
	samecpuid_list = NULL;
	count = 0;
	rc = 0;

	/* the lists are already ordered by cpuid */
	while (p && !rc) {
		uint32_t cpuid;
		unsigned int x86_family, x86_model, x86_mask;
		int add_status;

		/* select all that share the same cpuid */
		cpuid = p->cpuid;
		while (p && p->cpuid == cpuid) {
			add_status = uclist_merge_signature(p->id, p->gid,
						p->cpuid, p->pf_mask,
						p->uc_rev, p->uc, p->uc_size,
						0, &samecpuid_list);
			if (__uclist_add_print_errors(add_status))
				return add_status;

			p = p->next;
		}

		/* write to file in dirname/ as expected by the kernel */

		x86_family = (cpuid >> 8) & 0x0f;
		x86_model  = (cpuid >> 4) & 0x0f;
		x86_mask   = cpuid & 0x0f;
		if (x86_family == 0x0f)
			x86_family += (cpuid >> 20) & 0xff;
		if (x86_family >= 0x06)
			x86_model += ((cpuid >> 16) & 0x0f) << 4;

		rc = snprintf(fn, sizeof(fn), "%s/%02x-%02x-%02x", dirname,
			     x86_family, x86_model, x86_mask);
		if (rc < 1 || rc >= sizeof(fn)) {
			print_err("%s: directory name too long, cannot create files inside",
				  dirname);
			return EINVAL;
		}

		rc = write_intel_microcodes(fn, 0, samecpuid_list);

		free_uclist(samecpuid_list);
		samecpuid_list = NULL;

		if (!rc)
			count++;
	}
	if (count)
		print_msg(2, "%lu file(s) were written into %s", count, dirname);
	else
		print_msg(1, "no files were written into %s", dirname);

	return rc;
}

static int scan_system_processors(void)
{
	char cpuid_device[PATH_MAX];
	uint32_t cpuid_buf[8];
	struct microcode_filter_entry *uc_cpu = NULL;

	int cpuid_fd = -1;
	unsigned int i = 0;
	unsigned int ncpu = 0;
	int rc = 0;

	while (1) {
		if (cpuid_fd != -1)
			close(cpuid_fd);

		snprintf(cpuid_device, sizeof(cpuid_device),
			 CPUID_DEVICE_BASE, i);
		cpuid_fd = open(cpuid_device, O_RDONLY);
		if (cpuid_fd == -1) {
			if (errno == EINTR)
				continue; /* try again */
			else if (errno == ENOENT)
				break; /* EOL */

			/* skip this processor */
			print_msg(2, "skipping processor %u (not online?): %d",
				  i, errno);
			i++;
			continue;
		}
		print_msg(3, "trying to get CPUID information from %s",
			  cpuid_device);
		if (read(cpuid_fd, &cpuid_buf, sizeof(cpuid_buf)) == -1) {
			print_err("%s: access to CPUID(0) and CPUID(1) failed",
				  cpuid_device);
			/* this is in the should not happen list, so abort */
			rc = 1;
			goto err_out;
		}
		close(cpuid_fd);
		cpuid_fd = -1;

		ncpu++;

		/* Is it a supported Intel processor ? */
		if (cpuid_buf[0] > 0 && /* we need cpuid(1) to exist */
		    cpuid_buf[1] == 0x756e6547 && /* "Genu" */
		    cpuid_buf[3] == 0x49656e69 && /* "ineI" */
		    cpuid_buf[2] == 0x6c65746e) { /* "ntel" */
			/*
			 * Get main processor signature from cpuid(1) EAX
			 * and add it as a filter.
			 */
			switch (add_filter_to_list(cpuid_buf[4], 0, 0, &uc_cpu)) {
			case ENOMEM:
				print_err("out of memory");
				rc = 1;
				goto err_out;
			case EEXIST:
				break;
			default:
				print_msg(1, "system has processor(s) with signature 0x%08x",
					  cpuid_buf[4]);
			}
		} else {
			if (!uc_cpu) {
				/* shortcut search, assume none will be found */
				print_msg(2, "non-Intel processor found, skipping scan");
				break;
			}
		}

		i++;
	};

	/*
	 * Be helpful: tell user why we're not doing what he explicitly
	 * requested if we cannot get any cpuids at all.
	 *
	 * fail-safe: we only switch away from "select all microcodes by default"
	 * if we managed to scan, so we will include all microcodes if cpuid is not
	 * available, and no other microcode selection option was used
	 */
	if (i == 0 && ncpu == 0) {
		print_err("cpuid kernel driver unavailable, cannot scan system processor signatures");
		rc = 0;
		goto err_out;
	}

	/* we managed to scan, do not select every microcode by default */
	filter_list_allow = 0;

	if (uc_cpu) {
		/* tie linked lists */
		add_filter_list_to_list(&uc_filter_list, uc_cpu);
		uc_cpu = NULL;
	}
	print_msg(2, "checked the signature of %u processor(s)", ncpu);

err_out:
	if (cpuid_fd != -1)
		close(cpuid_fd);

	if (uc_cpu)
		free_filter_list(uc_cpu);

	return rc;
}

/* Command line processing */

const char program_version[] =
	PROGNAME " " VERSION "\n"
	"Copyright (c) 2010-2013 by Henrique de Moraes Holschuh\n\n"

	"Based on code from the Linux microcode_intel driver and from\n"
	"the microcode.ctl package, copyright (c) 2000 by Simon Trimmer\n"
	"and Tigran Aivazian.\n\n"

	"This is free software; see the source for copying conditions.\n"
	"There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR\n"
	"A PARTICULAR PURPOSE.";
const char program_bug_address[] = PACKAGE_BUGREPORT;

static char cmdline_doc[] =
	PROGNAME " - Tool to manipulate Intel IA32/X86_64 microcode bundles\n"

	"\v"

	"The microcode bundle files should be specified as arguments.  "
	"The bundle type is determined by the file name suffix.  It "
	"defaults to the binary format.\n\n"

	"Should the filename end with \".bin\", binary mode will be "
	"used.  Should the filename end with \".dat\", text mode will be "
	"used.  The -t option can be used to set the type of the "
	"microcode bundle files that come after it, e.g. "
	"-td /tmp/dat-file -tb /tmp/binary /tmp/binary2.\n\n"

	"To load microcode data from stdin, use \"-\" as the filename.  "
	"File type will be assumed to be text (\".dat\"), use option -tb "
	"to load binary data from stdin.\n\n"

	"To load all files from a directory, specify the directory name.  "
	"It will not recurse into subdirectories, they will be skipped.\n\n"

	"Empty files and directories will be ignored, and will be skipped.";

enum {
	IUCODE_ARGP_STRICTCHK = 0x81,
	IUCODE_ARGP_NOSTRICTCHK,
	IUCODE_ARGP_IGNOREBROKEN,
	IUCODE_ARGP_NOIGNOREBROKEN,
	IUCODE_ARGP_UNLINK,
	IUCODE_ARGP_NOUNLINK,
	IUCODE_ARGP_DOWNGRADE,
	IUCODE_ARGP_NODOWNGRADE,
	IUCODE_ARGP_DATEBEFORE,
	IUCODE_ARGP_DATEAFTER,
	IUCODE_ARGP_DATEFSTRICT,
	IUCODE_ARGP_DATEFLOOSE,
	IUCODE_ARGP_EIRFS,
};

static struct argp_option cmdline_options[] = {
	{ NULL, 'h', NULL, 0, "Give this help list", -1 },

	{ "quiet",   'q', NULL, 0, "Quiet operation",                1 },
	{ "verbose", 'v', NULL, 0, "Verbose operation (cumulative)", 1 },

	{ NULL, 't', "type", 0,
	   "Sets input file type for the next microcode files. The type is "
	   "a single character: \"b\" (binary), \"d\" (Intel .dat), or \"a\" "
	   "(type will be selected by filename suffix)",
	  10 },

	{ NULL, 's', "! | [!]signature[,pf_mask]", 0,
	   "Select microcodes by the specificed signature and processor "
	   "flags mask.  Specify more than once to select/unselect more "
	   "microcodes.  Prefix with ! to unselect microcodes.  Use -s ! "
	   "to disable the default behaviour of selecting all microcodes "
	   "when no -s or -S filter is specified",
	  20 },
	{ "scan-system", 'S', NULL, 0,
	   "Select microcodes by scanning all online processors on "
	   "this system for their signatures.  Can be combined with the -s "
	   "option.  Note: this option has precedence over the -s option, "
	   "microcodes selected by --scan-system cannot be unselected by -s",
	  20 },

	{ "downgrade", IUCODE_ARGP_DOWNGRADE, NULL, 0,
	  "Instead of discarding microcodes based on revision level, "
	  "keep the one from the file loaded last.  Files are loaded "
	  "in the order they were specified in the command line",
	  25 },
	{ "no-downgrade", IUCODE_ARGP_NODOWNGRADE, NULL, 0,
	  "Keep the microcode with the highest revision level, regardless "
	  "of the file load order (default)",
	  25 },

	{ "date-before", IUCODE_ARGP_DATEBEFORE, "YYYY-MM-DD", 0,
	  "Select only microcodes older than the specified date",
	  27 },
	{ "date-after", IUCODE_ARGP_DATEAFTER, "YYYY-MM-DD", 0,
	  "Select only microcodes newer than the specified date",
	  27 },
	{ "loose-date-filtering", IUCODE_ARGP_DATEFLOOSE, NULL, 0,
	  "Consider for selection other revisions (outside of the date range) "
	  "of every microcode that was selected within the date range",
	  28 },
	{ "strict-date-filtering", IUCODE_ARGP_DATEFSTRICT, NULL, 0,
	  "Select only microcodes strictly within the date range (default)",
	  28 },

	{ "list",     'l', NULL, 0, "List selected microcode signatures", 30 },
	{ "list-all", 'L', NULL, 0, "List all microcode signatures",      30 },

	{ "kernel", 'k', "device", OPTION_ARG_OPTIONAL,
	   "Upload selected microcodes to the kernel.  Optionally, the "
	   "device path can be specified (default: " MICROCODE_DEVICE_DEFAULT
	   ")",
	  40 },
	{ "write-firmware", 'K', "directory", OPTION_ARG_OPTIONAL,
	   "Write selected microcodes with the filenames expected by the "
	   "Linux kernel firmware loader.  Optionally, the destination "
	   "directory can be specified (default: " MICROCODE_DIR_DEFAULT ")",
	  40 },
	{ "write-to", 'w', "file", 0,
	   "Write selected microcodes to a file in binary format.  "
	   "The binary format is suitable to be uploaded to the kernel",
	  40 },
	{ "write-named-to", 'W', "directory", 0,
	  "Write selected microcodes to files in the specified directory, "
	  "in binary format.  The file name will reflect the microcode "
	  "signature, mask and revision",
	  40 },
	{ "write-earlyfw", IUCODE_ARGP_EIRFS, "file", 0,
	  "Write selected microcodes to an early initramfs file, which "
	  "should be prepended to the regular initramfs",
	  40 },

	{ "overwrite", IUCODE_ARGP_UNLINK, NULL, 0,
	   "Unlink (remove) destination files before writing",
	  45 },
	{ "no-overwrite", IUCODE_ARGP_NOUNLINK, NULL, 0,
	   "Do not remove existing files (default)",
	  45 },

	{ "strict-checks", IUCODE_ARGP_STRICTCHK, NULL, 0,
	   "Perform strict checks on the microcode data (default)",
	  50 },
	{ "no-strict-checks", IUCODE_ARGP_NOSTRICTCHK, NULL, 0,
	   "Perform less strict checks on the microcode data",
	  51 },

	{ "ignore-broken", IUCODE_ARGP_IGNOREBROKEN, NULL, 0,
	  "Skip broken microcode entries instead of aborting",
	  55 },
	{ "no-ignore-broken", IUCODE_ARGP_NOIGNOREBROKEN, NULL, 0,
	  "Abort on broken microcode entries (default)",
	  56 },

	{ NULL, 0 },
};
static char cmdline_nonarg_doc[] = "[[-t<type>] filename] ...";

static int new_filename(const char * const fn,
			intel_ucode_file_type_t ftype)
{
	struct filename_list **p, *n;
	size_t s, l;

	l = strlen(fn);
	if (l > 1 && fn[l-1] == '/')
		l--;

	if (!l)
		return EINVAL;

	s = sizeof(struct filename_list) + l + 1;
	n = malloc(s);
	if (!n)
		return ENOMEM;

	memset(n, 0, s);
	memcpy(n->path, fn, l);
	n->type = ftype;

	/* tail-add */
	p = &input_files;
	while (*p)
		p = &((*p)->next);
	*p = n;

	return 0;
}

static void free_filename_list(void)
{
	struct filename_list *p, *q;

	p = input_files;

	while (p) {
		q = p;
		p = p->next;
		free(q);
	}
	input_files = NULL;
}

/* -s ! | [!]cpuid[,pf_mask] */
static int add_ucode_filter(const char *arg)
{
	char *p;
	uint32_t acpuid, amask;
	int invert;

	amask = 0;
	invert = 0;

	while (isspace(*arg))
		arg++;
	if (*arg == '!') {
		invert = 1;
		arg++;

		/* handle -s ! */
		if (isspace(*arg)) {
			while (isspace(*arg))
				arg++;
			if (! *arg)
			    	return EINVAL;
		}
		if (! *arg) {
			filter_list_allow = 0;
			return 0;
		}
	}

	errno = 0;
	acpuid = strtoul(arg, &p, 0);
	if (errno || p == arg || !*arg)
		return EINVAL;
	while (isspace(*p))
		p++;

	if (*p == ',') {
		p++;
		arg = p;
		errno = 0;
		amask = strtoul(arg, &p, 0);
		if (errno || p == arg || !*arg)
			return EINVAL;
		while (isspace(*p))
			p++;
		if (*p)
			return EINVAL;
	} else if (*p) {
		return EINVAL;
	}

	if (!invert)
		filter_list_allow = 0;

	return add_filter_to_list(acpuid, amask, invert, &uc_filter_list);
}

/* YYYY-MM-DD */
static int cmdline_get_date(const char *arg, uint32_t *d)
{
	unsigned long int year, month, day;
	char *p;

	while (isspace(*arg))
		arg++;

	errno = 0;
	year = strtoul(arg, &p, 10);
	if (errno || p == arg || *p != '-')
		return EINVAL;
	arg = ++p;
	month = strtoul(arg, &p, 10);
	if (errno || p == arg || *p != '-')
		return EINVAL;
	arg = ++p;
	day = strtoul(arg, &p, 10);
	if (errno || p == arg)
		return EINVAL;
	while (isspace(*p))
		p++;
	if (*p)
		return EINVAL;

	if (year > 9999 || month > 99 || day > 99)
		return EINVAL;

	/* Encode in BCD: YYYYMMDD */
	*d = (year / 1000)     << 28 |
	     (year / 100 % 10) << 24 |
	     (year / 10 % 10)  << 20 |
	     (year % 10)       << 16 |
	     (month / 10)      << 12 |
	     (month % 10)      << 8 |
	     (day / 10)        << 4 |
	     (day % 10);

	return 0;
}

static error_t cmdline_do_parse_arg(int key, char *arg,
				    struct argp_state *state)
{
	int rc;

	switch (key) {
	case 'h':
		argp_state_help(state, stdout, ARGP_HELP_STD_HELP);

	case 'q':
		verbosity = 0;
		break;
	case 'v':
		verbosity++;
		break;

	case 'L':
		list_all_microcodes = 1;
		break;
	case 'l':
		list_sel_microcodes = 1;
		break;

	case 't':
		if (!arg || strlen(arg) > 1)
			argp_error(state, "unknown file type: %s\n", arg);
		switch (*arg) {
		case 'd': /* .dat */
			ucfiletype = INTEL_UC_FT_DAT;
			break;
		case 'b': /* .bin */
			ucfiletype = INTEL_UC_FT_BIN;
			break;
		case 'a': /* any (detect) */
			ucfiletype = INTEL_UC_FT_UNKNOWN;
			break;
		default:
			argp_error(state, "unknown file type: %c\n", *arg);
		}
		break;

	case 'k':
		if (command_line_actions & IUCODE_DO_UPLOADUC)
			argp_error(state,
				   "-k option can be specified only once\n");

		if (arg)
			upload_microcode = strdup(arg);
		else
			upload_microcode = strdup(MICROCODE_DEVICE_DEFAULT);

		command_line_actions |= IUCODE_DO_UPLOADUC;
		break;
	case 'K':
		if (command_line_actions & IUCODE_DO_WRITEFW)
			argp_error(state,
				   "-K option can be specified only once\n");

		if (arg)
			write_firmware = strdup(arg);
		else
			write_firmware = strdup(MICROCODE_DIR_DEFAULT);

		command_line_actions |= IUCODE_DO_WRITEFW;
		break;
	case 'w':
		if (command_line_actions & IUCODE_DO_WRITEUC)
			argp_error(state,
				   "-w option can be specified only once\n");

		write_microcode = strdup(arg);
		command_line_actions |= IUCODE_DO_WRITEUC;
		break;
	case IUCODE_ARGP_EIRFS:
		if (command_line_actions & IUCODE_DO_WRITEFWE)
			argp_error(state,
				   "--write-earlyfw option can be specified only once\n");

		write_early_firmware = strdup(arg);
		command_line_actions |= IUCODE_DO_WRITEFWE;
		break;
	case 'W':
		if (command_line_actions & IUCODE_DO_WRITEFWN)
			argp_error(state,
				   "-W option can be specified only once\n");

		write_named = strdup(arg);

		command_line_actions |= IUCODE_DO_WRITEFWN;
		break;

	case IUCODE_ARGP_UNLINK:
		unlink_files = 1;
		break;
	case IUCODE_ARGP_NOUNLINK:
		unlink_files = 0;
		break;

	case 's':
		rc = add_ucode_filter(arg);
		switch (rc) {
		case 0:
		case EEXIST:
			break; /* success */
		case EINVAL:
			argp_error(state, "invalid filter: %s", arg);
		default:
			argp_failure(state, EXIT_SWFAILURE, rc,
				     "could not add filter '%s'", arg);
		}
		command_line_actions |= IUCODE_F_UCSELECT;
		break;
	case 'S':
		command_line_actions |= IUCODE_DO_SELPROC | IUCODE_F_UCSELECT;
		break;

	case IUCODE_ARGP_DATEBEFORE:
	case IUCODE_ARGP_DATEAFTER:
		if (cmdline_get_date(arg, (key == IUCODE_ARGP_DATEBEFORE) ?
					   &datefilter_max : &datefilter_min))
			argp_error(state, "invalid date: %s", arg);
		command_line_actions |= IUCODE_F_UCSELECT;
		break;
	case IUCODE_ARGP_DATEFSTRICT:
		datefilter_loose = 0;
		break;
	case IUCODE_ARGP_DATEFLOOSE:
		datefilter_loose = 1;
		break;

	case IUCODE_ARGP_STRICTCHK:
		strict_checks = 1;
		break;
	case IUCODE_ARGP_NOSTRICTCHK:
		strict_checks = 0;
		break;
	case IUCODE_ARGP_IGNOREBROKEN:
		ignore_bad_ucode = 1;
		break;
	case IUCODE_ARGP_NOIGNOREBROKEN:
		ignore_bad_ucode = 0;
		break;
	case IUCODE_ARGP_DOWNGRADE:
		allow_downgrade = 1;
		break;
	case IUCODE_ARGP_NODOWNGRADE:
		allow_downgrade = 0;
		break;

	case ARGP_KEY_ARG: /* NON-OPTION ARGUMENTS */
		rc = new_filename(arg, ucfiletype);
		if (rc)
			argp_failure(state, EXIT_SWFAILURE, rc,
				     "could not add path '%s'", arg);
		command_line_actions |= IUCODE_DO_LOADFILE;
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}

	return 0;
}

static struct argp cmdline_argp = {
	.options  = cmdline_options,
	.parser   = cmdline_do_parse_arg,
	.args_doc = cmdline_nonarg_doc,
	.doc      = cmdline_doc };

int main(int argc, char *argv[])
{
	int rc;

	progname = argv[0];

	argp_err_exit_status = EXIT_USAGE;
	argp_program_version = program_version;
	argp_program_bug_address = NULL; /* FIXME */
	argp_parse(&cmdline_argp, argc, argv, ARGP_IN_ORDER, 0, NULL);

	if (!command_line_actions) {
		print_msg(1, "nothing to do...");
		goto do_nothing;
	}

	if ((command_line_actions & IUCODE_DO_SELPROC) &&
	    scan_system_processors()) {
		rc = EXIT_SWFAILURE;
		goto err_exit;
	}

	if (command_line_actions & IUCODE_DO_LOADFILE) {
		struct filename_list *fn = input_files;
		while (fn) {
			switch (load_intel_microcode(fn->path, fn->type)) {
			case 0:
				break;
			case ENOTSUP:
				rc = EXIT_USAGE;
				goto err_exit;
			default:
				if (!ignore_bad_ucode) {
					rc = EXIT_SWFAILURE;
					goto err_exit;
				}
			}
			fn = fn->next;
		}

		if (microcode_bundles && do_process_microcodes()) {
			rc = EXIT_SWFAILURE;
			goto err_exit;
		}
	}

	if (command_line_actions & IUCODE_DOMASK_NEEDSUC) {
		if (microcodes) {
			rc = 0;
			if (upload_microcode)
				rc = do_upload_microcode(upload_microcode);
			if (!rc && write_microcode)
				rc = do_write_microcode(write_microcode, 0);
			if (!rc && write_early_firmware)
				rc = do_write_microcode(write_early_firmware, 1);
			if (!rc && write_firmware)
				rc = do_write_firmware(write_firmware);
			if (!rc && write_named)
				rc = do_write_named(write_named);

			if (rc) {
				rc = EXIT_SWFAILURE;
				goto err_exit;
			}
		} else {
			if (filter_list_active()) {
				print_msg(1, "No valid microcodes were selected, nothing to do...");
			} else {
				print_msg(1, "No valid microcodes were loaded, nothing to do...");
			}
		}
	}

do_nothing:
	rc = 0;
err_exit:

	/* Disable all of this cleanup for some speedup... */
#ifdef VALGRIND_BUILD
	free(write_microcode);
	free(upload_microcode);
	free(write_firmware);
	free(write_named);

	free_filter_list(uc_filter_list);
	uc_filter_list = NULL;
	free_uclist(microcodes);
	microcodes = NULL;
	free_uclist(all_microcodes);
	all_microcodes = NULL;
	free_intel_microcode_bundles();
	free_filename_list();
#endif /* VALGRIND_BUILD */

	return rc;
}
