/*********************************************************************
 *
 *      Copyright (C) 2000-2004 Bill Smith
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: setCommand.java 1252 2004-05-10 02:09:02Z nfiedler $
 *
 ********************************************************************/

package com.bluemarsh.jswat.command;

import com.bluemarsh.jswat.ContextManager;
import com.bluemarsh.jswat.FieldNotObjectException;
import com.bluemarsh.jswat.Log;
import com.bluemarsh.jswat.Session;
import com.bluemarsh.jswat.expr.EvaluationException;
import com.bluemarsh.jswat.expr.Evaluator;
import com.bluemarsh.jswat.ui.UIAdapter;
import com.bluemarsh.jswat.util.VariableValue;
import com.bluemarsh.jswat.util.Strings;
import com.bluemarsh.jswat.util.Variables;
import com.sun.jdi.AbsentInformationException;
import com.sun.jdi.ArrayType;
import com.sun.jdi.BooleanValue;
import com.sun.jdi.BooleanType;
import com.sun.jdi.ByteType;
import com.sun.jdi.ByteValue;
import com.sun.jdi.CharType;
import com.sun.jdi.CharValue;
import com.sun.jdi.ClassNotLoadedException;
import com.sun.jdi.ClassType;
import com.sun.jdi.DoubleType;
import com.sun.jdi.DoubleValue;
import com.sun.jdi.FloatValue;
import com.sun.jdi.FloatType;
import com.sun.jdi.IncompatibleThreadStateException;
import com.sun.jdi.IntegerType;
import com.sun.jdi.IntegerValue;
import com.sun.jdi.InvalidStackFrameException;
import com.sun.jdi.InvalidTypeException;
import com.sun.jdi.LongType;
import com.sun.jdi.LongValue;
import com.sun.jdi.ObjectCollectedException;
import com.sun.jdi.ReferenceType;
import com.sun.jdi.ShortType;
import com.sun.jdi.ShortValue;
import com.sun.jdi.StackFrame;
import com.sun.jdi.ThreadReference;
import com.sun.jdi.Type;
import com.sun.jdi.Value;

/**
 * Defines the class that handles the 'set' command.
 *
 * @author  Bill Smith
 */
public class setCommand extends JSwatCommand {

    /**
     * Perform the 'set' command.
     *
     * @param  session  JSwat session on which to operate.
     * @param  args     Tokenized string of command arguments.
     * @param  out      Output to write messages to.
     */
    public void perform(Session session, CommandArguments args, Log out) {
        // Check for active session.
        if (!session.isActive()) {
            throw new CommandException(Bundle.getString("noActiveSession"));
        }

        // Get the current thread.
        ContextManager ctxtman = (ContextManager)
            session.getManager(ContextManager.class);
        ThreadReference thread = ctxtman.getCurrentThread();
        if (thread == null) {
            throw new CommandException(Bundle.getString("noCurrentThread"));
        }
        // Check for enough arguments.
        if (args.countTokens() < 3) {
            // 'obj = value' is three arguments
            throw new MissingArgumentsException();
        }

        try {
            String lvalue = args.nextToken();
            String eq = args.nextToken();
            if (!eq.equals("=")) {
                throw new CommandException(
                    Bundle.getString("set.missingEquals"));
            }
            // We process the rvalue manually, so return it as-is.
            args.returnAsIs(true);
            String rvalue = args.rest();
            setValue(lvalue, thread, ctxtman.getCurrentFrame(), rvalue);
            UIAdapter uiadapter = session.getUIAdapter();
            uiadapter.refreshDisplay();
        } catch (AbsentInformationException aie) {
            throw new CommandException(
                Bundle.getString("noVariableInfo1") + '\n'
                + Bundle.getString("noVariableInfo2"));
        } catch (ClassNotLoadedException cnle) {
            // How can this happen?
            throw new CommandException(cnle.toString(), cnle);
        } catch (EvaluationException ee) {
            throw new CommandException(Bundle.getString("evalError") + ' '
                                       + ee.getMessage(), ee);
        } catch (IllegalArgumentException iae) {
            throw new CommandException(iae.toString(), iae);
        } catch (IncompatibleThreadStateException itse) {
            throw new CommandException(Bundle.getString("threadNotSuspended"),
                                       itse);
        } catch (IndexOutOfBoundsException ioobe) {
            throw new CommandException(Bundle.getString("invalidStackFrame"),
                                       ioobe);
        } catch (NoSuchFieldException nsfe) {
            throw new CommandException(Bundle.getString("fieldNotFound")
                                       + ": " + nsfe.getMessage(), nsfe);
        } catch (FieldNotObjectException fnoe) {
            throw new CommandException(Bundle.getString("fieldNotObject")
                                       + '\n' + fnoe.toString(), fnoe);
        } catch (InvalidStackFrameException isfe) {
            throw new CommandException(Bundle.getString("invalidStackFrame"),
                                       isfe);
        } catch (InvalidTypeException ite) {
            throw new CommandException(Bundle.getString("set.invalidType")
                                       + ' ' + ite.getMessage(), ite);
        } catch (Exception e) {
            throw new CommandException(e);
        }
    } // perform

    /**
     * Set a named variable, using the stack frame of the given thread.
     *
     * @param  lValueString  name of variable to be set.
     * @param  thread        current thread.
     * @param  frame         stack frame index.
     * @param  rValueString  value to be set.
     * @throws  AbsentInformationException
     *          if class doesn't have local variable info.
     * @throws  ClassNotLoadedException
     *          if the object's class is not loaded.
     * @throws  EvaluationException
     *          if right-side expression was invalid.
     * @throws  FieldNotObjectException
     *          if a non-object is encountered.
     * @throws  IllegalArgumentException
     *          if an argument value is not allowed.
     * @throws  IllegalThreadStateException
     *          if thread is not currently running.
     * @throws  IncompatibleThreadStateException
     *          if thread is not suspended.
     * @throws  IndexOutOfBoundsException
     *          if <code>frame</code> is out of bounds.
     * @throws  InvalidStackFrameException
     *          if <code>frame</code> is out of bounds.
     * @throws  InvalidTypeException
     *          if type is invalid.
     * @throws  NoSuchFieldException
     *          if the field was not found in the object.
     * @throws  ObjectCollectedException
     *          if the referenced object has been collected.
     */
    private void setValue(String lValueString, ThreadReference thread,
                          int frame, String rValueString)
        throws AbsentInformationException,
               ClassNotLoadedException,
               EvaluationException,
               FieldNotObjectException,
               IllegalArgumentException,
               IllegalThreadStateException,
               IncompatibleThreadStateException,
               IndexOutOfBoundsException,
               InvalidStackFrameException,
               InvalidTypeException,
               NoSuchFieldException,
               ObjectCollectedException {

        // Get stack frame.
        StackFrame stackFrame = thread.frame(frame);
        if (stackFrame == null) {
            // Thread is not currently running.
            throw new IllegalThreadStateException("thread not running");
        }

        // Get the referenced field or local variable.
        VariableValue varValue = Variables.getField(lValueString,
                                                    thread, frame);
        Type lvalType;
        if (varValue.value() != null) {
            // Use the most reliable means of getting the variable's type.
            lvalType = varValue.value().type();
        } else {
            if (varValue.field() != null) {
                // It's a field of an object.
                // - varValue.field
                // - varValue.object
                // - varValue.value (may be null)
                lvalType = varValue.field().type();
            } else {
                // It's a local variable.
                // - varValue.localVar
                // - varValue.value (may be null)
                lvalType = varValue.localVariable().type();
            }
            if (varValue.arrayRef() != null) {
                lvalType = ((ArrayType) lvalType).componentType();
            }
        }

        // Now evaluate the rvalue.
        Evaluator eval = new Evaluator(rValueString);
        Object rValue = eval.evaluate(thread, frame);

        // Re-acquire the stack frame in case a method invocation in the
        // expression has invalidated it.
        stackFrame = thread.frame(frame);

        // Create a Value object based on the target variable's type.
        Value valueObj = null;
        if (rValue == null) {
            // valueObj = null
        } else if (lvalType instanceof BooleanType) {
            if (rValue instanceof Boolean) {
                Boolean bool = (Boolean) rValue;
                valueObj = thread.virtualMachine().mirrorOf(
                    bool.booleanValue());
            } else if (rValue instanceof BooleanValue) {
                valueObj = (BooleanValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected boolean, got " + rValue.getClass());
            }

        } else if (lvalType instanceof ByteType) {
            if (rValue instanceof Byte) {
                Byte byt = (Byte) rValue;
                valueObj = thread.virtualMachine().mirrorOf(
                    byt.byteValue());
            } else if (rValue instanceof ByteValue) {
                valueObj = (ByteValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected byte, got " + rValue.getClass());
            }

        } else if (lvalType instanceof CharType) {
            if (rValue instanceof Character) {
                Character v = (Character) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.charValue());
            } else if (rValue instanceof CharValue) {
                valueObj = (CharValue) rValue;
            } else if (rValue instanceof Short) {
                Short v = (Short) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.shortValue());
            } else if (rValue instanceof ShortValue) {
                valueObj = (ShortValue) rValue;
            } else if (rValue instanceof Byte) {
                Byte byt = (Byte) rValue;
                valueObj = thread.virtualMachine().mirrorOf(
                    byt.byteValue());
            } else if (rValue instanceof ByteValue) {
                valueObj = (ByteValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected char, got " + rValue.getClass());
            }

        } else if (lvalType instanceof DoubleType) {
            if (rValue instanceof Double) {
                Double v = (Double) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.doubleValue());
            } else if (rValue instanceof DoubleValue) {
                valueObj = (DoubleValue) rValue;
            } else if (rValue instanceof Float) {
                Float v = (Float) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.floatValue());
            } else if (rValue instanceof FloatValue) {
                valueObj = (FloatValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected double, got " + rValue.getClass());
            }

        } else if (lvalType instanceof FloatType) {
            if (rValue instanceof Float) {
                Float v = (Float) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.floatValue());
            } else if (rValue instanceof FloatValue) {
                valueObj = (FloatValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected float, got " + rValue.getClass());
            }

        } else if (lvalType instanceof IntegerType) {
            if (rValue instanceof Integer) {
                Integer v = (Integer) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.intValue());
            } else if (rValue instanceof IntegerValue) {
                valueObj = (IntegerValue) rValue;
            } else if (rValue instanceof Short) {
                Short v = (Short) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.shortValue());
            } else if (rValue instanceof ShortValue) {
                valueObj = (ShortValue) rValue;
            } else if (rValue instanceof Byte) {
                Byte byt = (Byte) rValue;
                valueObj = thread.virtualMachine().mirrorOf(
                    byt.byteValue());
            } else if (rValue instanceof ByteValue) {
                valueObj = (ByteValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected integer, got " + rValue.getClass());
            }

        } else if (lvalType instanceof LongType) {
            if (rValue instanceof Long) {
                Long v = (Long) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.longValue());
            } else if (rValue instanceof LongValue) {
                valueObj = (LongValue) rValue;
            } else if (rValue instanceof Integer) {
                Integer v = (Integer) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.intValue());
            } else if (rValue instanceof IntegerValue) {
                valueObj = (IntegerValue) rValue;
            } else if (rValue instanceof Short) {
                Short v = (Short) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.shortValue());
            } else if (rValue instanceof ShortValue) {
                valueObj = (ShortValue) rValue;
            } else if (rValue instanceof Byte) {
                Byte byt = (Byte) rValue;
                valueObj = thread.virtualMachine().mirrorOf(
                    byt.byteValue());
            } else if (rValue instanceof ByteValue) {
                valueObj = (ByteValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected long, got " + rValue.getClass());
            }

        } else if (lvalType instanceof ShortType) {
            if (rValue instanceof Short) {
                Short v = (Short) rValue;
                valueObj = thread.virtualMachine().mirrorOf(v.shortValue());
            } else if (rValue instanceof ShortValue) {
                valueObj = (ShortValue) rValue;
            } else if (rValue instanceof Byte) {
                Byte byt = (Byte) rValue;
                valueObj = thread.virtualMachine().mirrorOf(
                    byt.byteValue());
            } else if (rValue instanceof ByteValue) {
                valueObj = (ByteValue) rValue;
            } else {
                throw new EvaluationException(
                    Bundle.getString("set.wrongType")
                    + " expected short, got " + rValue.getClass());
            }

        } else if (lvalType instanceof ReferenceType) {
            // Assume it is a string for now.
            String s = rValue.toString();
            if (s.length() > 0) {
                s = Strings.trimQuotes(s);
            }
            valueObj = thread.virtualMachine().mirrorOf(s);

        } else {
            throw new IllegalArgumentException(
                Bundle.getString("set.nocomposite"));
        }

        // Set the variable.
        try {
            if (varValue.arrayRef() != null) {
                varValue.arrayRef().setValue(varValue.arrayIndex(),
                                             valueObj);
            } else if (varValue.field() == null) {
                // lvalue was a primitive on the stack.
                stackFrame.setValue(varValue.localVariable(), valueObj);
            } else if (varValue.object() == null) {
                // lvalue was a static field of the containing class.
                ReferenceType refType =
                    stackFrame.location().declaringType();
                if (refType instanceof ClassType) {
                    ClassType clazz = (ClassType) refType;
                    clazz.setValue(varValue.field(), valueObj);
                } else {
                    throw new IllegalArgumentException(
                        "cannot set field of abstract class");
                }
            } else {
                // lvalue was a field (or sub-field) of 'this' or of
                // some other variable.
                varValue.object().setValue(
                    varValue.field(), valueObj);
            }
        } catch (InvalidTypeException ite) {
            throw new InvalidTypeException("value: " + rValueString +
                                           ", type: " + valueObj.type().name() +
                                           ", expected: " + lvalType.name());
        }
    } // setValue
} // setCommand
