/*********************************************************************
 *
 *      Copyright (C) 2002-2003 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: TreeBuilder.java 1082 2003-12-03 08:18:08Z nfiedler $
 *
 ********************************************************************/

package com.bluemarsh.jswat.expr;

import com.bluemarsh.jswat.parser.java.analysis.AnalysisAdapter;
import com.bluemarsh.jswat.parser.java.node.*;
import com.bluemarsh.jswat.util.Strings;
import java.util.EmptyStackException;
import java.util.Stack;

/**
 * Class TreeBuilder builds out the abstract syntax tree based on the
 * tokenized infix expression.
 *
 * @author  Nathan Fiedler
 */
class TreeBuilder extends AnalysisAdapter {
    /** Parser expects anything. */
    private static final int START_STATE = 0;
    /** Parser expects an argument. */
    private static final int ARGUMENT_STATE = 1;
    /** Parser expects an operator. */
    private static final int OPERATOR_STATE = 2;
    /** Parser expects an identifier to append to the previous one. */
    private static final int APPEND_STATE = 3;
    /** Current parser state, one of the STATE constants. */
    private int searchState;
    /** True if processing is completed, either because the end of
     * the expression was reached, or there was an error. */
    private boolean doneProcessing;
    /** Indicates if an error occurred; null means no error. */
    private EvaluationException evalException;
    /** Root node of the tree. */
    private RootNode rootNode;
    /** Stack of arguments. */
    private Stack argumentStack;
    /** Stack of operators. */
    private Stack operatorStack;
    /** Counter for assisting in validating the expression. If non-zero,
     * parsing the (n-1)th nested method invocation. */
    private int methodCount;

    // Infix expressions can be parsed from a series of tokens using two
    // stacks. One stack is used to hold parse trees under construction
    // (the argument stack), the other to hold operators (and left
    // parentheses, for matching purposes; the operator stack).

    // As we read in each new token (from left to right), we either push
    // the token (or a related tree) onto one of the stacks, or we
    // reduce the stacks by combining an operator with some arguments.
    // Along the way, it will be helpful to maintain a search state
    // which tells us whether we should see an argument or operator next
    // (the search state helps us to reject malformed expressions).

    /**
     * Translates the given string to a character. Handles character
     * escapes such as \r and Unicode escapes.
     *
     * @param  charStr  string representing a character.
     * @return  the character.
     */
    protected static char translateChar(String charStr) {
        if (charStr.length() == 0) {
            throw new IllegalArgumentException("empty character");
        }
        // May just be a single character.
        if (charStr.length() == 1) {
            return charStr.charAt(0);
        }
        if (charStr.charAt(0) == '\\') {
            char ch = charStr.charAt(1);
            if (ch == 'b') {
                return '\b';
            } else if (ch == 'f') {
                return '\f';
            } else if (ch == 't') {
                return '\t';
            } else if (ch == 'n') {
                return '\n';
            } else if (ch == 'r') {
                return '\r';

            } else if (ch == 'u') {
                // Unicode escape.
                String hex = charStr.substring(2);
                try {
                    int i = Integer.parseInt(hex, 16);
                    return (char) i;
                } catch (NumberFormatException nfe) {
                    throw new IllegalArgumentException("invalid Unicode: "
                                                       + hex);
                }

            } else if (ch >= '0' && ch <= '3') {
                // Octal escape.
                String octal = charStr.substring(1);
                try {
                    int i = Integer.parseInt(octal, 8);
                    return (char) i;
                } catch (NumberFormatException nfe) {
                    throw new IllegalArgumentException("invalid octal: "
                                                       + octal);
                }
            } else {
                return ch;
            }
        } else {
            throw new IllegalArgumentException("not a character: " + charStr);
        }
    } // translateChar

    /**
     * Constructs a TreeBuilder.
     */
    public TreeBuilder() {
        rootNode = new RootNode();
        argumentStack = new Stack();
        operatorStack = new Stack();
    } // TreeBuilder

    /**
     * Are we done yet?
     *
     * @return  True if done processing the expression.
     */
    public boolean doneProcessing() {
        return doneProcessing;
    } // doneProcessing

    /**
     * Returns the parser exception, if any.
     *
     * @return  exception; null indicates no error.
     */
    public EvaluationException getException() {
        return evalException;
    } // getException

    /**
     * Returns the root node of the tree.
     *
     * @return  root node.
     */
    public RootNode getRoot() {
        return rootNode;
    } // getRoot

    /**
     * Handles the given literal node appropriately.
     *
     * @param  lit  literal node.
     */
    protected void handleLiteral(Node lit) {
        if (searchState == APPEND_STATE) {
            setError(Errors.DOT_REQUIRES_ID, lit.getToken());
            return;
        }

        // Push the literal onto the tree stack.
        argumentStack.push(lit);
        searchState = OPERATOR_STATE;
    } // handleLiteral

    /**
     * Parse the string as a number, either integer or floating.
     *
     * @param  token  token to be parsed as a number.
     * @param  radix  radix to parse token as; if zero, token is
     *                treated as floating point.
     * @param  node   original node from the lexer.
     */
    protected void handleNumber(String token, int radix, Token node) {
        Number n;
        int last = token.length() - 1;
        char ch = token.charAt(last);
        if (radix > 0) {
            if (ch == 'l' || ch == 'L') {
                token = token.substring(0, last);
                try {
                    n = Long.valueOf(token, radix);
                } catch (NumberFormatException nfe) {
                    setError(Errors.NUMBER_FORMAT, node);
                    return;
                }

            } else {
                try {
                    // Use integer by default...
                    n = Integer.valueOf(token, radix);
                } catch (NumberFormatException nfe) {
                    try {
                        // but try long if that didn't work.
                        n = Long.valueOf(token, radix);
                    } catch (NumberFormatException nfe2) {
                        setError(Errors.NUMBER_FORMAT, node);
                        return;
                    }
                }
            }

        } else {
            if (ch == 'f' || ch == 'F') {
                token = token.substring(0, last);
                try {
                    n = Float.valueOf(token);
                } catch (NumberFormatException nfe) {
                    setError(Errors.NUMBER_FORMAT, node);
                    return;
                }
            } else if (ch == 'd' || ch == 'D') {
                token = token.substring(0, last);
                try {
                    n = Double.valueOf(token);
                } catch (NumberFormatException nfe) {
                    setError(Errors.NUMBER_FORMAT, node);
                    return;
                }

            } else {
                try {
                    n = Float.valueOf(token);
                } catch (NumberFormatException nfe) {
                    try {
                        n = Double.valueOf(token);
                    } catch (NumberFormatException nfe2) {
                        setError(Errors.NUMBER_FORMAT, node);
                        return;
                    }
                }
            }
        }
        handleLiteral(new LiteralNode(node, n));
    } // handleNumber

    /**
     * Handles the given binary or unary operator node.
     *
     * @param  op  operator node.
     */
    protected void handleOperator(OperatorNode op) {
        if (searchState == APPEND_STATE) {
            setError(Errors.DOT_REQUIRES_ID, op.getToken());
            return;
        }

        // If the operator stack is empty, push the new operator. If it
        // has an operator on top, compare the precedence of the two and
        // push the new one if it has lower precedence (or equal
        // precedence: this will force left associativity). Otherwise
        // reduce the two stacks.
        if (!operatorStack.empty()) {
            OperatorNode topop = (OperatorNode) operatorStack.peek();
            int topopp = topop.precedence();
            if (!topop.isSentinel() && op.precedence() >= topopp) {
                reduce();
            }
        }
        operatorStack.push(op);
        searchState = ARGUMENT_STATE;
    } // handleOperator

    /**
     * Handles the given postfix operator node.
     *
     * @param  op  postfix operator node.
     */
    protected void handlePostfixOp(OperatorNode op) {
        // check that there is a variable leaf node on the argument
        // stack, then pop it off and push a postfix operation node
        // (built from the variable and the postfix operator) back onto
        // the argument stack.
    } // handlePostfixOp

    /**
     * Reduce the operator stack by one. If the operator stack top is
     * a left paren, no change is made.
     */
    protected void reduce() {
        // If there is a binary operator on top of the operator stack,
        // there should be two trees on top of the argument stack, both
        // representing expressions. Pop the operator and two trees off,
        // combining them into a single tree node, which is then pushed
        // back on the argument stack. Note that the trees on the
        // argument stack represent the right and left arguments,
        // respectively.
        OperatorNode topop = (OperatorNode) operatorStack.pop();
        if (topop.isSentinel()) {
            // Cleverly do nothing and let the caller handle it.
        } else if (topop instanceof BinaryOperatorNode) {
            try {
                Node arg2 = (Node) argumentStack.pop();
                Node arg1 = (Node) argumentStack.pop();
                topop.addChild(arg1);
                topop.addChild(arg2);
                argumentStack.push(topop);
            } catch (EmptyStackException ese) {
                setError(Errors.MISSING_ARGS, topop.getToken());
            }
        } else if (topop instanceof UnaryOperatorNode) {
            try {
                Node arg = (Node) argumentStack.pop();
                topop.addChild(arg);
                argumentStack.push(topop);
            } catch (EmptyStackException ese) {
                setError(Errors.MISSING_ARGS, topop.getToken());
            }
        } else {
            setError(Errors.UNEXPECTED_TOKEN, topop.getToken());
        }
    } // reduce

    /**
     * Set the error message and set the done flag to true.
     *
     * @param  msg  error message.
     */
    protected void setError(String msg) {
        evalException = new EvaluationException(msg);
        doneProcessing = true;
    } // setError

    /**
     * Set the error value and set the done flag to true.
     *
     * @param  error  error value.
     */
    protected void setError(int error) {
        setError(Errors.getMessage(error));
    } // setError

    /**
     * Set the error value and set the done flag to true.
     * Appends the 'add' string to the error message, separated
     * by a single space character.
     *
     * @param  error  error value.
     * @param  add    additional error suffix.
     */
    protected void setError(int error, String add) {
        setError(Errors.getMessage(error) + ' ' + add);
    } // setError

    /**
     * Set the error value and set the done flag to true.
     * Appends the token information to the message.
     *
     * @param  error  error value.
     * @param  token  token providing additional details.
     */
    protected void setError(int error, Token token) {
        setError(Errors.getMessage(error) + ' ' + token.getText()
                 + " @ " + token.getPos());
    } // setError

    /**
     * Processes the given string, looking for character escapes and
     * translating them to their actual values. Handles single character
     * escapes, Unicode, and octal escapes.
     *
     * @param  str  string to be processed.
     * @return  processed string.
     */
    protected String translateString(String str) {
        int strlen = str.length();
        StringBuffer buf = new StringBuffer(strlen);
        for (int ii = 0; ii < strlen; ii++) {
            char ch = str.charAt(ii);
            if (ch == '\\') {
                ii++;
                ch = str.charAt(ii);
                if (ch == 'b') {
                    buf.append('\b');
                } else if (ch == 'f') {
                    buf.append('\f');
                } else if (ch == 't') {
                    buf.append('\t');
                } else if (ch == 'n') {
                    buf.append('\n');
                } else if (ch == 'r') {
                    buf.append('\r');

                } else if (ch == 'u') {
                    // Unicode character.
                    ii++;
                    String hex = str.substring(ii, ii + 4);
                    try {
                        int i = Integer.parseInt(hex, 16);
                        buf.append((char) i);
                    } catch (NumberFormatException nfe) {
                        throw new IllegalArgumentException(
                            "invalid Unicode: " + hex);
                    }
                    ii += 3; // for loop will increment i again

                } else if (ch >= '0' && ch <= '7') {
                    // Octal escape.
                    int jj = ii;
                    while (jj < strlen) {
                        char oc = str.charAt(jj);
                        if (oc < '0' || oc > '7') {
                            break;
                        }
                        jj++;
                    }
                    String octal = str.substring(ii, jj);
                    try {
                        int i = Integer.parseInt(octal, 8);
                        buf.append((char) i);
                    } catch (NumberFormatException nfe) {
                        throw new IllegalArgumentException("invalid octal: "
                                                           + octal);
                    }
                    ii = jj - 1; // for loop will increment i again

                } else {
                    buf.append(ch);
                }
            } else {
                buf.append(ch);
            }
        }
        return buf.toString();
    } // translateString

    //
    // Methods inherited from AnalysisAdapter.
    //

    /**
     * @param  node  end-of-file node.
     */
    public void caseEOF(EOF node) {
        doneProcessing = true;
        if (searchState == APPEND_STATE) {
            setError(Errors.DOT_REQUIRES_ID, node);
            return;
        }

        // If there is only one tree on the argument stack and the
        // operator stack is empty, return the single tree as the
        // result. If there are more trees and/or operators, reduce the
        // stacks as far as possible.

        int count = 0;
        while (!operatorStack.empty()) {
            OperatorNode topop = (OperatorNode) operatorStack.peek();
            if (topop instanceof LeftParen) {
                setError(Errors.UNMATCHED_LPAREN, topop.getToken());
                return;
            } else if (topop instanceof LeftBracket) {
                setError(Errors.UNMATCHED_LBRACKET, topop.getToken());
                return;
            } else if (topop.isSentinel()) {
                setError(Errors.INVALID_EXPR, topop.getToken());
                return;
            }
            reduce();
            count++;
            // Reduce no more than 100 times.
            if (count > 100) {
                break;
            }
        }

        if (!argumentStack.empty()) {
            Node topArg = (Node) argumentStack.pop();
            if (argumentStack.empty() && operatorStack.empty()) {
                rootNode.addChild(topArg);
            } else {
                setError(Errors.INVALID_EXPR, " Argument stack not empty.");
            }
        }
    }

    //
    // Identifiers
    //

    /**
     *
     * @param  node  identifier node.
     */
    public void caseTIdentifier(TIdentifier node) {
        String name = node.getText();
        if (searchState == APPEND_STATE) {
            // Assumes stack is non-empty.
            IdentifierNode idn = (IdentifierNode) argumentStack.peek();
            idn.append(node.getText());
            searchState = OPERATOR_STATE;
        } else {
            handleLiteral(new IdentifierNode(node, name));
        }
    }

    /**
     *
     * @param  node  'this' node.
     */
    public void caseTThis(TThis node) {
        if (searchState == APPEND_STATE) {
            setError(Errors.DOT_REQUIRES_ID, node);
            return;
        }
        handleLiteral(new IdentifierNode(node, "this"));
    }

    //
    // Literal values.
    //

    /**
     *
     * @param  node  decimal integer literal node.
     */
    public void caseTDecimalIntegerLiteral(TDecimalIntegerLiteral node) {
        handleNumber(node.getText(), 10, node);
    }

    /**
     *
     * @param  node  hexadecimal integer literal node.
     */
    public void caseTHexIntegerLiteral(THexIntegerLiteral node) {
        String t = node.getText();
        // Chop off the 0x prefix.
        t = t.substring(2);
        handleNumber(t, 16, node);
    }

    /**
     *
     * @param  node  octal integer literal node.
     */
    public void caseTOctalIntegerLiteral(TOctalIntegerLiteral node) {
        handleNumber(node.getText(), 8, node);
    }

    /**
     *
     * @param  node  floating point literal node.
     */
    public void caseTFloatingPointLiteral(TFloatingPointLiteral node) {
        handleNumber(node.getText(), 0, node);
    }

    /**
     *
     * @param  node  character literal node.
     */
    public void caseTCharacterLiteral(TCharacterLiteral node) {
        String t = node.getText();
        if (t.length() > 0) {
            t = Strings.trimQuotes(t);
            try {
                handleLiteral(new LiteralNode(node, new Character(
                                                  translateChar(t))));
            } catch (Exception e) {
                setError(Errors.INVALID_EXPR, node);
            }
        } else {
            setError(Errors.INVALID_EXPR, node);
        }
    }

    /**
     *
     * @param  node  string literal node.
     */
    public void caseTStringLiteral(TStringLiteral node) {
        String t = node.getText();
        if (t.length() > 0) {
            t = Strings.trimQuotes(t);
            try {
                t = translateString(t);
                handleLiteral(new LiteralNode(node, t));
            } catch (Exception e) {
                setError(Errors.INVALID_EXPR, node);
            }
        } else {
            setError(Errors.INVALID_EXPR, node);
        }
    }

    /**
     *
     * @param  node  true keyword node.
     */
    public void caseTTrue(TTrue node) {
        handleLiteral(new LiteralNode(node, Boolean.TRUE));
    }

    /**
     *
     * @param  node  false keyword node.
     */
    public void caseTFalse(TFalse node) {
        handleLiteral(new LiteralNode(node, Boolean.FALSE));
    }

    /**
     *
     * @param  node  null keyword node.
     */
    public void caseTNull(TNull node) {
        handleLiteral(new LiteralNode(node, null));
    }

    //
    // Operators
    //

    /**
     *
     * @param  node  left parenthesis.
     */
    public void caseTLParenthese(TLParenthese node) {
        LeftParen lp = new LeftParen(node);
        // Simply push it on the operator stack; the search should
        // have been for an argument and should remain so.
        if (!argumentStack.empty()
            && argumentStack.peek() instanceof IdentifierNode) {
            // This is the start of a method invocation.
            IdentifierNode ident = (IdentifierNode) argumentStack.pop();
            MethodNode method = new MethodNode(ident.getToken());
            method.addChild(ident);
            // Put it on the argument stack as a sentinel, to mark
            // the beginning of the method arguments.
            argumentStack.push(method);
            // Put it on the operator stack so when we find the
            // right parenthesis, we can determine that we were
            // making a method call.
            operatorStack.push(method);
            searchState = ARGUMENT_STATE;
            methodCount++;
        }
        // Else, must be the start of a type-cast.
        operatorStack.push(lp);
    }

    /**
     *
     * @param  node  right parenthesis.
     */
    public void caseTRParenthese(TRParenthese node) {
        if (operatorStack.empty()) {
            setError(Errors.UNMATCHED_RPAREN, node);
            return;
        }
        // If there is a left parenthesis on the operator stack, we can
        // "cancel" the pair. If the operator stack contains some other
        // operator on top, reduce the stacks.
        OperatorNode topop = (OperatorNode) operatorStack.peek();
        while (!(topop instanceof LeftParen)) {
            reduce();
            if (operatorStack.empty() || topop.isSentinel()) {
                setError(Errors.UNMATCHED_RPAREN, node);
                return;
            }
            topop = (OperatorNode) operatorStack.peek();
        }
        operatorStack.pop();

        // Now check for the method invocation case.
        if (!operatorStack.empty()
            && operatorStack.peek() instanceof MethodNode) {
            // It was a method invocation.
            MethodNode method = (MethodNode) operatorStack.pop();
            // Pop off the arguments and add them in reverse order.
            Node n = (Node) argumentStack.pop();
            Stack args = new Stack();
            while (n != method) {
                args.push(n);
                n = (Node) argumentStack.pop();
            }
            while (!args.empty()) {
                Node arg = (Node) args.pop();
                method.addChild(arg);
            }
            // Put the method invocation back on the argument stack
            // because it is treated as a value, not an operator.
            argumentStack.push(method);
            methodCount--;

        } else {
            // Maybe it is a type-cast operation; otherwise it was a
            // grouping operator and that has been taken care of.
            try {
                // Pop off the type for the type-cast.
                Node n = (Node) argumentStack.peek();
                if (n instanceof IdentifierNode) {
                    argumentStack.pop();
                    IdentifierNode in = (IdentifierNode) n;
                    TypeCastOperatorNode tcon = new TypeCastOperatorNode(
                        in.getToken());
                    tcon.addChild(in);
                    // The type-cast is an operator.
                    handleOperator(tcon);
                }
            } catch (EmptyStackException ese) {
                setError(Errors.MISSING_ARGS, topop.getToken());
            }
        }
    }

    /**
     *
     * @param  node  left bracket.
     */
    public void caseTLBracket(TLBracket node) {
        LeftBracket lb = new LeftBracket(node);
        // Simply push it on the operator stack; the search should
        // have been for an argument and should remain so.
        if (!argumentStack.empty()
            && argumentStack.peek() instanceof IdentifierNode) {
            // This is the start of an array access.
            IdentifierNode ident = (IdentifierNode) argumentStack.pop();
            ArrayRefNode arrayref = new ArrayRefNode(ident.getToken());
            arrayref.addChild(ident);
            // Put it on both stacks.
            argumentStack.push(arrayref);
            operatorStack.push(arrayref);
            searchState = ARGUMENT_STATE;
        }
        operatorStack.push(lb);
    }

    /**
     *
     * @param  node  right bracket.
     */
    public void caseTRBracket(TRBracket node) {
        // If there is a left bracket on the operator stack, we can
        // "cancel" the pair If the operator stack contains some other
        // operator on top, reduce the stacks.
        if (operatorStack.empty()) {
            setError(Errors.UNMATCHED_RBRACKET, node);
            return;
        }
        OperatorNode topop = (OperatorNode) operatorStack.peek();
        while (!(topop instanceof LeftBracket)) {
            reduce();
            if (operatorStack.empty() || topop.isSentinel()) {
                setError(Errors.UNMATCHED_RBRACKET, node);
                return;
            }
            topop = (OperatorNode) operatorStack.peek();
        }
        operatorStack.pop();

        // Now check for the array reference case.
        if (!operatorStack.empty()
            && operatorStack.peek() instanceof ArrayRefNode) {
            ArrayRefNode arrayref = (ArrayRefNode) operatorStack.pop();
            // It was an array reference.
            // Pop off the arguments until the [ is found.
            Node n = (Node) argumentStack.pop();
            int count = 0;
            while (n != arrayref) {
                arrayref.addChild(n);
                count++;
                n = (Node) argumentStack.pop();
            }
            if (count > 1) {
                setError(Errors.ARRAY_MULTI_INDEX, arrayref.getToken());
            }
            // Put the array reference back on the argument stack.
            argumentStack.push(arrayref);
        } else {
            // Brackets without a preceeding identifier.
            setError(Errors.UNEXPECTED_TOKEN, topop.getToken());
        }
    }

    /**
     *
     * @param  node  comma.
     */
    public void caseTComma(TComma node) {
        if (methodCount == 0) {
            setError(Errors.UNEXPECTED_TOKEN, node);
        } else {
            // Reduce the operator stack to the left parenthesis.
            if (operatorStack.empty()) {
                setError(Errors.UNEXPECTED_TOKEN, node);
                return;
            }
            OperatorNode topop = (OperatorNode) operatorStack.peek();
            while (!topop.isSentinel()) {
                reduce();
                topop = (OperatorNode) operatorStack.peek();
            }
        }
        // Was there anything before this comma?
        // Doesn't really matter, simply invokes the wrong method
        // or fails to find a matching method. Stupid user error.
    }

    /**
     *
     * @param  node  period.
     */
    public void caseTDot(TDot node) {
        // Dot is special and needs processing before evaluation.
        if (!argumentStack.empty()) {
            Node n = (Node) argumentStack.peek();
            if (n instanceof IdentifierNode) {
                // If top argument is an identifier, append the dot
                // to the identifier's name.
                IdentifierNode idn = (IdentifierNode) n;
                idn.append(node.getText());
                // The next identifier we see must be appended as well.
                searchState = APPEND_STATE;
            } else if (n instanceof LiteralNode) {
                // If top argument is a string literal, this starts a
                // method invocation on that string.
                setError(Errors.UNSUPPORTED_FEATURE, node);
            } else {
                setError(Errors.UNEXPECTED_TOKEN, node);
            }
        } else {
            setError(Errors.UNEXPECTED_TOKEN, node);
        }
    }

    /**
     *
     * @param  node  plus sign.
     */
    public void caseTPlus(TPlus node) {
        // The plus is unary if we are expecting an argument or if
        // the argument stack is empty (ie. search state is 'start').
        if (searchState == START_STATE || searchState == ARGUMENT_STATE) {
            handleOperator(new PlusUnaryOperatorNode(node));
        } else {
            handleOperator(new PlusBinaryOperatorNode(node));
        }
    }

    /**
     *
     * @param  node  minus sign.
     */
    public void caseTMinus(TMinus node) {
        // The minus is unary if we are expecting an argument or if
        // the argument stack is empty (ie. search state is 'start').
        if (searchState == START_STATE || searchState == ARGUMENT_STATE) {
            handleOperator(new MinusUnaryOperatorNode(node));
        } else {
            handleOperator(new MinusBinaryOperatorNode(node));
        }
    }

    /**
     *
     * @param  node  asterisk.
     */
    public void caseTStar(TStar node) {
        handleOperator(new MultBinaryOperatorNode(node));
    }

    /**
     *
     * @param  node  division sign.
     */
    public void caseTDiv(TDiv node) {
        handleOperator(new DivBinaryOperatorNode(node));
    }

    /**
     *
     * @param  node  modulus sign.
     */
    public void caseTMod(TMod node) {
        handleOperator(new ModBinaryOperatorNode(node));
    }

    /**
     *
     * @param  node  shift left.
     */
    public void caseTShiftLeft(TShiftLeft node) {
        handleOperator(new LeftShiftOperatorNode(node));
    }

    /**
     *
     * @param  node  signed shift right.
     */
    public void caseTSignedShiftRight(TSignedShiftRight node) {
        handleOperator(new RightShiftOperatorNode(node));
    }

    /**
     *
     * @param  node  unsigned shift right.
     */
    public void caseTUnsignedShiftRight(TUnsignedShiftRight node) {
        handleOperator(new UnsignedRightShiftOperatorNode(node));
    }

    /**
     *
     * @param  node  bitwise and.
     */
    public void caseTBitAnd(TBitAnd node) {
        handleOperator(new BitwiseAndOperatorNode(node));
    }

    /**
     *
     * @param  node  bitwise or.
     */
    public void caseTBitOr(TBitOr node) {
        handleOperator(new BitwiseOrOperatorNode(node));
    }

    /**
     *
     * @param  node  bitwise exclusive or.
     */
    public void caseTBitXor(TBitXor node) {
        handleOperator(new BitwiseXorOperatorNode(node));
    }

    /**
     *
     * @param  node  bitwise complement.
     */
    public void caseTBitComplement(TBitComplement node) {
        handleOperator(new BitwiseNegOperatorNode(node));
    }

    /**
     *
     * @param  node  logical complement.
     */
    public void caseTComplement(TComplement node) {
        handleOperator(new BooleanNegOperatorNode(node));
    }

    /**
     *
     * @param  node  logical and.
     */
    public void caseTAnd(TAnd node) {
        handleOperator(new BooleanAndOperatorNode(node));
    }

    /**
     *
     * @param  node  logical or.
     */
    public void caseTOr(TOr node) {
        handleOperator(new BooleanOrOperatorNode(node));
    }

    /**
     *
     * @param  node  equality.
     */
    public void caseTEq(TEq node) {
        handleOperator(new EqualsOperatorNode(node));
    }

    /**
     *
     * @param  node  negation.
     */
    public void caseTNeq(TNeq node) {
        handleOperator(new NotEqualsOperatorNode(node));
    }

    /**
     *
     * @param  node  greater than.
     */
    public void caseTGt(TGt node) {
        handleOperator(new GtOperatorNode(node));
    }

    /**
     *
     * @param  node  less than.
     */
    public void caseTLt(TLt node) {
        handleOperator(new LtOperatorNode(node));
    }

    /**
     *
     * @param  node  less than or equal.
     */
    public void caseTLteq(TLteq node) {
        handleOperator(new LtEqOperatorNode(node));
    }

    /**
     *
     * @param  node  greater than or equal.
     */
    public void caseTGteq(TGteq node) {
        handleOperator(new GtEqOperatorNode(node));
    }

    //
    // Unsupported features (that may be supported in the future)
    //

    /**
     *
     * @param  node  super keyword.
     */
    public void caseTSuper(TSuper node) {
        setError(Errors.UNSUPPORTED_FEATURE, node);
    }

    /**
     *
     * @param  node  question mark.
     */
    public void caseTQuestion(TQuestion node) {
        setError(Errors.UNSUPPORTED_FEATURE, node);
    }

    /**
     *
     * @param  node  colon.
     */
    public void caseTColon(TColon node) {
        setError(Errors.UNSUPPORTED_FEATURE, node);
    }

    /**
     *
     * @param  node  boolean keyword.
     */
    public void caseTBoolean(TBoolean node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "boolean"));
    }

    /**
     *
     * @param  node  byte keyword.
     */
    public void caseTByte(TByte node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "byte"));
    }

    /**
     *
     * @param  node  char keyword
     */
    public void caseTChar(TChar node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "char"));
    }

    /**
     *
     * @param  node  class keyword
     */
    public void caseTClass(TClass node) {
        setError(Errors.UNSUPPORTED_FEATURE, node);
    }

    /**
     *
     * @param  node  double keyword
     */
    public void caseTDouble(TDouble node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "double"));
    }

    /**
     *
     * @param  node  float keyword
     */
    public void caseTFloat(TFloat node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "float"));
    }

    /**
     *
     * @param  node  int keyword
     */
    public void caseTInt(TInt node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "int"));
    }

    /**
     *
     * @param  node  long keyword
     */
    public void caseTLong(TLong node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "long"));
    }

    /**
     *
     * @param  node  short keyword
     */
    public void caseTShort(TShort node) {
        // Treat this as an identifier.
        handleLiteral(new IdentifierNode(node, "short"));
    }

    //
    // Unsupported tokens.
    //

    /**
     *
     * @param  node  comment
     */
    public void caseTTraditionalComment(TTraditionalComment node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  comment
     */
    public void caseTDocumentationComment(TDocumentationComment node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  comment
     */
    public void caseTEndOfLineComment(TEndOfLineComment node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  void keyword
     */
    public void caseTVoid(TVoid node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  left brace.
     */
    public void caseTLBrace(TLBrace node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  right brace.
     */
    public void caseTRBrace(TRBrace node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  abstract keyword.
     */
    public void caseTAbstract(TAbstract node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  assert keyword.
     */
    public void caseTAssert(TAssert node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  break keyword.
     */
    public void caseTBreak(TBreak node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  case keyword.
     */
    public void caseTCase(TCase node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  catch keyword.
     */
    public void caseTCatch(TCatch node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  const keyword.
     */
    public void caseTConst(TConst node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  continue keyword.
     */
    public void caseTContinue(TContinue node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  default keyword.
     */
    public void caseTDefault(TDefault node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  do keyword.
     */
    public void caseTDo(TDo node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  else keyword.
     */
    public void caseTElse(TElse node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  extends keyword.
     */
    public void caseTExtends(TExtends node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  final keyword.
     */
    public void caseTFinal(TFinal node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  finally keyword.
     */
    public void caseTFinally(TFinally node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  for keyword.
     */
    public void caseTFor(TFor node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  goto keyword.
     */
    public void caseTGoto(TGoto node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  if keyword.
     */
    public void caseTIf(TIf node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  implements keyword.
     */
    public void caseTImplements(TImplements node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  import keyword.
     */
    public void caseTImport(TImport node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  instanceof keyword.
     */
    public void caseTInstanceof(TInstanceof node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  interface keyword.
     */
    public void caseTInterface(TInterface node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  native keyword.
     */
    public void caseTNative(TNative node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  new keyword.
     */
    public void caseTNew(TNew node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  package keyword.
     */
    public void caseTPackage(TPackage node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  private keyword.
     */
    public void caseTPrivate(TPrivate node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  protected keyword.
     */
    public void caseTProtected(TProtected node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  public keyword.
     */
    public void caseTPublic(TPublic node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  return keyword.
     */
    public void caseTReturn(TReturn node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  static keyword.
     */
    public void caseTStatic(TStatic node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  strictfp keyword.
     */
    public void caseTStrictfp(TStrictfp node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  switch keyword.
     */
    public void caseTSwitch(TSwitch node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  synchronized keyword.
     */
    public void caseTSynchronized(TSynchronized node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  throw keyword.
     */
    public void caseTThrow(TThrow node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  throws keyword.
     */
    public void caseTThrows(TThrows node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  transient keyword.
     */
    public void caseTTransient(TTransient node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  try keyword.
     */
    public void caseTTry(TTry node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  volatile keyword.
     */
    public void caseTVolatile(TVolatile node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  while keyword.
     */
    public void caseTWhile(TWhile node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  semicolon keyword.
     */
    public void caseTSemicolon(TSemicolon node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  assign keyword.
     */
    public void caseTAssign(TAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  plus plus sign.
     */
    public void caseTPlusPlus(TPlusPlus node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  minus minus sign.
     */
    public void caseTMinusMinus(TMinusMinus node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  plus assign.
     */
    public void caseTPlusAssign(TPlusAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  minus assign.
     */
    public void caseTMinusAssign(TMinusAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  multiply assign.
     */
    public void caseTStarAssign(TStarAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  divide assign.
     */
    public void caseTDivAssign(TDivAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  bitwise and assign.
     */
    public void caseTBitAndAssign(TBitAndAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  bitwise or assign.
     */
    public void caseTBitOrAssign(TBitOrAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  bitwise exclusive or assign.
     */
    public void caseTBitXorAssign(TBitXorAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  modulus assign.
     */
    public void caseTModAssign(TModAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  shift left assign.
     */
    public void caseTShiftLeftAssign(TShiftLeftAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  signed shift right assign.
     */
    public void caseTSignedShiftRightAssign(TSignedShiftRightAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }

    /**
     *
     * @param  node  unsigned shift right assign.
     */
    public void caseTUnsignedShiftRightAssign(TUnsignedShiftRightAssign node) {
        setError(Errors.UNSUPPORTED_TOKEN, node);
    }
} // TreeBuilder
