package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.*;
import com.limegroup.gnutella.browser.ExternalControl;
import com.limegroup.gnutella.settings.ApplicationSettings;
import com.limegroup.gnutella.settings.SharingSettings;
import com.limegroup.gnutella.settings.StartupSettings;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.FileUtils;
import com.limegroup.gnutella.gui.init.*;
import com.limegroup.gnutella.gui.notify.NotifyUserProxy;

import java.lang.reflect.*;
import java.io.*;
import javax.swing.UIManager;

/**
 * This class instantiates all of the "top-level" application classes.  These
 * include the <tt>ResourceManager</tt>, the <tt>GUIMediator</tt>, the
 * update checking classes, the <tt>SettingsManager</tt>, the
 * <tt>FileManager</tt>, <tt>RouterService</tt>, etc.  <tt>GUIMediator</tt>
 * and <tt>RouterService</tt> construct the bulk of the front and back ends,
 * respectively. This class also links together any top-level classes that need
 * to know about each other.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class Initializer {

    /**
     * Suppress the default constructor to ensure that this class can never be
     * constructed.
     */
    private Initializer() {}

    /**
     * Initializes all of the necessary application classes.  This begins with
     * the <tt>ResourceManager</tt> class that handles the look and feel of the
     * application as well as providing access to any string or image resources
     * used in the application.  This then constructs the <tt>GUIMediator</tt>,
     * which in turn handles the construction of the entire frontend.<p>
     *
     * The <tt>SettingsManager</tt> and the <tt>FileManager</tt> are
     * initialized early to make sure that their resources are available and
     * loading as soon as possible.  The update and init code is also kicked
     * off early to check for any available updates and to make sure that all
     * of the settings have been properly set up (going into the init setup
     * sequence otherwise).
     *
     * Finally, all of the primary backend classes are created, such as the
     * <tt>RouterService</tt> interface and <tt>StandardMessageRouter</tt>.
     * <tt>RouterService</tt> handles the creation of the other backend
     * classes.
     *
     * If this throws any exceptions, then LimeWire was not able to construct
     * properly and must be shut down.
     */
    static void initialize(String args[]) throws Throwable {
        ErrorService.setErrorCallback(new ErrorHandler());

        // Register the MacOS X magnet URL handler.
        // TODO: need to get GURL stuff working on Mac OS X using java 1.4
        if (CommonUtils.isMacOSX() && !CommonUtils.isJava14OrLater())
            GURLHandler.getInstance().register();

        // If this is a request to launch a pmf then just do it and exit.
        if ( args.length >= 2 && "-pmf".equals(args[0]) ) {
            PackagedMediaFileLauncher.launchFile(args[1], false); 
            return;
        }

        // Test for preexisting LimeWire and pass it a magnet URL if one
        // has been passed in.
        String arg = null;
        if (args.length > 0) {
            arg = ExternalControl.preprocessArgs(args);
            ExternalControl.checkForActiveLimeWire(arg);
			ExternalControl.enqueueMagnetRequest(arg);	
        } else if (!StartupSettings.ALLOW_MULTIPLE_INSTANCES.getValue()) {
            // if we don't want multiple instances, we need to check if
            // limewire is already active.
            ExternalControl.checkForActiveLimeWire();
        }

        Initializer.setOSXSystemProperties();

        ResourceManager.instance();

        // Show the splash screen
        SplashWindow splash = new SplashWindow();

        SplashWindow.setStatusText(
            GUIMediator.getStringResource("SPLASH_STATUS_USER_SETTINGS"));

        // TODO: Initialize SettingsHandler things?

        SplashWindow.setStatusText(
            GUIMediator.getStringResource("SPLASH_STATUS_SHARED_FILES"));

        // Run through the initialization sequence -- this must always be
        // called before GUIMediator constructs the LibraryTree!!
        if (!ApplicationSettings.INSTALLED.getValue()) {
            new SetupManager();
        }

        File saveDir = SharingSettings.getSaveDirectory();
        
        // if doesn't exist, isn't directory, or can't write, make a new one.
        if (( saveDir==null || 
            !saveDir.exists() ||
            !saveDir.isDirectory() ||
            !saveDir.canWrite()) &&
            !saveDir.mkdirs() ) {
                if(!FileUtils.setWriteable(saveDir) )
                    SaveDirectoryWindow.showSaveDirectoryWindow();
        }

        SplashWindow.setStatusText(
            GUIMediator.getStringResource("SPLASH_STATUS_INTERFACE"));

        // Construct the frontend
        GUIMediator mediator = GUIMediator.instance();

        SplashWindow.setStatusText(
            GUIMediator.getStringResource("SPLASH_STATUS_UPDATES"));

        // Create the update object
//              UpdateManager um = UpdateManager.instance();
//  
//              // Perform the update check
//              um.doUpdate();

        SplashWindow.setStatusText(
            GUIMediator.getStringResource("SPLASH_STATUS_CORE_COMPONENTS"));

        // Construct gui callback class
        ActivityCallback ac = new VisualConnectionCallback();

        // Construct the RouterService object, which functions as the
        // backend initializer as well as the interface from the GUI to the
        // front end.
        RouterService routerService = new RouterService(ac);

        // Notify GUIMediator of the RouterService interface class to the
        // backend.
        mediator.setRouterService(routerService);

        // Instruct the gui to perform tasks that can only be performed
        // after the backend has been constructed.
        mediator.startTimer();
        
        // Create the user desktop notifier object.
        // This must be done before the GUI is made visible,
        // otherwise the user can close it and not see the
        // tray icon.
        NotifyUserProxy notifyProxy = NotifyUserProxy.instance();
        // Hide the notifier after it has been initialized.
        notifyProxy.hideNotify();            

        // Hide the splash screen and recycle its memory.
        splash.dispose();

        // Make the GUI visible.
        GUIMediator.setAppVisible(true);

        routerService.start();

        if (!GUIMediator.hasDonated()) {
            new PROWindow().showDialog();
        }            

        // Activate a download for magnet URL locally if one exists
        ExternalControl.runQueuedMagnetRequest();
    }

    /**
     * Uses reflection to set system properties on OS X.  Reflection is
     * necessary because the static System.setProperty(String, String) method
     * is a Java2 method.
     */
    static void setOSXSystemProperties() {
        if (!CommonUtils.isMacOSX())
            return;

        try {
            final Method setPropertyMethod = System.class.getDeclaredMethod(
                "setProperty", new Class[] {String.class, String.class});
            if (CommonUtils.isJava14OrLater()) {
                UIManager.put("ProgressBar.repaintInterval", new Integer(500));
                setPropertyMethod.invoke(null, new String[] {
                    "apple.laf.useScreenMenuBar", "true"});
            } else {
                setPropertyMethod.invoke(null, new String[] {
                "com.apple.macos.useScreenMenuBar", "true"});
				setPropertyMethod.invoke(null, new String[] {
					"com.apple.mrj.application.apple.menu.about.name", "LimeWire"});
                setPropertyMethod.invoke(null, new String[] {
                    "com.apple.macos.use-file-dialog-packages", "true"});
			}
        } catch (Exception e) {
            // We don't really care what type of exception it is, as there's
            // nothing we can do about it anyway.
        }
    }        
}
