package com.limegroup.gnutella.gui.download;

import com.limegroup.gnutella.gui.mp3.*;
import com.limegroup.gnutella.gui.tables.*;
import com.limegroup.gnutella.gui.RefreshListener;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.PaddedPanel;
import com.limegroup.gnutella.*;
import com.limegroup.gnutella.settings.QuestionsHandler;
import com.limegroup.gnutella.downloader.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.xml.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.gui.search.SearchMediator;
import javax.swing.event.*;
import javax.swing.*;
import javax.swing.table.*;
import java.awt.event.*;
import java.awt.*;
import com.sun.java.util.collections.*;
import java.io.*;

/**
 * This class acts as a mediator between all of the components of the
 * download window.  It also constructs all of the download window
 * components.
 */
public final class DownloadMediator extends AbstractTableMediator {

	/**
	 * Variable for the total number of downloads that have been added in this
	 * session.
	 */
	private static int _totalDownloads = 0;

	/**
	 * Flag for whether or not an mp3 file has been launched from the download
	 * window.
	 */
    private static boolean _audioLaunched = false;

    /**
     * instance, for singleton acces
     */
    private static DownloadMediator _instance = new DownloadMediator();

    /**
     * Variable for whether or not the currently selected host has chat
     * enabled.
     */
    private static boolean _chatEnabled;

    /**
     * Variable for whether or not the currently selected host has browse
     * host enabled.
     */
    private static boolean _browseEnabled;


    public static DownloadMediator instance() { return _instance; }

    /**
     * Variables so only one ActionListener needs to be created for both
     * the buttons & popup menu.
     */
    public ActionListener CHAT_LISTENER;
    public ActionListener CLEAR_LISTENER;
    public ActionListener BROWSE_LISTENER;
    public ActionListener LAUNCH_LISTENER;
    public ActionListener RESUME_LISTENER;

    /** The actual download buttons instance.
     */
    private DownloadButtons _downloadButtons;

    /**
     * Build some extra listeners
     */
    protected void buildListeners() {
        super.buildListeners();
        CHAT_LISTENER = new ChatListener(this);
        CLEAR_LISTENER = new ClearListener(this);
        BROWSE_LISTENER = new BrowseListener(this);
        LAUNCH_LISTENER = new LaunchListener(this);
        RESUME_LISTENER = new ResumeListener(this);
    }

	/**
	 * Set up the necessary constants.
	 */
	protected void setupConstants() {
		MAIN_PANEL =
		    new PaddedPanel(GUIMediator.getStringResource("DOWNLOAD_TITLE"));
		DATA_MODEL = new DownloadModel();
		TABLE = new LimeJTable(DATA_MODEL);
        _downloadButtons = new DownloadButtons(this);
		BUTTON_ROW = _downloadButtons.getComponent();
    }

    /**
     * Update the splash screen.
     */
	protected void updateSplashScreen() {
		GUIMediator.setSplashScreenString(
            GUIMediator.getStringResource("SPLASH_STATUS_DOWNLOAD_WINDOW"));
    }

	/**
	 * Constructs all of the elements of the download window, including
	 * the table, the buttons, etc.
	 */
	private DownloadMediator() {
		GUIMediator.addRefreshListener(this);
	}

	/**
	 * Override the default refreshing so that we can
	 * set the clear button appropriately.
	 */
	public void doRefresh() {
	    boolean inactivePresent =
	        ((Boolean)DATA_MODEL.refresh()).booleanValue();
	    setButtonEnabled(DownloadButtons.CLEAR_BUTTON, inactivePresent);
        int[] selRows = TABLE.getSelectedRows();
        if (selRows.length > 0) {
            DownloadDataLine dataLine = 
                (DownloadDataLine)DATA_MODEL.get(selRows[0]);
            if (dataLine.getState() == Downloader.WAITING_FOR_USER)
                _downloadButtons.transformResumeButton();
            else
                _downloadButtons.transformSourcesButton();
        }
	}

	/**
	 * Returns the total number of Downloads that have occurred in this session.
	 *
	 * @return the total number of Downloads that have occurred in this session
	 */
    public int getTotalDownloads() {
        return _totalDownloads;
    }

 	/**
	 * Returns the total number of current Downloads.
	 *
	 * @return the total number of current Downloads
	 */
    public int getCurrentDownloads() {
        return ((DownloadModel)DATA_MODEL).getCurrentDownloads();
    }

 	/**
	 * Returns the total number of active Downloads.
     * This includes anything that is still viewable in the Downloads view.
	 *
	 * @return the total number of active Downloads
	 */
    public int getActiveDownloads() {
        return ((DownloadModel)DATA_MODEL).getRowCount();
    }

    /**
     * Overrides the default add.
     *
	 * Adds a new Downloads to the list of Downloads, obtaining the necessary
	 * information from the supplied <tt>Downloader</tt>.
	 *
	 * If the download is not already in the list, then it is added.
     *  <p>
	 */
    public void add(Object downloader) {
        if ( !DATA_MODEL.contains(downloader) ) {
            _totalDownloads++;
            super.add(downloader);
        }
    }

	/**
	 * Overrides the default remove.
	 *
	 * Takes action upon downloaded theme files, asking if the user wants to
	 * apply the theme.
	 *
	 * Removes a download from the list if the user has configured their system
	 * to automatically clear completed download and if the download is
	 * complete.
	 *
	 * @param downloader the <tt>Downloader</tt> to remove from the list if it is
	 *  complete.
	 */
    public void remove(Object downloader) {
        Downloader dloader = (Downloader)downloader;
        int state = dloader.getState();
        
        if(state == Downloader.COMPLETE &&
         isThemeFile(dloader.getFileName())) {
            File themeFile = dloader.getDownloadFragment();
            themeFile = copyToThemeDir(themeFile);
            int response = GUIMediator.showYesNoMessage(
                "DOWNLOAD_APPLY_NEW_THEME_START",
                ThemeSettings.formatName(dloader.getFileName()),
                "DOWNLOAD_APPLY_NEW_THEME_END",
                QuestionsHandler.THEME_DOWNLOADED);
            if( response == GUIMediator.YES_OPTION )
                applyNewTheme(themeFile);
        }
        
        if(SharingSettings.CLEAR_DOWNLOAD.getValue()
		   && ( state == Downloader.COMPLETE ||
		        state == Downloader.ABORTED ) ) {
            super.remove(downloader);
		} else {
		    DownloadDataLine ddl = (DownloadDataLine)DATA_MODEL.get(downloader);
		    if (ddl != null) ddl.setEndTime(System.currentTimeMillis());
		}
    }
    
    private File copyToThemeDir(File themeFile) {
        File themeDir = ThemeSettings.THEME_DIR_FILE;
        File realLoc = new File(themeDir, themeFile.getName());
        // if they're the same, just use it.
        if( realLoc.equals(themeFile) )
            return themeFile;

        // otherwise, if the file already exists in the theme dir, remove it.
        realLoc.delete();
        
        // copy from shared to theme dir.
        CommonUtils.copy(themeFile, realLoc);
        return realLoc;
    }
    
    private boolean isThemeFile(String name) {
        return name.toLowerCase().endsWith(ThemeSettings.EXTENSION);
    }
    
    private void applyNewTheme(File themeFile) {
        ThemeSettings.THEME_FILE.setValue(themeFile);
        ThemeFileHandler.reload();
        GUIMediator.instance().setCurrentTheme(themeFile);
    }        
        

	/**
	 * Launches the selected files in the <tt>Launcher</tt> or in the built-in
	 * media player.
	 */
	void launchSelectedDownloads() {
		final int[] rows = TABLE.getSelectedRows();
		int i = 0;
		int length = rows.length;
        _audioLaunched = false;
		while(i < length) {
            final int iLocal = rows[i];
            (new Thread("DownloadLineLauncher") {
                public void run() {
        		    DataLine dl = (DataLine)DATA_MODEL.get(iLocal);
        			Downloader downloader=(Downloader)dl.getInitializeObject();
                    File toLaunch = downloader.getDownloadFragment();
					if (toLaunch == null) return;
					if (PlayerSettings.PLAYER_ENABLED.getValue() &&
						!_audioLaunched &&
						(CommonUtils.canLaunchFileWithQuickTime(toLaunch) ||
						 LimeXMLUtils.isMP3File(toLaunch))) {
						GUIMediator.instance().launchAudio(toLaunch);
						_audioLaunched = true;
					}
					else {
						try {
							GUIMediator.launchFile(toLaunch);
						}
						catch (IOException ignored) {}
						catch (SecurityException ignored) {}
					}
                }
            }).start();
            i++;
		}
	}

	/**
	 * Forces the selected downloads in the download window to resume.
	 */
	void resumeSelectedDownloads() {
		int[] sel = TABLE.getSelectedRows();
		for(int i=0; i<sel.length; i++) {
			DataLine dd = DATA_MODEL.get(sel[i]);
			Downloader downloader=(Downloader)dd.getInitializeObject();
			try {
				downloader.resume();
			} catch (AlreadyDownloadingException e) {
				GUIMediator.showError("ERROR_ALREADY_DOWNLOADING",
									  "\""+e.getFilename()+"\".",
									  QuestionsHandler.ALREADY_DOWNLOADING);
			}
		}
	}

	/**
	 * Opens up a chat session with the selected hosts in the download
	 * window.
	 */
	void chatWithSelectedDownloads() {
		int[] sel = TABLE.getSelectedRows();
		for (int i =0; i<sel.length; i++) {
		    DataLine dl = (DataLine)DATA_MODEL.get(sel[i]);
			Downloader downloader=(Downloader)dl.getInitializeObject();
			Endpoint end = downloader.getChatEnabledHost();
			if (end!=null)
				RouterService.createChat(end.getHostname(), end.getPort());
		}
	}

	/**
	 * Opens up a browse session with the selected hosts in the download
	 * window.
	 */
	void browseSelectedDownloads() {
		int[] sel = TABLE.getSelectedRows();
		for (int i =0; i<sel.length; i++) {
		    DataLine dl = (DataLine)DATA_MODEL.get(sel[i]);
			Downloader downloader=(Downloader)dl.getInitializeObject();
			RemoteFileDesc end = downloader.getBrowseEnabledHost();
            if (end != null)
                SearchMediator.doBrowseHost(end);
		}
	}

	/**
	 * Handles a double-click event in the table.
	 */
	public void handleMouseDoubleClick(Point p) {
		launchSelectedDownloads();
	}

	/**
	 * Clears the downloads in the download window that have completed.
	 */
	void clearCompletedDownloads() {
		((DownloadModel)DATA_MODEL).clearCompleted();
		clearSelection();
        setButtonEnabled(DownloadButtons.CLEAR_BUTTON, false);
	}

    // inherit doc comment
    protected JPopupMenu createPopupMenu() {
        JPopupMenu menu = (new DownloadPopupMenu(this)).getComponent();
        boolean selection = !TABLE.getSelectionModel().isSelectionEmpty();
		menu.getComponent(DownloadPopupMenu.KILL_INDEX).
            setEnabled(selection);
		menu.getComponent(DownloadPopupMenu.RESUME_INDEX).
            setEnabled(selection);
		menu.getComponent(DownloadPopupMenu.LAUNCH_INDEX).
            setEnabled(selection);
		menu.getComponent(DownloadPopupMenu.CHAT_INDEX).
            setEnabled(_chatEnabled);
		menu.getComponent(DownloadPopupMenu.BROWSE_INDEX).
            setEnabled(_browseEnabled);   
        return menu;
    }
    
	/**
	 * Handles the selection of the specified row in the download window,
	 * enabling or disabling buttons and chat menu items depending on
	 * the values in the row.
	 *
	 * @param row the selected row
	 */
	public void handleSelection(int row) {

		DownloadDataLine dataLine = (DownloadDataLine)DATA_MODEL.get(row);
		_chatEnabled = dataLine.getChatEnabled();
        _browseEnabled = dataLine.getBrowseEnabled();

        if (dataLine.getState() == Downloader.WAITING_FOR_USER)
            _downloadButtons.transformResumeButton();
        else
            _downloadButtons.transformSourcesButton();
		
		setButtonEnabled(DownloadButtons.KILL_BUTTON, true);
		setButtonEnabled(DownloadButtons.RESUME_BUTTON, true);
		setButtonEnabled(DownloadButtons.LAUNCH_BUTTON, true);
		setButtonEnabled(DownloadButtons.CHAT_BUTTON, _chatEnabled);
	}

	/**
	 * Handles the deselection of all rows in the download table,
	 * disabling all necessary buttons and menu items.
	 */
	public void handleNoSelection() {
        _chatEnabled = false;
        _browseEnabled = false;

		setButtonEnabled(DownloadButtons.KILL_BUTTON, false);
		setButtonEnabled(DownloadButtons.RESUME_BUTTON, false);
		setButtonEnabled(DownloadButtons.LAUNCH_BUTTON, false);
		setButtonEnabled(DownloadButtons.CHAT_BUTTON, false);
	}
}
