package com.limegroup.gnutella.gui.search;

import com.sun.java.util.collections.*;
import com.limegroup.gnutella.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.xml.*;
import java.io.IOException;

/** 
 * The underlying model for one line of a result table. These are also nodes in
 * the JTree backing the JTreeTable.  Leaf nodes represent a single result from
 * a single host.  Non-leaf nodes represent groups of similar results.  A
 * special non-leaf node also exists to represent the root of all searches.<p>
 *
 * Design note: this class has a minimal set of methods and package access
 * variables.  This is intentional.  Because ResultTableModel must implement
 * many TreeModel methods like getChild(Object node, int i), it made sense to 
 * put all such methods in that class.  So if the representation of TableLine
 * changes, the changes will be localized in ResultTableModel.<p>
 *
 * <b>This class is not thread-safe.  A TableLine should only be used by
 * one thread.</b>
 */
public class TableLine {
    /** The processed version of the filename used for approximate matching.
     *  Not allocated until a match must be done.  The assumption here is that
     *  all matches will use the same ApproximateMatcher.  TODO3: when we move
     *  to Java 1.3, this should be a weak reference so the memory is reclaimed
     *  after GC. */
    private String processedFilename=null;

    /** If children==null, this has no children.  Otherwise its children are
     *  similar TableLine's.  Also, the top-most result node has a list of all
     *  children.  */
    volatile List /* of TableLine */ children;
    /** 
     * If this is not root, then this is the childName'th child of its parent, 
     *  i.e., parent.getChildAt(childName)==this. */
    public int childName=0;
    /**
     * If this is not the root, or a child of the root (ie, it is a grandchild),
     * then this is the parent.
     */
    public TableLine parentLine = null;
    
    /**
     * The underlying RemoteFileDesc from where this get's it's display values
     */ 
    private final RemoteFileDesc _rfd;

    //Used to create ResultSpeed object -- never used for anything else.
    private boolean _isMeasuredSpeed;

    //////////////////Cached Properties only for root node ////////////////
    //Sumeet:TODO1: add these so the root node can show "best values"
    ResultSpeed _speed = null;
    Integer _quality = null;
    Boolean _chatEnabled = null;
    LimeXMLDocument _doc = null;
    CachingEndpoint _location = null;
    //We can assert that these are null if the this is a child
    
    
    ///////////////////////////////Constructors////////////////////////////


    /** Creates a root TableLine.  This should not be displayed. */
    TableLine() {
        this.children=new ArrayList();
        this._rfd = null;//root TableLine has not visible properties
    }   

    /** Creates a standard TableLine to represent one result (no children).
     * This may be upgraded to multiple results later.  The childName is 
     * undefined. 
     * @param isMeasuredSpeed true iff the speed was measured, not
     * set by user.  (This is determined by looking at QHD.)
     */
    TableLine(RemoteFileDesc rfd, boolean measuredSpeed) {
        _rfd = rfd;
        _isMeasuredSpeed = measuredSpeed;
    }

    /** Replaces the shallow copy constructor. Creates a new table line 
     * that SHARES all of this state EXCEPT this' childName and children.  
     * The new TableLine  has no children and an undefined childName.  
     * <p>
     * Changed from copy constructor to  clone because, we wanted to 
     * be able to dynamically clone an XMLTable line when we have one, which
     * is not possible with a copy constructor.
     */
    TableLine createClone() {
        return new TableLine(_rfd,_isMeasuredSpeed);
    }

    /** 
     * @requires 0<=field<=5
     * @effects creates a new comparator that compares TableLines by the
     *  give field.  I.e., if field==1, they are compared by size.
     */
    static Comparator newComparator(final Pair pair, boolean ascending) {
        return new TableLineComparator(pair, ascending);
    }

    /////////////////////Setter Methods//////////////////////
    /////////////////All package access/////////////////////
    void setSpeed(ResultSpeed speed) {
        Assert.that(children!=null,"Setting speed for a leaf node");
        _speed = speed;
    }
    
    void setQuality(int quality) {
        Assert.that(children!=null, "Setting quality for a leaf node");
        _quality = new Integer(quality);
    }

    void setChatEnabled(boolean chatEnabled) {
        Assert.that(children!=null,"Setting chatEnabled for a leaf node");
        _chatEnabled = new Boolean(chatEnabled);
    }
    
    void setXMLDocument(LimeXMLDocument doc) {
        Assert.that(children!=null,"Setting document for leaf node");
        _doc = doc;
    }

    void setLocation(CachingEndpoint location) {
        Assert.that(children!=null,"Setting location for leaf node");
        _location = location;
    }

    /////////////////////////Accessors/////////////////////////
    ResultSpeed getSpeed() {
        if(_speed==null) //leaf?
            return new ResultSpeed(_rfd.getSpeed(),_isMeasuredSpeed);
        return _speed;
    }
    
    int getQuality() {
        if(_quality==null) //leaf?
            return _rfd.getQuality();
        return _quality.intValue();
    }
    
    boolean getChatEnabled() {
        if(_chatEnabled==null) 
            return _rfd.chatEnabled();
        return _chatEnabled.booleanValue();
    }
    
    LimeXMLDocument getXMLDocument() {
        if(_doc == null) 
            return _rfd.getXMLDoc();
        return _doc;
    }
    
    CachingEndpoint getLocation() {
        if(_location == null)
            return new CachingEndpoint
            (_rfd.getHost(),_rfd.getPort(),_rfd.isReplyToMulticast());
        return _location;
    }
    
    private String getProcessedFilename(ApproximateMatcher matcher) {
        if(processedFilename!=null)
            return processedFilename;
        if(_rfd==null)
            return "";
        processedFilename = matcher.process(getFilenameNoExtension());
        return processedFilename;
    }
    
    String getFilenameNoExtension() {
        if(_rfd==null)
            return "";
        String fullname = _rfd.getFileName();
        int i = fullname.lastIndexOf(".");
        if(i<0)
            return fullname;
        return fullname.substring(0,i);
    }

    int getSize() {
        return _rfd.getSize();
    }


    String getExtension() {
        if(_rfd==null)
            return "";
        String fullname = _rfd.getFileName();
        int i = fullname.lastIndexOf(".");
        if(i<0)
            return "";
        return fullname.substring(i+1);
    }

    /** 
	 * Returns <tt>true</tt> if browse host feature is enabled 
     * for this <tt>TableLine</tt>, <tt>false</tt> otherwise.
	 * @return <tt>true</tt> if browse host feature is enabled 
     * for this <tt>TableLine</tt>, <tt>false</tt> otherwise.
	 */
	final boolean getBrowseHostEnabled() {
		return _rfd.browseHostEnabled();
	}

    /**
     * Returns this filename, as passed to the constructor.  Limitation:
     * if the original filename was "a.", the returned value will be
     * "a".
     */
    String getFilename() {
        return _rfd.getFileName();
    }

    /**
     * Returns the vendor code of the result - not a default column but useful.
     */
    String getVendor() {
        return _rfd.getVendor();
    }
    
    /**
     * Return SHA1 URN, if available, null otherwise.
     */
    String getSHA1() {
        URN urn = _rfd.getSHA1Urn();
        if(urn == null) return null;
        return urn.toString();
    }

    /** returns the hostname if available*/
    String getHostname() {
        return _rfd.getHost();
    }

    /**
     * The new way of getting a value. The responsibility of determining
     * what object to return based on the index of the column now 
     * lies on the TableLine. This (basic) version of TableLine
     * decided what obect to return based in the value of the column index,
     * based on the assumption that the result Panel is displaying 
     * results for a search that is NOT rich
     * <p>
     * The XMLTableLine class overrides this method only, and returns 
     * Objects based on the schema of the search for which this TableLine
     * was constrcuted. 
     */
    public Object getValue(LimeXMLSchema schema, int index){
        LimeXMLDocument doc = _rfd.getXMLDoc();
        if(schema == null) {
            switch (index) {
            case DisplayManager.NAME_COL: return getFilenameNoExtension();
            case DisplayManager.EXTENSION_COL: return getExtension();
            case DisplayManager.SIZE_COL: 
                return GUIUtils.toUnitbytes(_rfd.getSize());
            case DisplayManager.SPEED_COL: return getSpeed();
            case DisplayManager.LOCATION_COL: return getLocation();
            case DisplayManager.QUALITY_COL: return new Integer(getQuality());
            case DisplayManager.CHAT_COL: return new Boolean(getChatEnabled());
            case DisplayManager.GROUP_COUNT: 
                if(getLocation().numLocations()>1)
                    return ""+getLocation().numLocations();
                return "";
            case DisplayManager.VENDOR_COL: return _rfd.getVendor();
            default:
                Assert.that(false, "Bad column in table "+index);
                return null;
            }
        }
        else if(doc==null)//valid schema...check we have valid doc
            return "";
        else {
            if(doc.getSchemaURI().equals(schema.getSchemaURI())) {
                //OK. get the index'th field.
                List fields = schema.getCanonicalizedFields();
                String fieldName=((SchemaFieldInfo)fields.get(index)).
                getCanonicalizedFieldName();
                String ret = doc.getValue(fieldName);
                if (ret==null)
                    return "";
                return ret;
            }
            else
                return "";
        }
    }

    /** 
     * Compares this against o approximately:
     * <ul>
     *  <li> Returns 0 if o is similar to this. 
     *  <li> Returns 1 if they have non-similar extensions.
     *  <li> Returns 2 if they have non-similar sizes.
     *  <li> Returns 3 if they have non-similar names.
     * <ul>
     *
     * Design note: this takes an ApproximateMatcher as an argument so that many
     * comparisons may be done with the same matcher, greatly reducing the
     * number of allocations.<b>
     *
     * <b>This method is not thread-safe.</b>
     */
    int match(TableLine o, final ApproximateMatcher matcher) {
        return matchInternal(o,matcher);
    }

    protected int matchInternal(TableLine o, final ApproximateMatcher matcher) {
        //Preprocess the processed fileNames
        getProcessedFilename(matcher);
        o.getProcessedFilename(matcher);

        //Same file type?
        if (! getExtension().equals(o.getExtension()))
            return 1;


		float floatSizeThis = (float)getSize();
		float floatSizeNew  = (float)o.getSize();

        //Similar file size (within 60k)?  This value was determined empirically
        //to optimize grouping aggressiveness with minimal performance cost.
        float sizeDiff=Math.abs(floatSizeThis - floatSizeNew);
        if (sizeDiff > 60000) 
            return 2;
        //Filenames close?  This is the most expensive test, so it should go
        //last.  Allow 5% edit difference in filenames or 4 characters,
        //whichever is smaller.
        int allowedDifferences=Math.round(Math.min(
             0.10f*((float)getFilenameNoExtension().length()),
             0.10f*((float)o.getFilenameNoExtension().length())));
        allowedDifferences=Math.min(allowedDifferences, 4);
        if (! matcher.matches(getProcessedFilename(matcher), 
                              o.getProcessedFilename(matcher),
                              allowedDifferences))
            return 3;
        return 0;
    }

    /** Returns a new RemoteFileDesc with this name, port, etc. */
    public RemoteFileDesc toRemoteFileDesc() {
        return _rfd;
    }

    public String toString() {
        return getFilenameNoExtension();//needed for TableCellRenderer
    }

	/**
	 * Returns whether or not this <tt>TableLine</tt> has children.
	 *
	 * @return <tt>true if this <tt>TableLine</tt> has children,
	 *  <tt>false</tt> otherwise
	 */
	final boolean hasChildren() {
		return (children != null);
	}

	/**
	 * Chats with the host for this table line if they are chat-enabled
	 * and chats with the first chat-enabled host in the group if this
	 * is a parent node.  If no host related to this <tt>TableLine</tt>
	 * is chat enabled, then the chat will fail silently.
	 */
	final void doChat() {
		if(!this.getChatEnabled()) return;
		String host = this.getChatEnabledHost();
		if(host == null) return;
		int port = this.getChatEnabledPort();
		if(port == -1) return;
		RouterService.createChat(host, port);
	}
    
    /**
	 * Opens external web browser to Bitzi info page for
     * selected files' SHA1 value
	 */
	final void doBitziLookup() {
        String urn = getSHA1();
		if(urn==null) return;
        int hashstart = 1+urn.indexOf(":",4);
        // TODO: this url could come from a template set elsewhere
        String lookupUrl = "http://bitzi.com/lookup/"+urn.substring(hashstart)+"?ref=limewire";
        try {
            Launcher.openURL(lookupUrl);
        } catch (IOException ioe) {
            // do nothing
        }
	}


	/**
	 * Obtains the ip address string of the first chat-enabled host for
	 * this <tt>TableLine</tt>, returning the ip address stored in this
	 * <tt>TableLine</tt> if it is not a parent, and the first chat-enabled
	 * host among its children if it is a parent.  If no host related to
	 * this <tt>TableLine</tt> is chat-enabled, this method returns 
	 * <tt>null</tt>.
	 *
	 * @return the ip address string of the first chat-enabled host related
	 *  to this <tt>TableLine</tt> instance (stored either in this 
	 *  <tt>TableLine</tt> or in one of its children if it is a parent),
	 *  and <tt>null</tt> if their is no chat-enabled host
	 */
	private final String getChatEnabledHost() {
		if(!getChatEnabled()) return null;
		if(!hasChildren()) return getLocation().getHostname();		
		for(int i=0; i<children.size(); i++) {
			TableLine child = (TableLine)children.get(i);
			if(child.getChatEnabled()) return child.getLocation().getHostname();
		}
		return null;
	}

    /**
     * package access, that returns the string of all the metadata
     */
    public String[] getMetaText(){
        LimeXMLDocument doc = getXMLDocument();
        if (doc == null)
            return DataUtils.EMPTY_STRING_ARRAY;;
        ArrayList retList = new ArrayList();
        LimeXMLSchemaRepository rep = LimeXMLSchemaRepository.instance();
        DisplayManager dispMan = DisplayManager.instance();
        LimeXMLSchema schema = rep.getSchema(doc.getSchemaURI());
        if(schema!=null) {
            Iterator fieldIter=schema.getCanonicalizedFields().iterator();
            while(fieldIter.hasNext()){
                SchemaFieldInfo sf = (SchemaFieldInfo)fieldIter.next();
                String fieldName = sf.getCanonicalizedFieldName();
                String value = doc.getValue(fieldName);
                if(value!=null && !value.equals("") )
                    retList.add(dispMan.getDisplayName
                                (fieldName,schema.getSchemaURI())
                                +": "
                                +value);
            }
        }
        String[] retArray = new String[retList.size()];
        retList.toArray(retArray);
        return retArray;        
    }

    
    /**
	 * Obtains the ip address string of the first browse-host-enabled host for
	 * this <tt>TableLine</tt>, returning the ip address stored in this
	 * <tt>TableLine</tt> if it is not a parent, and the first 
     * browse-host-enabled
	 * host among its children if it is a parent.  If no host related to
	 * this <tt>TableLine</tt> is browse-host-enabled, this method returns 
	 * <tt>null</tt>.
	 *
	 * @return the ip address string of the first 
     * browse-host-enabled host related
	 * to this <tt>TableLine</tt> instance (stored either in this 
	 * <tt>TableLine</tt> or in one of its children if it is a parent),
	 * and <tt>null</tt> if their is no browse-host-enabled host
	 */
	final String getBrowseHostEnabledHost() {
		if(!getBrowseHostEnabled()) return null;
		if(!hasChildren()) return getLocation().getHostname();		
		for(int i=0; i<children.size(); i++) {
			TableLine child = (TableLine)children.get(i);
			if(child.getBrowseHostEnabled()) 
                return child.getLocation().getHostname();
		}
		return null;
	}


	/**
	 * Obtains the port of the first chat-enabled host for this 
	 * <tt>TableLine</tt>, returning the port stored in this
	 * <tt>TableLine</tt> if it is not a parent, and the port of the 
	 * first chat-enabled host among its children if it is a parent.  
	 * If no host related to this <tt>TableLine</tt> is chat-enabled, 
	 * this method returns -1.
	 *
	 * @return the port of the first chat-enabled host related
	 *  to this <tt>TableLine</tt> instance (stored either in this 
	 *  <tt>TableLine</tt> or in one of its children if it is a parent),
	 *  and returns -1 if their is no chat-enabled host
	 */
	private final int getChatEnabledPort() {
		if(!getChatEnabled()) return -1;
		if(!hasChildren()) return getLocation().getPort();
		for(int i=0; i<children.size(); i++) {
			TableLine child = (TableLine)children.get(i);
			if(child.getChatEnabled()) 
                return child.getLocation().getPort();
		}
		return -1;
	}
    
    /**
	 * Obtains the port of the first browse-host-enabled host for this 
	 * <tt>TableLine</tt>, returning the port stored in this
	 * <tt>TableLine</tt> if it is not a parent, and the port of the 
	 * first browse-host-enabled host among its children if it is a parent.  
	 * If no host related to this <tt>TableLine</tt> is browse-host-enabled, 
	 * this method returns -1.
	 *
	 * @return the port of the first browse-host-enabled host related
	 *  to this <tt>TableLine</tt> instance (stored either in this 
	 *  <tt>TableLine</tt> or in one of its children if it is a parent),
	 *  and returns -1 if their is no browse-host-enabled host
	 */
	final int getBrowseHostEnabledPort() {
		if(!getBrowseHostEnabled()) return -1;
		if(!hasChildren()) return getLocation().getPort();
		for(int i=0; i<children.size(); i++) {
			TableLine child = (TableLine)children.get(i);
			if(child.getBrowseHostEnabled()) 
                return child.getLocation().getPort();
		}
		return -1;
	}
    
}

class TableLineComparator implements Comparator {
    int offset;
    boolean ascending;
    LimeXMLSchema schema;

    TableLineComparator(Pair pair, boolean ascending) {
        this.offset = pair.getKey();
        this.schema = (LimeXMLSchema)pair.getElement();
        this.ascending=ascending;
    }

    public int compare(Object a, Object b) {
        TableLine atl=(TableLine)a;
        TableLine btl=(TableLine)b;
        int factor = ascending ? 1 : -1;
        if(schema == null){
            switch (offset) {
            case DisplayManager.NAME_COL:
                return factor * StringUtils.compareIgnoreCase(
                                    atl.getFilenameNoExtension(),
                                    btl.getFilenameNoExtension());
                case DisplayManager.GROUP_COUNT:
                    return factor * -(atl.getLocation().numLocations() - 
                                      btl.getLocation().numLocations());
                case DisplayManager.EXTENSION_COL:
                    return factor * StringUtils.compareIgnoreCase(
                                      atl.getExtension(),btl.getExtension());
                case DisplayManager.VENDOR_COL:
                    return factor * StringUtils.compareIgnoreCase(
                                      atl.getVendor(),btl.getVendor());
                case DisplayManager.SIZE_COL:
                    return factor * -(atl.getSize() - btl.getSize());
                case DisplayManager.SPEED_COL:
                    return factor * -(atl.getSpeed().compareTo(btl.getSpeed()));
                case DisplayManager.QUALITY_COL:
                    return factor * -(atl.getQuality() - btl.getQuality());
                case DisplayManager.CHAT_COL:
                    boolean b1 = atl.getChatEnabled();
                    boolean b2 = btl.getChatEnabled();
                    if(b1 == b2) return 0;
                    if(b1 && !b2) return -factor;			
                    if(!b1 && b2) return factor;			
                case DisplayManager.LOCATION_COL:            
                    //First key: number of locations
                    //Secondary key: isPrivate
                    //Third key: IP address
                    CachingEndpoint ea=atl.getLocation();             
                    CachingEndpoint eb=btl.getLocation();
                    int cmp=factor * (eb.numLocations()-ea.numLocations());  
                    if (cmp!=0)                                  //primary key
                        return cmp;
                    boolean aIsPrivate=ea.isPrivateAddress();
                    boolean bIsPrivate=eb.isPrivateAddress();            
                    if (aIsPrivate!=bIsPrivate)                  //secondary key
                        return factor * (aIsPrivate ? 1 : -1);            
                    return factor * StringUtils.compareIgnoreCase(
                                               atl.getLocation().getHostname(),
                                               btl.getLocation().getHostname());
                default:
                    return 0; //don't sort
                }
            }
            else{//schema column sort
                List fields = schema.getCanonicalizedFields();
                SchemaFieldInfo sf = (SchemaFieldInfo)fields.get(offset);
                Class compClass = sf.getJavaType();//used to figure out how to comp
                String aStr = (String)atl.getValue(schema,offset);
                String bStr = (String)btl.getValue(schema,offset);
                if(compClass.equals(String.class)){//String class
                    return factor * StringUtils.compareIgnoreCase(aStr,bStr);
                }
                else if(compClass.equals(Integer.class)){//Integer class
                    int aVal;
                    int bVal;
                    try{
                        aVal = (new Integer(aStr)).intValue();
                    }catch (NumberFormatException e1){
                        aVal = 0;
                    }
                    try{
                        bVal = (new Integer(bStr)).intValue();
                    }catch(NumberFormatException e2){
                        bVal = 0;
                    }
                    return factor * -(aVal - bVal);
                }
                else{//this means a Double class
                    double aVal; 
                    double bVal;
                    try{
                        aVal = (new Double(aStr)).doubleValue();
                    }catch(NumberFormatException e3){
                        aVal = 0;
                    }
                    try{
                        bVal= (new Double(bStr)).doubleValue();
                    }catch (NumberFormatException e4){
                        bVal = 0;
                    }
                    return (int)(factor * -(aVal - bVal));
                }
            }
        }
        
        public boolean equals(Object o) {
            //TODO: why the hell did I do this?  Was it just for testing?
            return false;
        }
}



