package com.limegroup.gnutella.update;

import java.net.URLEncoder;
import java.io.IOException;

/**
 * This class handles creating the update information for client requesting
 * the update.  It constructs the appropriate update information based
 * on the data supplied by the client, such as the operating system, the
 * LimeWire version, the version of ads the client is serving, etc.<p>
 *
 * This class is reconstructed on the client side by the 
 * <tt>ClientRemoteUpdateInfo</tt> class.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class ServletRemoteUpdateInfo extends AbstractRemoteUpdateInfo {

	/**
	 * The url for the ad war file to download.
	 */
	private final String AD_URL =
		"http://version.limewire.com/updates/ads.war";

    /** 
     * Variable for the directive, or type of update, to perform.
     */
    private String _directive = NO_UPDATE_STRING;

    /** 
     * Variable for the url(s) to use for the update
     */
    private String _url = "";

    /** 
     * Variable for the message to display to the user.
     */
    private String _message = "";

    /** 
     * Variable for the classpath that should be used on the local machine.
     */
    private String _classpath = "";

    /** 
     * Variable for the main class to call on the local machine.
     */
	private String _mainClass = "";

	/**
	 * Useful constructor for unit tests.
	 */
	private ServletRemoteUpdateInfo() {}

	/**
	 * List of versions to ignore.
	 */
	private final String[] IGNORE_LIST = {
		"@version@",
	};

    /**
     * Constructs a new <tt>ServletRemoteUpdateInfo</tt> instance based on the 
     * the data in the <tt>ServletLocalUpdateInfo</tt> instance.<p> 
     * 
     * This constructor is used on the servlet to generate the appropriate 
     * update data based on the information supplied by the client requesting 
     * the update.
     *
     * @param localInfo the <tt>ServletLocalUpdateInfo</tt> instance 
     *  containing data about the cleint requesting the update
     */
    public ServletRemoteUpdateInfo(final ServletLocalUpdateInfo localInfo) 
		throws java.io.IOException {          
        String version     = localInfo.getLimeWireVersion();
        String os          = localInfo.getOS();	
        String javaVersion = localInfo.getJavaVersion();
		//String adVersion = localInfo.getAdVersion();

		// turn off updates
        //if(true) return;

        if(version == null || version.equals("")) 
            throw new IOException("invalid version");        
        if(os == null || os.equals(""))      
            throw new IOException("invalid operating system");

		// activate this to turn updates off
        //if(true) return;

		for(int i=0; i<IGNORE_LIST.length; i++) {            
            if(version.endsWith(IGNORE_LIST[i])) return;
		}

		boolean isPro = false;
		if(version.endsWith("Pro")) isPro = true;

        // ignore any letters in the version number, so "1.5c" will be
        // reduced to "1.5".
        version = version.substring(0,3);   
        if(version.equals("2.9")) return;

        // if it's a 2.8.x version and the jvm is 1.4, then they've
        // already received the update message via distributed update,
        // so don't give them another one
        if(version.equals("2.8") && javaVersion.startsWith("1.4")) 
            return;

        // if this is a version over 3.0, ignore it completely
        String majorVersion = version.substring(0,1);
        int mv = Integer.parseInt(majorVersion);
        if(mv >= 3) return;

        // compare the version passed in to the servlet with the most recent
        // version available.  if the client version is less than the most
        // recent version on the server, then there is an update.  otherwise,
        // there is not an update, so return.
		//if(version.compareTo(LIMEWIRE_VERSION_ON_SERVER_TO_COMPARE) < 0) {
		if(!isPro && !version.equals(LIMEWIRE_VERSION_ON_SERVER_TO_COMPARE)) {
			setValuesForFreeVersionUpdate(version, os);
		} else if(isPro && 
                  !version.equals(LIMEWIRE_VERSION_ON_SERVER_TO_COMPARE)) {
			setValuesForProVersionUpdate(version, os);
		}
	}

	/**
	 * Sets an appropriate values for an ad update, first making sure
	 * that the LimeWire version and the OS both support it.
	 */
	private void setValuesForAdUpdate(String lwVersion, String os) {
		// if this is any LimeWire 1.X product, it does not have
		// ad update available (excluding maybe some random betas),
		// so just return
		if(lwVersion.startsWith("1.")) return;

		// also, this is currently not supported on Windows, so 
		// ignore Windows.
		//if(os.indexOf("Win") != -1) return;

		// otherwise, we can just go ahead and do it
		_directive = UPDATE_ADS_STRING;
		_url = AD_URL;
	}

	/**
	 * This should only be called when there is a newer version 
	 * available on the server.  This sets the appropriate update
	 * properties in that case.
	 */
	private void setValuesForFreeVersionUpdate(String version, String os) {
        // the version on the server is newer than the version in the
        // client now we need to determine whether to update the jar, 
        // send them to a web page, or display a message to the user.
        if(os.indexOf("Win") != -1) {
			// the client is using Windows
			setValuesForFreeWindowsUpdate(version);
        } else if(os.indexOf("Sol") != -1) {
			// the client is using Solaris
			setValuesForFreeOtherUpdate(version);
        } else if(os.indexOf("Lin") != -1) {
            // the client is using Linux
			setValuesForFreeOtherUpdate(version);
        } else if(os.startsWith("Mac")) {
            if(os.endsWith("X")) {
                // the client is using OSX
				setValuesForFreeOSXUpdate(version);
            } else {
                // the client is using Mac Classic
				//if(version.compareTo("1.8") < 0) {
				//setValuesForFreeMacUpdatePre18();
				//} else {
				setValuesForFreeMacUpdate(version);
					//}
            }
        } else {
            // we're probably on some other unix variety
			setValuesForFreeOtherUpdate(version);
        }
	}

	/**
	 * This should only be called when there is a newer version 
	 * available on the server.  This sets the appropriate update
	 * properties in that case.
	 */
	private void setValuesForProVersionUpdate(String version, String os) {
        // the version on the server is newer than the version in the
        // client now we need to determine whether to update the jar, 
        // send them to a web page, or display a message to the user.
        if(os.indexOf("Win") != -1) {
			// the client is using Windows
			setValuesForProWindowsUpdate(version);
        } else if(os.indexOf("Sol") != -1) {
			// the client is using Solaris
			setValuesForProOtherUpdate(version);
        } else if(os.indexOf("Lin") != -1) {
            // the client is using Linux
			setValuesForProOtherUpdate(version);
        } else if(os.startsWith("Mac")) {
            if(os.endsWith("X")) {
                // the client is using OSX
				setValuesForProOSXUpdate(version);
            } else {
                // the client is using Mac Classic
				setValuesForProMacUpdate(version);
            }
        } else {
            // we're probably on some other unix variety
			setValuesForProOtherUpdate(version);
        }

	}
    
	/**
	 * Returns a string in url encoding containing the data for the remote
	 * remote update.<p>
	 *
     * This method is used on the servlet to obtain the url-encoded string
     * of the update action to take to pass back to the client.
     *
	 * @return an url-encoded <tt>String</tt> containing all of the 
	 *         necessary fields for performing the update
	 */
    public String getURLEncodedString() {
		StringBuffer sb = new StringBuffer();
		sb.append(DIRECTIVE);
		sb.append("=");
		sb.append(URLEncoder.encode(_directive));
		sb.append("&");
		sb.append(UPDATE_URL);
		sb.append("=");
		sb.append(URLEncoder.encode(_url));
		sb.append("&");
		sb.append(MESSAGE_TO_DISPLAY);
		sb.append("=");
		sb.append(URLEncoder.encode(_message));
		sb.append("&");
		sb.append(UPDATE_CLASSPATH);
		sb.append("=");
		sb.append(URLEncoder.encode(_classpath));
		sb.append("&");
		sb.append(UPDATE_MAIN_CLASS);
		sb.append("=");
		sb.append(URLEncoder.encode(_mainClass));
		return sb.toString();	
    }

	
	
	/**
	 * Constant for the major version to update to.  If we're updating to
	 * 2.5.5, for example, the major version is 2.
	 */
	private static final String MAJOR_VERSION = "3";

	/**
	 * Constant for the minor version to update to.  If we're updating to
	 * 2.5.5, for example, the major version is the first 5.
	 */
	private static final String MINOR_VERSION = "0";

	/**
	 * Constant for the minor version to update to.  If we're updating to
	 * 2.5.5, for example, the minor version is the second 5.
	 */
	private static final String BUILD_VERSION = "2";

    /**
     * Constant string for the current most recent version.
     */
    static final String LIMEWIRE_VERSION_ON_SERVER = 
		MAJOR_VERSION + "." + MINOR_VERSION + "." + BUILD_VERSION;

    /**
     * Constant string for the current most recent version.
     */
    private static final String LIMEWIRE_VERSION_ON_SERVER_TO_COMPARE = 
		MAJOR_VERSION + "." + MINOR_VERSION;

    /**
     * Constant string for the name of the update jar if it's used.
     */
    private static final String UPDATE_JAR_NAME = 
		"LimeWire"+MAJOR_VERSION+MINOR_VERSION+BUILD_VERSION+".jar";

    /**
     * Constant string for the current most recent version of ads
	 * on the server.
     */
    private final String AD_VERSION_ON_SERVER = "1.1";	

    /**
     * Message for informing free version users of the new version and its 
     * features.
     */
    private final static String FREE_UPDATE_NOW_MESSAGE = 
        "Upgrade to LimeWire " + LIMEWIRE_VERSION_ON_SERVER +"!  "+
        "This version includes better search results, "+
        "especially for rare files, more robust downloads, decreased "+
        "CPU, memory, and bandwidth use, and much more. "+
        "Would you like to update now?";

    private final static String STRONG_FREE_UPDATE_NOW_MESSAGE = 
        "Upgrade to LimeWire " + LIMEWIRE_VERSION_ON_SERVER +"!  "+
        "ALL USERS OF LIMEWIRE VERSIONS PRIOR TO 2.6 MUST UPGRADE NOW!  "+
        "THE NEW VERSIONS INCLUDE GWEBCACHE SUPPORT, WHICH IS REQUIRED TO "+
        "CONNECT TO THE NETWORK!  This version also includes better "+
        "search results and faster downloads.  Update now?";


    /**
     * Message for informing pro version users of the new version and its 
     * features.
     */
    private final static String PRO_UPDATE_NOW_MESSAGE = 
          "Upgrade to LimeWire " + LIMEWIRE_VERSION_ON_SERVER +" Pro!  "+
        "This version includes better search results, "+
        "especially for rare files, more robust downloads, decreased "+
        "CPU, memory, and bandwidth use, and much more. "+
        "Would you like to update now?";


    private final static String STRONG_PRO_UPDATE_NOW_MESSAGE = 
        "Upgrade to LimeWire " + LIMEWIRE_VERSION_ON_SERVER +" Pro!  "+
        "ALL USERS OF LIMEWIRE VERSIONS PRIOR TO 2.6 MUST UPGRADE NOW!  "+
        "THE NEW VERSIONS INCLUDE GWEBCACHE SUPPORT, WHICH IS REQUIRED TO "+
        "CONNECT TO THE NETWORK!  This version also includes better "+
        "search results and faster downloads.  Update now?";

    /**
     * Message to send to platforms that will only receive a message
     * about the update
     */
    private final String FREE_DISPLAY_MESSAGE_UPDATE_MESSAGE = 
        "Version " + LIMEWIRE_VERSION_ON_SERVER + " of LimeWire is "+
        "available from www.limewire.com. Would you like to start your old "+
        "version now?";

    /**
     * Message to send to pro users on platforms that can only receive a message
     * about the update.
     */
    private final String PRO_DISPLAY_MESSAGE_UPDATE_MESSAGE = 
        "Version " + LIMEWIRE_VERSION_ON_SERVER + " of LimeWire Pro is "+
        "available, and you can access it from the web page that we sent you "+
		"when you first purchased LimeWire Pro.  Would you like to start your "+
        "old version now?";

    /**
     * Constant main class value to supply to the client.
     */
	private final String MAIN_CLASS = "com.limegroup.gnutella.gui.Main";
    
    
	private static final String COLLECTIONS  = "collections.jar";
	private static final String XERCES       = "xerces.jar";
	private static final String PLAYER       = "jl011.jar";
	private static final String MESSAGES     = "MessagesBundles.jar";
	private static final String OLD_MESSAGES = "MessagesBundle.properties";

    /**
     * Constant classpath string for non-windows updates.
     */
    private final String NON_WINDOWS_CLASSPATH = 
		UPDATE_JAR_NAME;

    /**
     * Constant classpath string for the update.
     */
    private final String WINDOWS_CLASSPATH = 
		UPDATE_JAR_NAME;

    /**
     * The url of the free update web page.
     */
    private final String WEB_FREE_UPDATE_URL = 
		"http://www.limewire.com/index.jsp/download";

    /**
     * The url of the update web page for pro users.
     */
    private final String WEB_PRO_UPDATE_URL = 
		"http://www.limewire.com/index.jsp/update";

	
	///////////////////////////////////////////////////
	////////             JAR UPDATES           ////////
	///////////////////////////////////////////////////
	/**
	 * Constant for the top-level url for updates.
	 */
	private static final String BASE_UPDATE_URL =
		"http://version.limewire.com/updates/";

	/**
	 * Constant for the top-level url for pro updates.
	 */
	private static final String PRO_UPDATE_URL =
		BASE_UPDATE_URL + "pro/";

	/**
	 * Constant for the top-level url for free updates.
	 */
	private static final String FREE_UPDATE_URL =
		BASE_UPDATE_URL + "free/";

	/**
	 * Constant URL for the new MessagesBundle jar file
	 */
	private static final String MESSAGES_BUNDLE_URL =
		BASE_UPDATE_URL + MESSAGES;

	/**
	 * Constant URL for the new MessagesBundle file in the legacy style --
	 * the raw .properties file
	 */
	private static final String OLD_MESSAGES_BUNDLE_URL =
		BASE_UPDATE_URL + OLD_MESSAGES;
		
	/**
	 * Constant URL for the xml.war file.
	 */
	private static final String XML_URL =
		BASE_UPDATE_URL + "xml.war";

	/**
	 * Constant URL for the ads file.
	 */
	private static final String ADS_URL =
		BASE_UPDATE_URL + "ads.war";

	/**
	 * The url to use for a jar update for pro users.
	 */
	private final String PRO_JAR_UPDATE_URL =
		PRO_UPDATE_URL + UPDATE_JAR_NAME +
		";" + MESSAGES_BUNDLE_URL +
		";" + XML_URL +
		";" + OLD_MESSAGES_BUNDLE_URL;

	/**
	 * The url to use for a jar update for pro users.
	 */
	private final String FREE_JAR_UPDATE_URL =
		FREE_UPDATE_URL + UPDATE_JAR_NAME +
		";" + ADS_URL +
		";" + MESSAGES_BUNDLE_URL +
		";" + XML_URL +
		";" + OLD_MESSAGES_BUNDLE_URL;
	///////////////////////////////////////////////////
	////////            /JAR UPDATES           ////////
	///////////////////////////////////////////////////


	///////// Pro version updates /////////
	private void setValuesForProWindowsUpdate(String lwVersion) {
		_directive = OPEN_WEB_PAGE_STRING;
		_url       = WEB_PRO_UPDATE_URL;
        if(!lwVersion.startsWith("2.6") &&
           !lwVersion.startsWith("2.7") &&
		   !lwVersion.startsWith("2.8")) {
            _message = STRONG_PRO_UPDATE_NOW_MESSAGE;
        } else {
            _message = PRO_UPDATE_NOW_MESSAGE;
        }
	}

	private void setValuesForProMacUpdate(String lwVersion) {
		setValuesForProWindowsUpdate(lwVersion);
	}

	private void setValuesForProOSXUpdate(String lwVersion) {
		setValuesForProWindowsUpdate(lwVersion);
	}

	private void setValuesForProOtherUpdate(String lwVersion) {
		_directive = DISPLAY_MESSAGE_STRING;
		_message   = PRO_DISPLAY_MESSAGE_UPDATE_MESSAGE;
	}

	///////// Free version updates /////////
	private void setValuesForFreeWindowsUpdate(String lwVersion) {
		_directive = OPEN_WEB_PAGE_STRING;
		_url       = WEB_FREE_UPDATE_URL;
        if(!lwVersion.startsWith("2.6") &&
           !lwVersion.startsWith("2.7")) {
            _message = STRONG_FREE_UPDATE_NOW_MESSAGE;
        } else {
            _message = FREE_UPDATE_NOW_MESSAGE;
        }
	}

	private void setValuesForFreeMacUpdate(String lwVersion) {
		setValuesForFreeWindowsUpdate(lwVersion);
	}

	private void setValuesForFreeOSXUpdate(String lwVersion) {
		setValuesForFreeWindowsUpdate(lwVersion);
	}

	private void setValuesForFreeOtherUpdate(String lwVersion) {
		_directive = DISPLAY_MESSAGE_STRING;
		_message   = FREE_DISPLAY_MESSAGE_UPDATE_MESSAGE;
	}


	// overrides Object.toString
	public String toString() {
		return ("ServletRemoteUpdateInfo\r\n"+
				"directive:  "+_directive+"\r\n"+
				"url:        "+_url+"\r\n"+
				"message:    "+_message+"\r\n"+
				"classpath:  "+_classpath+"\r\n"+
				"main class: "+_mainClass);
	}

	//
	// UPDATE SERVLET NOTES:
  	//
	// 2.4.0 was also the first version where every platform except
	// for Mac 9.x and below shipped with RunLime.jar.  This means
	// that pro users for all operating systems other than Mac 9.x
	// and below can use the automatic multiple file updates.
	//
	// LimeWire 1.4 was the first LimeWire that used this servlet
	//
	// Due to the fact that java loads files in the current 
	// directory before loading files in the classpath, we also
	// always have to update MessagesBundle.properties in 
	// addition to MessagesBundles.jar
	//
	// NOTE ON CLASSPATH:  Remember that LimeWire 2.4.0 was the 
	// first version that embedded the classpath in the manifest
	// of LimeWire.jar.  This means that the classpath for earlier
	// versions needs to be overidden with just this value

    public static void main(String[] args) {
        String version = args[0];//"2.8.6 Pro";
        version = version.substring(0,3);                
        if(version.equals("2.8")) {
            System.out.println("2.8...returning..."); 
            return;
        }
        if(version.equals("2.9")) {
            System.out.println("2.9...returning..."); 
            return;
        }

        // if this is a version over 3.0, ignore it completely
        String majorVersion = version.substring(0,1);
        int mv = Integer.parseInt(majorVersion);
        if(mv >= 3) {
            System.out.println("3.0 or above...returning..."); 
            return;        
        }
        
        ServletRemoteUpdateInfo updator = new ServletRemoteUpdateInfo();
        String os = "Windows";		
        String[] versions = {
            "2.7.13 Pro",
            "2.7.13",
            "2.6.5"
        };

        for(int i=0; i<versions.length; i++) {
            version = versions[i];
            boolean isPro = false;
            if(version.endsWith("Pro")) isPro = true;
            
            // ignore any letters in the version number, so "1.5c" will be
            // reduced to "1.5".
            version = version.substring(0,3);   
            System.out.println("version: "+version); 
            
            // compare the version passed in to the servlet with the most recent
            // version available.  if the client version is less than the most
            // recent version on the server, then there is an update.  otherwise,
            // there is not an update, so return.
            //if(version.compareTo(LIMEWIRE_VERSION_ON_SERVER_TO_COMPARE) < 0) {
            if(!isPro && !version.equals(LIMEWIRE_VERSION_ON_SERVER_TO_COMPARE)) {
                System.out.println("about to set free update"); 
                updator.setValuesForFreeVersionUpdate(version, os);
            } else if(isPro && 
                      !version.equals(LIMEWIRE_VERSION_ON_SERVER_TO_COMPARE)) {
                System.out.println("about to set pro update"); 
                updator.setValuesForProVersionUpdate(version, os);
            }        
        }
    }
}













