package com.limegroup.gnutella.update;

import com.limegroup.gnutella.settings.ApplicationSettings;
import java.io.*;
import java.util.*;

/**
 * This class manages checking the servlet for an update and executing
 * the update based on the response.  It passes off most of this 
 * functionality to auxiliary classes, such as <tt>ServletAccessor</tt>,
 * <tt>RemoteUpdateInfo</tt>, <tt>ClientLocalUpdateInfo</tt>, and the various
 * update classes that implement the <tt>Updator</tt> interface.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class UpdateManager {

	/** 
	 * Final boolean value for whether we are currently testing
	 */
	private final boolean TESTING = false;

	/** 
	 * Constant handle to the <tt>UpdateManager</tt> instance for following 
	 * singleton.
	 */
	private static final UpdateManager INSTANCE = new UpdateManager();

	/**
	 * Variable for the class that will perform the final update.	
	 */
	private Updator _updator;

	/** 
	 * Instance accessor following the singleton pattern.
	 *
	 * @return the <tt>UpdateManager</tt> instance
	 */
	public static UpdateManager instance() {
		return INSTANCE;
	}

	/**
	 * Overrides the default constructor to ensure that this cannot be 
	 * constructed from any other class.
	 */
	private UpdateManager() {}

	/**
	 * Makes the calls to access the servlet and to initiate the update based 
	 * on the update object that is created.
	 */
	public void doUpdate() {
		//if(TESTING) startTest();
		//else performStandardUpdate();
		performStandardUpdate();
	}
	
	/**
	 * Notifies the Updator object that the update should be cancelled
	 */
	public void cancelUpdate() {
		_updator.cancelUpdate();
	}

	/**
	 * Mutator method for the Updator object --
	 * used only for testing purposes.
	 */
	void setUpdator(Updator updator) {
		_updator = updator;
	}

	/**
	 * Performs a standard update without any added testing functionality.
	 */
	private void performStandardUpdate() {
		ClientLocalUpdateInfo localInfo = new ClientLocalUpdateInfo();	
		//localInfo.setLocalData();

		ServletAccessor sa = new ServletAccessor();			
		ClientRemoteUpdateInfo remoteInfo = 
		    sa.getRemoteUpdateInfo(localInfo);
		if(ApplicationSettings.CHECK_AGAIN.getValue()) {
			_updator = remoteInfo.getUpdator();
			_updator.doUpdate();
		}
	}

	/**
	 * testing method that accesses a local file that can be easily 
	 * modified for testing different platforms and LimeWire versions.
	 */
//  	private void startTest() {
//  		Properties props = new Properties();
//  		FileInputStream fis;
//  		try {
//  			fis = new FileInputStream(new File("update.props"));
//  			try {
//  				props.load(fis);
//  			} catch(IOException fne) {
//  				performStandardUpdate();
//  				return;
//  			}
//  		} catch(FileNotFoundException fnfe) {
//  			performStandardUpdate();
//  			return;
//  		} catch(SecurityException se) {
//  			performStandardUpdate();
//  			return;
//  		}
//  		ServletAccessor sa = new ServletAccessor();
//  		String version = props.getProperty("limewire.version");
//  		String os      = props.getProperty("os.version");
//  		//System.out.println("version: "+version+ "os: "+os);
//  		// use the alternate constructor for ClientLocalUpdateInfo 
//  		ClientLocalUpdateInfo localInfo = new ClientLocalUpdateInfo(version, os);
//  		RemoteUpdateInfo remoteInfo = sa.getRemoteUpdateInfo(localInfo);
//  		_updator = remoteInfo.getUpdator();
//  		_updator.doUpdate();
//  	}

}
