/****************************************************************************
 **
 ** Copyright (C) 2001-2005 Tilo Riemer <riemer@lincvs.org> and
 **                         Frank Hemer <frank@hemer.org>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "Commit.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "TextEncoder.h"
#include "CommitDialogImpl.h"

Commit * Commit::commitFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false);
}

Commit * Commit::commitDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true);
}

Commit * Commit::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   if (dir->getType() == DirBase::Cvs) {

      QStringList filesList;
      if (!isDir) {
	 filesList = dir->getSelectedFiles();
	 if (filesList.isEmpty()) {
	    return NULL;
	 }
      }

      Commit * p = new Commit(whatsThisIconSet, parent, workBench, dir, isDir);
      p->m_filesList = filesList;

      CommitDialogImpl * dlg = new CommitDialogImpl(whatsThisIconSet,
	    parent, "CommitDlg",
	    LookAndFeel::g_nonModalF);

      if (filesList.isEmpty()) {
	 dlg->setCaption(tr("Commit -")+" \""+dir->fullName()+"\"");
      } else {
	 dlg->setCaption(tr("Commit -")+" "+masqWs(filesList.join("\" \"")));
      }
      connect(dlg,SIGNAL(acceptCvs()),p,SLOT(acceptCvs()));
      connect(dlg,SIGNAL(reject()),p,SLOT(reject()));

      dlg->updateTemplate();
      dlg->reset();

      p->m_dlg = dlg;

      dlg->show();

      return p;

   } else {
      qDebug("Commit: Method not implemented");
   }
   return NULL;
}

Commit::Commit(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_commentFile(QString::null)
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
   connect(this,SIGNAL(checkInProgress(bool)),parent,SLOT(checkInProgress(bool)));
   connect(this,SIGNAL(showWarning(const QString &,const QString &)),parent,SLOT(showWarning(const QString &,const QString &)));
}

Commit::~Commit() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Commit::acceptCvs() {

   if (!m_workBench->validate(m_dir)) {
      emit showWarning(tr("Commit - Warning"),tr("The former selected dir has\nbeen modified or removed,\naborting"));
      reject();
      return;
   }
   assert(m_dlg);

   CommitDialogImpl * dlg = static_cast<CommitDialogImpl*>(m_dlg);
   assert(dlg);

   QString rev = dlg->getRev();
   QString comment = dlg->getComment();

   delete m_dlg;
   m_dlg = NULL;

   m_commentFile = writeTmpFile(comment);
   QString command = "";
   if (!bRWPermission) command += "-r ";
   if (m_commentFile.length()) {
      command += CvsOptions::cmprStr() +
	 " commit -F " + masqWs(m_commentFile);
   } else {
      // fallback to option -m
      command += CvsOptions::cmprStr() + " commit -m \"";
      command += I18n::g_pTextEncoder->encode(comment);
      command += "\"";
   }
   
   if (rev.length()) {
      command += " -r " + rev;
   }
   
   QString files;
   if (!m_filesList.isEmpty()) {
      QStringList list = m_filesList;
      masqQuoteMarks(&list);
      files = masqWs(list.join("\" \""));
   }
   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, command,
	 files, CVS_COMMIT_CMD,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void Commit::reject() {
   emit deleteObject(this);
}

void Commit::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Commit::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Commit::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   cvsCallFinished();
   if (m_commentFile.length()) QFile::remove(m_commentFile);
   if (failed) {
      reject();
      return;
   }
   assert(m_dir);
   switch( cmd) {
      case NOOP: {
	 break;
      }
      case CVS_COMMIT_CMD: {
	 m_dir->parseCallResult( buf, cmd);
	 emit checkInProgress(true);
	 m_dir->postCallCheck(DirBase::Controled|(m_filesList.isEmpty() ? DirBase::Tree : 0));
	 emit checkInProgress(false);
	 break;
      }
      default: {
	 qDebug("Commit::afterCall: Unknown cmd");
      }
   }
   reject();
}
