/****************************************************************************
 **
 ** Copyright (C) 2001-2005 Tilo Riemer <riemer@lincvs.org> and
 **                         Frank Hemer <frank@hemer.org>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "Edit.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "EditorsDialogImpl.h"

Edit * Edit::editFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, EDIT);
}

Edit * Edit::uneditFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, UNEDIT);
}

Edit * Edit::showEditorsFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, SHOW);
}

Edit * Edit::editDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, EDIT);
}

Edit * Edit::uneditDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, UNEDIT);
}

Edit * Edit::showEditorsDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, SHOW);
}

Edit * Edit::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir, int mode) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   if (dir->getType() == DirBase::Cvs) {
      Edit * p = new Edit(whatsThisIconSet, parent, workBench, dir, isDir, mode);
      p->acceptCvs();
   } else {
      qDebug("Edit: Method not implemented");
   }
   return NULL;
}

Edit::Edit(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir, int mode)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_mode(mode)
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
   connect(this,SIGNAL(checkInProgress(bool)),parent,SLOT(checkInProgress(bool)));
   connect(this,SIGNAL(checkTree(DirBase *)),parent,SLOT(slot_checkStatusOfTree(DirBase *)));
   m_fileList.clear();
}

Edit::~Edit() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Edit::acceptCvs() {
   
   assert(m_dir);

   QString files = "";
   if (!m_isDir) {
      m_fileList = m_dir->getSelectedFiles();
      if (m_fileList.isEmpty()) {
         reject();
         return;
      }
      QStringList list = m_fileList;
      masqQuoteMarks(&list);
      files = masqWs(list.join("\" \""));
   }
   
   QString cvsRoot;
   int cmd;
   bool root;
   switch (m_mode) {
      case EDIT: {
	 cvsRoot = "edit";
	 cmd = CVS_EDIT_CMD;
	 root = NOROOT;
	 break;
      }
      case UNEDIT: {
	 cvsRoot = masqWs(m_dir->getRoot());
	 cmd = CVS_UNEDIT_CMD;//needs removal in cvscontrol::callInteractiveCmd
	 root = false;
	 break;
      }
      case SHOW: {
	 cvsRoot = "editors";
	 cmd = CVS_EDITORS_CMD;
	 root = NOROOT;
	 break;
      }
      default: {
	 assert(false);
      }
   }
   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, cvsRoot,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 root);
}

void Edit::reject() {
   emit deleteObject(this);
}

void Edit::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Edit::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Edit::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   cvsCallFinished();
   if (failed) {
      reject();
      return;
   }
   assert(m_dir);
   switch( cmd) {
      case CVS_UNEDIT_CMD:
      case CVS_EDIT_CMD: {
	 m_dir->parseCallResult( buf, cmd, (m_isDir ? NULL : &m_fileList));
	 emit checkInProgress(true);
	 m_dir->postCallCheck(DirBase::Controled|DirBase::Force);
	 emit checkInProgress(false);
	 break;
      }
      case CVS_EDITORS_CMD: {
	 EditorsDialogImpl * dlg =
	    new EditorsDialogImpl ( m_whatsThisIconSet,
		  m_parent, "EditorsDlg",
		  LookAndFeel::g_nonModalF | WDestructiveClose);
	 
	 dlg->parseEditors (buf, m_dir->repository());
	 dlg->show();
	 break;

	 break;
      }

      default: {
	 qDebug("Edit::afterCall: Unknown cmd");
	 assert(false);
      }
   }
   reject();
}
