/* DefaultEmbedding.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin;

import java.util.HashMap;
import java.util.Map;

/**
 * Default implementation of {@link Embedding}. Stores
 * coordinates in a hash table.
 */
public class DefaultEmbedding implements Embedding {
    
    /**
     * Dimension of this embedding.
     */
    protected int dimension;
    
    //
    private GraphView peer;
    
    //
    private Map<Vertex,double[]> map;
    
    /**
     * Create an embedding of the given dimension for the given peer graph.
     */
    public DefaultEmbedding(GraphView peer, int dimension) {
        this.peer = peer;
        this.dimension = dimension;
        this.map = new HashMap<Vertex,double[]> ();
    }
    
    /**
     * Create a copy of a given embedding which shares the same peer.
     */
    public DefaultEmbedding(EmbeddingView original) {
        this.peer = original.getPeer();
        this.dimension = original.getDimension();
        this.map = new HashMap<Vertex,double[]> ();
        for (Vertex v: peer.vertices())
            map.put(v, original.getCoordinates(v));
    }
    
    // implements EmbeddingView
    public double[] getCoordinates(Vertex v) {
        if (!peer.contains(v))
            throw new IllegalArgumentException  ("Vertex does not belong to the graph");
        double[] result = map.get(v);
        if (result == null) {
            // lazily create all zero coordinates
            result = new double[dimension];
            map.put(v, result);
        }
        return result.clone();
    }
    
    // implements Embedding
    public void setCoordinates(Vertex vertex, double[] coordinates) {
        if (!peer.contains(vertex))
            throw new IllegalArgumentException  ("Vertex does not belong to the graph");
        else if (coordinates == null || coordinates.length != dimension)
            throw new IllegalArgumentException("Invalid coordinates");
        map.put(vertex, coordinates.clone());
    }
    
    // implements Embedding
    public void setDimension(int dimension) {
        if (dimension != this.dimension) {
            map.clear();
            this.dimension = dimension;
        }
    }
    
    // implements EmbeddingView
    public GraphView getPeer() {
        return this.peer;
    }
    
    // implements EmbeddingView
    public int getDimension() {
        return this.dimension;
    }
    
    // implements Embedding
    public void copy(EmbeddingView original) {
        if (this == original)
            return;
        this.peer = original.getPeer();
        this.dimension = original.getDimension();
        map.clear();
        for (Vertex v: peer.vertices())
            map.put(v, original.getCoordinates(v));
    }
    
    // implements Embedding
    public void copy(EmbeddingView original, GraphView newPeer) {
        if (this == original)
            throw new IllegalArgumentException("Cannot make copy of self with different peer");
        GraphView originalPeer = original.getPeer();
        if (newPeer == originalPeer)
            copy(original);
        else {
            this.peer = newPeer;
            this.dimension = original.getDimension();
            this.map.clear();
            for (Vertex v: originalPeer.vertices())
                map.put(newPeer.getVertex(v.getIndex()), original.getCoordinates(v));
        }
    }
}
