/* GraphListener.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin;

import java.util.EventListener;

/**
 * Listens to changes in a graph model. <p>
 * 
 * <b>Note:</b> It is the responsibility of
 * the listener to keep track of the graph being listened
 * to. This graph is not handed to the listener as a parameter 
 * of the various methods. 
 */
public interface GraphListener extends EventListener {
    
    /**
     * Called after a vertex was added to the graph.
     * @param vertex Vertex that was added to the graph
     */
    public void vertexAdded (Vertex vertex);
    
    /**
     * Called after an edge was added to the graph.
     * @param edge Edge that was added to the graph.
     */
    public void edgeAdded (Edge edge);
    
    /**
     * Called when an edge was removed from the graph.
     */
    public void edgeRemoved (Edge edge);
    
    /**
     * Called when an edge that was previously removed
     * was restored to the graph.
     */
    public void edgeRestored (Edge edge);

    /**
     * Called when a vertex was removed from the graph. Implies that
     * all edges through this vertex were also removed. These removals will
     * not be notified separately.
     * @param vertex vertex that was removed
     */
    public void vertexRemoved (Vertex vertex);
    
    /**
     * Called when a vertex that was previously removed
     * was restored to the graph.
     */
    public void vertexRestored (Vertex vertex);
    
    /**
     * Called when the annotation of a vertex changes.
     */
    public void vertexChanged (Vertex vertex);
    
    /**
     * Called when the annotation of an edge changes.
     */
    public void edgeChanged (Edge edge);
    
    /**
     * Called when the graph was changed in a major way. For instance, when all
     * vertices and edges were removed in one go or when the graph was 
     * reinitialized from a file.
     */
    public void graphChanged ();
    
}
