/* APEngine.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.engine;

import org.grinvin.engine.apengine.LabeledBinaryTree;
import org.grinvin.engine.apengine.LabeledTreeGenerator;
import org.grinvin.engine.apengine.Operators.BinaryOperator;
import org.grinvin.engine.apengine.Operators.Invariant;
import org.grinvin.engine.apengine.Operators.UnaryOperator;
import org.grinvin.engine.apengine.TreeGenerator;
import org.grinvin.invariants.InvariantValue;

/**
 * Proof of concept (quick and dirty) implementation of {@link Engine}.<p> As
 * opposed to {@link QDEngine} this one is implemented in Java.
 */
public class APEngine extends AbstractInequalityEngine {
    
    //
    private LabeledTreeGenerator labeledTreeGenerator;
    
    //
    public APEngine() {
    }
    
    //
    public String run(InvariantValue [][] values) {
        
        try {
            Thread.sleep(2000);
        } catch (InterruptedException e) {
            //
        }
        
        int numberOfInvariants = values[0].length;
        
        labeledTreeGenerator = new LabeledTreeGenerator(new TreeGenerator(), numberOfInvariants, getMainInvariant());
        
        String[] names = new String[numberOfInvariants];
        for (int i=0; i < numberOfInvariants; i++) {
            names[i] = values[0][i].getInvariant().getName();
        }
        
        //LabeledBinaryTree resultTree = nextCorrectLabeledTree(values);
        LabeledBinaryTree resultTree = bestEffortTree(values);
        return translate("(I" + getMainInvariant() + ") <= " + resultTree.toString(), names);
    }
    
    //
    public LabeledBinaryTree nextCorrectLabeledTree(InvariantValue [][] values) {
        LabeledBinaryTree result = labeledTreeGenerator.nextLabeledTree();
        while ((result != null) && (!check(result, values))) {
            result = labeledTreeGenerator.nextLabeledTree();
        }
        return result;
    }
    
    //
    public LabeledBinaryTree bestEffortTree(InvariantValue [][] values) {
        
        int numberOfGraphs = values.length;
        
        LabeledBinaryTree tree = nextCorrectLabeledTree(values);
        double error = valueError(tree, values);
        
        LabeledBinaryTree bestTree = tree.clone();
        double bestError = error;
        
        //System.out.println(treeError(tree) + " " + numberOfGraphs + " " + (treeError(tree) * numberOfGraphs) + " " + tree + " " + error + " " + bestError);
        while(treeError(tree) * numberOfGraphs < bestError) {
            tree = nextCorrectLabeledTree(values);
            error = valueError(tree, values);
            //System.out.println(treeError(tree) + " " + numberOfGraphs + " " + (treeError(tree) * numberOfGraphs) + " " + tree + " " + error + " " + bestError);
            if(error < bestError) {
                bestTree = tree.clone();
                bestError = error;
            }
        }
        //System.out.println(treeError(bestTree) + " " + numberOfGraphs + " " + (treeError(bestTree) * numberOfGraphs) + " " + bestTree + " " + bestError);
        return bestTree;
    }
    
    //
    private int treeError(LabeledBinaryTree tree) {
        return (1 << (  2 * tree.getBinaryCount() + tree.getUnaryCount()));
    }
    
    //
    private double valueError(LabeledBinaryTree tree, InvariantValue[][] values) {
        double worst = 0;
        int worstcount = 0;
        int numberOfGraphs = values.length;
        double result = treeError(tree) * numberOfGraphs;
        for (InvariantValue [] valuelist : values) {
            double sum = valuelist[getMainInvariant()].asDouble() - evaluate(valuelist, tree, 0);
            sum = sum*sum;
            if (Double.isNaN(sum) || Double.isInfinite(sum)) {
                sum = 0;
                worstcount++;
                if (worstcount >= (numberOfGraphs * 2 / 3)) {
                    //return result + (worstcount * worst * 2);
                    return Double.MAX_VALUE;
                }
            }
            if (sum > worst) {
                worst = sum;
            }
            result = result + sum;
        }
        return result + (worstcount * worst * 2);
    }
    
    //
    public boolean check(LabeledBinaryTree tree, InvariantValue[][] values) {
        boolean result = true;
        for (InvariantValue [] valuelist : values) {
            if (valuelist[getMainInvariant()].asDouble() > evaluate(valuelist, tree, 0))
                result = false;
        }
        return result;
    }
    
    //
    private double evaluate(InvariantValue[] valuelist, LabeledBinaryTree tree, int parent) {
        double result = 0;
        if (tree.hasRightChild(parent)) {
            BinaryOperator binop = (BinaryOperator)tree.operators[parent];
            result = binop.eval(evaluate(valuelist, tree, tree.leftChild(parent)), evaluate(valuelist, tree, tree.rightChild(parent)));
        } else if (tree.hasLeftChild(parent)) {
            UnaryOperator monop = (UnaryOperator)tree.operators[parent];
            result = monop.eval(evaluate(valuelist, tree, tree.leftChild(parent)));
        } else {
            Invariant inv = (Invariant)tree.operators[parent];
            result = valuelist[inv.number].asDouble();
        }
        return result;
    }
    
}
