/* EngineRunner.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.engine;

import java.awt.EventQueue;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;

import org.grinvin.invariants.IntegerValue;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.invariants.RealValue;
import org.grinvin.list.GraphInvariantList;

/**
 * Object which can be used to run a given conjecturing engine in a background thread.
 * <p>
 * Allows listeners to be notified when the run starts and when it ends.
 */
public class EngineRunner implements Runnable {
    
    //
    private Engine engine;
    
    //
    private GraphInvariantList model;
    
    /**
     * Return the associated engine.
     */
    public Engine getEngine() {
        return engine;
    }
    
    //
    private EventListenerList listenerList = new EventListenerList();
    
    //
    private static final Class<ChangeListener> CHANGE_LISTENER_CLASS = ChangeListener.class;
    
    //
    private final ChangeEvent EVENT = new ChangeEvent(this);
    
    /**
     * Register an object which is informed of changes in the
     * 'running' state of this runner.
     */
    public void addChangeListener(ChangeListener l) {
        listenerList.add(CHANGE_LISTENER_CLASS, l);
    }
    
    /**
     * Remove a change listener.
     */
    public void removeChangeListener(ChangeListener l) {
        listenerList.remove(CHANGE_LISTENER_CLASS, l);
    }
    
    /**
     * Notify every listener of a change in state.
     */
    protected void fireStateChanged() {
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length-2; i>=0; i-=2)
            // if (listeners[i]==CHANGE_LISTENER_CLASS)
            ((ChangeListener)listeners[i+1]).stateChanged(EVENT);
    }
    
    /**
     * Construct a runner object for the given engine.
     */
    public EngineRunner(Engine engine, GraphInvariantList model) {
        this.engine = engine;
        this.model = model;
    }
    
    //
    private String result;
    
    /**
     * Return the latest result generated by the engine (may be null).
     */
    public String getResult() {
        return this.result;
    }
    
    //
    public enum Status {
        STOPPED,
        INVARIANTS,
        CONJECTURE;
    }
    
    //
    private Status status = Status.STOPPED;
    
    //
    public Status getStatus() {
        return status;
    }
    
    //
    private void setStatus(Status status) {
        this.status = status;
        EventQueue.invokeLater(new Runnable() {
            public void run() {
                fireStateChanged();
            }
        });
    }
    
    public List<Invariant> getUsableInvariants() {
        List<Invariant> usableInvariants = new ArrayList<Invariant>();
        for(Invariant invariant : model.getInvariantList()) {
            if (invariant.getType() == IntegerValue.class || invariant.getType() == RealValue.class)
                usableInvariants.add(invariant);
        }
        return usableInvariants;
    }
    
    public List<Invariant> getSortedUsableInvariants() {
        Set<Invariant> usableInvariants = new TreeSet<Invariant>(new InvariantIdComparator());
        usableInvariants.addAll(getUsableInvariants());
        return new ArrayList<Invariant>(usableInvariants);
    }
        
    //
    private class InvariantIdComparator implements Comparator<Invariant> {
        
        public int compare(Invariant inv1, Invariant inv2) {
            return inv1.getId().compareTo(inv2.getId());
        }
        
    }

    /**
     * Extract the invariant values from the table and run the engine. Changes are
     * reported to all listeners in the event dispatching thread, even when this
     * method is run in a different thread (which is generally a good idea).
     */
    public void run() {
        this.result = null;
        
        List<Invariant> usableInvariants = getSortedUsableInvariants();
        
        int numberOfRows = model.getGraphList().size();
        int numberOfInvariants = usableInvariants.size();
        if (numberOfInvariants == 0 || numberOfRows == 0)
            return; // nothing to do
        setStatus(Status.INVARIANTS);
        InvariantValue[][] values = new InvariantValue[numberOfRows][numberOfInvariants];
        //TODO: what happens if lists change before this loop?
        for (int i=0; i < numberOfRows; i++) {
            for (int j=0; j < numberOfInvariants; j++) {
                values[i][j] = model.getGraphList().get(i).getInvariant(usableInvariants.get(j));
            }
        }
        setStatus(Status.CONJECTURE);
        this.result = engine.run(values);
        setStatus(Status.STOPPED);
    }
    
}
