/* QDEngine.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.engine;

import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import org.grinvin.invariants.InvariantValue;

/**
 * Proof of concept (quick and dirty) implementation of {@link Engine}.<p>
 * Interfaces to a C-program <code>conj</code>
 * written by Gunnar Brinkmann. This program must be part of the execution path of
 * the computer.
 */
public class QDEngine extends AbstractInequalityEngine {

    //
    public String run (InvariantValue[][] values) {
        try {
            ProcessBuilder pb = new ProcessBuilder ("conj", "" + getMainInvariant());
            Process p = pb.start ();
            
            try {
                Thread.sleep (3000);
            } catch (InterruptedException e) {
                //
            }
            
            // cache names of invariants
            int nrOfInvariants = values[0].length;
            String[] names = new String[nrOfInvariants];
            for (int i=0; i < nrOfInvariants; i++) {
                names[i] = values[0][i].getInvariant ().getName ();
            }
            
            // send values
            DataOutputStream output = new DataOutputStream (p.getOutputStream ());
            output.writeInt (values[0].length);
            for (InvariantValue[] row : values) {
                for (InvariantValue ival : row) {
                    //TODO: should conjecturing wait for all invariant values to be computed?
                    if(ival != null)
                        output.writeDouble (ival.asDouble ());
                    else
                        output.writeDouble(Double.NaN);
                }
            }
            output.close ();
            
            // wait for results
            if (p.waitFor () == 0) {
                BufferedReader is = new BufferedReader (new InputStreamReader (p.getInputStream ()));
                String line = is.readLine ();
                is.close ();
                return translate ("(I" + getMainInvariant() + ") <= " + line, names);
            } else {
                BufferedReader is = new BufferedReader (new InputStreamReader (p.getErrorStream()));
                String line = is.readLine ();
                while (line != null) {
                    System.err.println(line);
                    line = is.readLine();
                }
                is.close ();
                return "Error: program terminated with exit code != 0";
            }
        } catch (InterruptedException ie) {
            return "Error: program interrupted: " + ie;
        } catch (IOException ioex) {
            return "I/O error: " + ioex;
        }
    }
    
}
