/* GraphFactoryManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.xml.XMLUtils;

import org.jdom.Element;

/**
 * Keeps track of all graph factories that are currently supported by the system.<p>
 * This is a singleton class. The single shared object of this class can be retreived
 * using the method {@link #getInstance()}.
 */
public final class GraphFactoryManager {
    
    
    //
    private static final GraphFactoryManager SINGLETON =
            new GraphFactoryManager();
    
    /**
     * Retrieve the single (shared) instance of this class.
     */
    public static GraphFactoryManager getInstance() {
        return SINGLETON;
    }
    
    /**
     * Maps an id to the corresponding grapn factory.
     */
    private final Map<String,GraphFactory> factories;
    
    /**
     * Ordered list of ids.
     */
    private final List<String> ids;
    
    
    /** Not to be instantiated */
    private GraphFactoryManager() {
        try {
            this.factories = new HashMap<String,GraphFactory> ();
            this.ids = new ArrayList<String> ();
            Element element = XMLUtils.loadFromClassPath(
                    "org/grinvin/factories/graphs/resources/graphfactories.xml");
            if (element != null) {
                for (Object o : element.getChildren("package")) {
                    Element pkg = (Element)o;
                    String packageName = pkg.getAttributeValue("name");
                    for (Object obj : pkg.getChildren("id")) {
                        Element child = (Element)obj;
                        String id = child.getTextTrim();
                        String className = child.getAttributeValue("class");
                        if (className == null) {
                            int pos = id.lastIndexOf('.');
                            className = id.substring(pos+1);
                        }
                        try {
                            String fullClassName = packageName + "." + className;
                            factories.put(id, (GraphFactory)Class.forName(fullClassName).newInstance());
                            ids.add(id);
                        } catch (Exception ex) {
                            Logger.getLogger("org.grinvin.factories.graphs")
                            .log(Level.WARNING,
                                    "Failed to create graph factory '" + id + "' of type '" + className +"'",
                                    ex);
                        }
                        
                    }
                }
            }
        } catch (IOException ex) {
            throw new RuntimeException("Unable to initialize graph factory manager", ex);
        }
    }
    
    /**
     * Add all graph factories to the given list.
     */
    public void addToList(List<GraphFactory> list) {
        for (String id : ids)
            list.add(factories.get(id));
        // list.addAll (factories.values()) cannot be used because it loses the order
    }
    
    /**
     * Return the list of all ids, in the form of an <code>Iterable</code>.
     */
    public Iterable<String> getIds() {
        return ids;
    }
    
    /**
     * Return the factory with the given id.
     */
    public GraphFactory getGraphFactory(String id) {
        return factories.get(id);
    }
}
