/* InvariantFactoryDropHandler.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphbrowser;

import be.ugent.caagt.swirl.dnd.DropHandler;
import java.text.MessageFormat;

import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JComponent;
import javax.swing.JOptionPane;

import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryManager;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantFactory;
import org.grinvin.invariants.InvariantManager;

/**
 * Allows an invariant factory to be dropped onto objects that support
 * {@link HasInvariantList}. After dropping, the user is asked
 * to present parameters for the invariant.
 */
public class InvariantFactoryDropHandler implements DropHandler {
    
    //
    private static final Class INVARIANT_FACTORY = InvariantFactory.class;
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle("org.grinvin.graphbrowser.resources");
    
    // cannot be instantiated by clients
    private InvariantFactoryDropHandler() {
        // empty
    }
    
    /**
     * The unique shared object of this type.
     */
    public static final InvariantFactoryDropHandler DROP_HANDLER = new InvariantFactoryDropHandler();
    
    public boolean allowsMultipleDrops(JComponent target) {
        return false;
    }
    
    public Class<?> getDropClass(JComponent target) {
        return INVARIANT_FACTORY;
    }
    
    public boolean acceptDrop(JComponent target, Object object, int seqNr) {
        if (seqNr > 0)
            return false;
        if (target instanceof HasInvariantList && object instanceof InvariantFactory) {
            try {
                InvariantFactory factory = (InvariantFactory)object;
                if (FactoryManager.configureFactory(factory)) {
                    Invariant inv = InvariantManager.getInstance().getInvariantForFactory(factory);
                    if(!((HasInvariantList)target).getInvariantList().contains(inv)){
                        ((HasInvariantList)target).getInvariantList().add(inv);
                        return true;
                    } else {
                        JOptionPane.showMessageDialog(target, MessageFormat.format(BUNDLE.getString("InvariantList.duplicate"), inv), BUNDLE.getString("InvariantList.warning.title"), JOptionPane.DEFAULT_OPTION);
                        return false;
                    }
                } else {
                    return false;
                }
            } catch (FactoryParameterException ex) {
                Logger.getLogger("org.grinvin.factories").log(Level.WARNING, "Failed to create invariant from factory", ex);
                return false;
            } catch (FactoryException ex) {
                Logger.getLogger("org.grinvin.factories").log(Level.WARNING, "Failed to create invariant from factory", ex);
                return false;
            }
        } else if (target instanceof AcceptsInvariant && object instanceof InvariantFactory) {
            try {
                InvariantFactory factory = (InvariantFactory)object;
                if (FactoryManager.configureFactory(factory)) {
                    Invariant inv = InvariantManager.getInstance().getInvariantForFactory(factory);
                    ((AcceptsInvariant)target).addInvariant(inv);
                    return true;
                } else {
                    return false;
                }
            } catch (FactoryParameterException ex) {
                Logger.getLogger("org.grinvin.factories").log(Level.WARNING, "Failed to create invariant from factory", ex);
                return false;
            } catch (FactoryException ex) {
                Logger.getLogger("org.grinvin.factories").log(Level.WARNING, "Failed to create invariant from factory", ex);
                return false;
            }
        } else {
            return false;
        }
    }
    
}
