/* InvariantListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphbrowser;

import be.ugent.caagt.swirl.dnd.DragHandler;
import be.ugent.caagt.swirl.dnd.LocalTransferHandler;

import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.help.CSH;
import javax.swing.AbstractAction;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;

import org.grinvin.GraphURIType;
import org.grinvin.help.InvariantListHelpManager;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.list.DefaultInvariantList;
import org.grinvin.list.InvariantList;
import org.grinvin.list.InvariantListModel;


/**
 * List component with entries of type {@link Invariant}. Provides drag support
 * for invariants. Dropping is not allowed.
 */
public class InvariantListComponent extends JList implements HasInvariantList {
    
    // shared transfer handler
    protected static final LocalTransferHandler TRANSFER_HANDLER;
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle("org.grinvin.graphbrowser.resources");
    
    static {
        TRANSFER_HANDLER = new LocalTransferHandler();
        TRANSFER_HANDLER.addDropHandler(InvariantDropHandler.DROP_HANDLER);
        TRANSFER_HANDLER.addDropHandler(InvariantFactoryDropHandler.DROP_HANDLER);
        TRANSFER_HANDLER.setDragHandler(new InvariantListDragHandler());
    }
    
    /**
     * Creates a new instance of InvariantListComponent
     */
    public InvariantListComponent(InvariantListModel model) {
        super(model);
        setTransferHandler(TRANSFER_HANDLER);
        setDragEnabled(true);
        setCellRenderer(new ListCellRenderer());
        
        // Keyboard interaction
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements");
        getActionMap().put("deleteSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                deleteSelectedElements();
            }
        });
        
        CSH.addManager(new InvariantListHelpManager(this));
    }
    
    //
    public InvariantList getInvariantList() {
        return (InvariantList)super.getModel();
    }
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // gather all elements to be deleted
        InvariantList list = new DefaultInvariantList();
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                list.add(getInvariantList().get(i));
        
        // and remove them
        getInvariantList().removeAll(list);
        //TODO: make sure changed messages are sent
    }
    
    // drag handler for lists of type InvariantListComponent
    private static class InvariantListDragHandler implements DragHandler {
        
        public InvariantListDragHandler() {
        }
        
        //
        public int getSourceActions(JComponent source) {
            return LocalTransferHandler.COPY;
        }
        
        //
        public void exportDone(JComponent source, JComponent target, Object[] objects, Class<?> type, int action) {
            // MOVE not supported
            return;
        }
        
        // copy to another type array
        public  Object[] getExportedObjects(JComponent source) {
            return ((InvariantListComponent)source).getSelectedValues();
        }
        
        //
        private static final Class INVARIANT = Invariant.class;
        
        public Class getExportedClass(JComponent source) {
            return INVARIANT;
        }
        
        //
        public boolean isTransferAllowed(JComponent source, JComponent target) {
            return true;
        }
        
    }
    
    //
    private static class ListCellRenderer extends DefaultListCellRenderer {
        
        public Component getListCellRendererComponent
                (JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            if (value instanceof Invariant) {
                Invariant invariant = (Invariant)value;
                setText(invariant.getName());
                setIcon(GraphURIType.INVARIANT.getSmallIcon());
                if (InvariantManager.getInstance().getInvariantComputerFor(invariant) == null)
                    setForeground(Color.RED);
            } else
                throw new IllegalArgumentException("GraphListElementRenderer can only be used with lists of Invariant");
            return this;
        }
    }
    
}
