/* TransformedEmbedding.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred;
import org.grinvin.EmbeddingListener;
import org.grinvin.EmbeddingModel;
import org.grinvin.EmbeddingModelSupport;
import org.grinvin.EmbeddingView;
import org.grinvin.GraphView;
import org.grinvin.Vertex;

/**
 * Applies a {@link Transformation} to an existing embedding.
 */
public class TransformedEmbedding implements EmbeddingModel {
    
    //
    private EmbeddingModelSupport support;
    
    //
    private Transformation trans;
    
    //
    private EmbeddingView delegate;
       
    //
    private Listener listener;
    
    /**
     * Create an embedding of this type.
     */
    public TransformedEmbedding (EmbeddingView delegate, Transformation trans) {
        if (delegate.getDimension () != trans.getDimension ()) 
            throw new IllegalArgumentException ("Incompatible dimensions of view and transformation");
        this.trans = trans;
        this.delegate = delegate;
        this.support = new EmbeddingModelSupport ();
        this.listener = new Listener ();
        trans.addTransformationListener (listener);
        if (delegate instanceof EmbeddingModel) {
            ((EmbeddingModel)delegate).addEmbeddingListener (listener);
        }
    }
    
    /**
     * Decouple listener from delegate. Should be called before this 
     * embedding is removed so it can be garbage collected.
     */
    public void close () {
        if (trans != null)
            trans.removeTransformationListener (listener);
        if (delegate instanceof EmbeddingModel)
            ((EmbeddingModel)delegate).removeEmbeddingListener (listener);
    }
    
    // implements EmbeddingModel
    public void removeEmbeddingListener (EmbeddingListener l) {
        support.removeEmbeddingListener (l);
    }
    
    // implements EmbeddingModel
    public void addEmbeddingListener (EmbeddingListener l) {
        support.addEmbeddingListener (l);
    }

    
    // implements EmbeddingView
    public double[] getCoordinates (Vertex v) {
        double[] coord = delegate.getCoordinates (v);
        return trans.transform (coord, new double[2]);
    }

    // implements EmbeddingView
    public int getDimension () {
        return 2;
    }

    // implements EmbeddingView
    public GraphView getPeer () {
        return delegate.getPeer ();
    }
    
    /**
     * Internal class that listens to changes in the underlying embedding
     * and propagates them to listeners of this class.
     */
    private class Listener implements EmbeddingListener, TransformationListener {
        
        // implements EmbeddingListener
        public void vertexCoordinatesChanged (Vertex vertex) {
            support.fireCoordinatesChanged (vertex);
        }

        // implements EmbeddingListener
        public void dimensionChanged (int oldDimension) {
             throw new IllegalStateException ("Wrapped embedding cannot change dimension");
        }

        // implements EmbeddingListener
        public void embeddingChanged () {
            support.fireEmbeddingChanged ();
        }

        // implements TransformationListener
        public void transformationChanged() {
            if (delegate.getDimension () != trans.getDimension ()) 
                throw new IllegalStateException ("Incompatible dimensions of view and transformation");
            support.fireEmbeddingChanged ();
        }
        
    }
    
}
