/* AddEdges.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.mouse;

import be.ugent.caagt.swirl.mouse.NullMouseHandler;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Stroke;
import java.awt.event.MouseEvent;
import java.awt.geom.Line2D;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;

import org.grinvin.Edge;
import org.grinvin.Element;
import org.grinvin.Graph;
import org.grinvin.Vertex;
import org.grinvin.gred.GraphPanel;
import org.grinvin.gred.MutableGraphPanel;

/**
 * Mouse handler for adding edges. Consumes
 * all first button click events which were initiated on a vertex.
 */
public class AddEdges extends NullMouseHandler{
    
    // create a singleton
    private AddEdges() { }
    
    //
    private static AddEdges SINGLETON = new AddEdges();
    
    /**
     * Return the singleton object of this type.
     */
    public static AddEdges getHandler() {
        return SINGLETON;
    }
    
    /**
     * Register the vertex clicked as the current 'anchor' vertex.
     */
    @Override public void doMousePressed(JComponent parent, MouseEvent mouseEvent) {
        if (SwingUtilities.isLeftMouseButton(mouseEvent)) {
            GraphPanel panel = (GraphPanel)parent;
            Element element = panel.getContext().getRollOver();
            if (element instanceof Vertex) {
                panel.setAnchorElement(element);
                mouseEvent.consume();
            }
        }
    }
    
    /**
     * Do nothing when clicked, but consume event.
     */
    @Override public void doMouseClicked(JComponent parent, MouseEvent mouseEvent) {
        if (SwingUtilities.isLeftMouseButton(mouseEvent)) {
            GraphPanel panel = (GraphPanel)parent;
            Element element = panel.getContext().getRollOver();
            if (element instanceof Vertex) {
                mouseEvent.consume();
            }
        }
    }
    
    /**
     * When released on a vertex, join this vertex to the anchor vertex, unless
     * such an edge already exists. If the panel is of type 
     * {@link MutableGraphPanel} then releasing the mouse on an edge will 
     * break that edge, and releasing it on the background will create a new vertex.
     */
    @Override public void doMouseReleased (JComponent parent, MouseEvent mouseEvent, MouseEvent pressedEvent) {
        parent.repaint();
        GraphPanel panel = (GraphPanel)parent;
        Element element = panel.getContext().getRollOver ();
        if (element instanceof Vertex)
            panel.joinAnchorToVertex ((Vertex)element, true);
        else if (panel instanceof MutableGraphPanel) {
            MutableGraphPanel mpanel = (MutableGraphPanel)panel;
            if (element == null) {
                Vertex v = mpanel.addNewVertex (mouseEvent.getX(), mouseEvent.getY(), null, true);
                mpanel.joinAnchorToVertex (v, false);
            } else if (element instanceof Edge) {
                mpanel.splitEdge ((Edge)element, true, 
                    mouseEvent.getX(), mouseEvent.getY(), null, true);
            }
        }
    }
    
    // color of the 'ghost edge'
    private final Paint edgeColor = new Color(0, 127, 255);
    
    // stroke used to draw the 'ghost edge'
    private final Stroke stroke = new BasicStroke(1.0f);
    
    /**
     * Repaint the 'ghost' line.
     */
    @Override public void doMouseDragged(JComponent parent, MouseEvent mouseEvent, MouseEvent previousEvent, MouseEvent pressedEvent) {
        parent.repaint(); // TODO: more specific?
    }
    
    // temporary object, shared for reasons of speed
    private int[] coord = new int[2];
    
    /**
     * Paints the rubber band line.
     */
    @Override public void paintDragging(JComponent parent, Graphics g, MouseEvent mouseEvent, MouseEvent previousEvent, MouseEvent pressedEvent) {
        Graphics2D g2 = (Graphics2D)g;
        g2.setPaint(edgeColor);
        g2.setStroke(stroke);
        GraphPanel panel = (GraphPanel)parent;
        panel.mouseCoordinates((Vertex)panel.getAnchorElement(), coord);
        g2.draw(new Line2D.Double(coord[0],  coord[1], mouseEvent.getX(), mouseEvent.getY()));
    }
    
    
    /**
     * {@inheritDoc} This mouse handler needs a {@link GraphPanel}
     * as parent which is backed by a (mutable) {@link Graph}.
     */
    @Override public boolean canHandle(JComponent parent) {
        return parent instanceof GraphPanel &&
               ((GraphPanel)parent).getGraph() instanceof Graph;
    }
}
