/* Grinvin.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.grinvin;

import java.io.File;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import org.grinvin.grinvin.commands.LoadWorkspaceCommand;

import org.grinvin.grinvin.commands.NewWorkspaceCommand;
import org.grinvin.io.GrinvinURLStreamHandlerFactory;
import org.grinvin.logging.Logging;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.OsType;
import org.grinvin.preferences.GrinvinPreferences.Preference;
import org.grinvin.workspace.GrinvinWindowWrapper;
import org.grinvin.workspace.Workspace;
import org.grinvin.workspace.WorkspaceException;

/**
 *
 * @author adpeeter
 */
public class Grinvin {
    
    //
    private static final ResourceBundle BUNDLE
            = ResourceBundle.getBundle("org.grinvin.grinvin.resources");
    
    //
    private static final Logger LOGGER = Logger.getLogger
            ("org.grinvin.io", "org.grinvin.grinvin.resources");
    
    /**
     * Set the look and feel. Chooses the system look and feel unless the swing.defaultlaf
     * is set.
     */
    private static void setLookAndFeel() {
        String defaultlaf = System.getProperty("swing.defaultlaf");
        if (defaultlaf == null) {
            
            // use the system look and feel on windows
            if (GrinvinPreferences.INSTANCE.getOsType() == OsType.WINDOWS) {
                try {
                    UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
                } catch (ClassNotFoundException ex) {
                    //ignore
                } catch (InstantiationException ex) {
                    //ignore
                } catch (IllegalAccessException ex) {
                    //ignore
                } catch (UnsupportedLookAndFeelException ex) {
                    //ignore
                }
            }
        }
        /* [kc] Does not work reliably
        else if (defaultlaf.contains(".Substance")) {
            JFrame.setDefaultLookAndFeelDecorated(true);
            JDialog.setDefaultLookAndFeelDecorated(true);
        }
         */
    }
    
    //
    public static void main(String[] args) {
        
        setLookAndFeel();
        
        // install logging handler
        Logging.installHandler(BUNDLE.getString("logWindow.title"));
        
        // install url stream handler
        URL.setURLStreamHandlerFactory(new GrinvinURLStreamHandlerFactory());
        
        // install about handler
        if (GrinvinPreferences.INSTANCE.getOsType() == OsType.MAC_OS_X)
            new MacOSXHandler();
        
        // don't hide tooltips
        ToolTipManager.sharedInstance().setDismissDelay(Integer.MAX_VALUE);
        
        //internationalize
        internationalize();
        
        // install subsystems
        Workspace.getInstance().registerSubsystem(new GrinvinWindowWrapper());
        
        // obtain name of workspace directory
        String directoryName;
        if (args.length > 0) {
            // load workspace from disk if specified on the command line
            directoryName = args[0];
        } else {
            // otherwise load from disk from the location in the preferences
            directoryName = GrinvinPreferences.INSTANCE.getStringPreference(Preference.WORKSPACE_LOCATION);
            if (directoryName.startsWith("file:")) {
                directoryName = directoryName.substring(5); // old workspaces use URLs
                GrinvinPreferences.INSTANCE.setStringPreference(Preference.WORKSPACE_LOCATION, directoryName);
            }
        }
        
        // try to load prior workspace
        if (directoryName != null && directoryName.length() > 0) {
            File directory = new File(directoryName);
            if (directory.exists()) {
                try {
                    Workspace.getInstance().load(directory);
                    return;
                } catch (WorkspaceException ex) {
                    LOGGER.log(Level.WARNING, "Failed to load workspace (error while loading) from " + directory, ex);
                    int result = JOptionPane.showConfirmDialog(null,
                            MessageFormat.format(BUNDLE.getString("workspace.loadfailed.message"), directory),
                            BUNDLE.getString("workspace.loadfailed.title"),
                            JOptionPane.OK_CANCEL_OPTION, JOptionPane.ERROR_MESSAGE);
                    if (result != JOptionPane.OK_OPTION) {
                        System.exit(0); //TODO: define exit codes
                    }
                }
            }
        }
        
        // will only be reached when no prior workspace exists
        String[] options = {BUNDLE.getString("noworkspace.option.exit"),
                            BUNDLE.getString("noworkspace.option.load"),
                            BUNDLE.getString("noworkspace.option.new")};
        boolean workspaceLoaded = false;
        while(!workspaceLoaded){
            int choice = JOptionPane.showOptionDialog(null,
                    BUNDLE.getString("noworkspace.message"),
                    BUNDLE.getString("noworkspace.title"),
                    JOptionPane.YES_NO_OPTION,
                    JOptionPane.INFORMATION_MESSAGE,
                    null,
                    options,
                    options[2]);
            if(choice==2){ //new workspace
                try {
                    File directory = NewWorkspaceCommand.selectNewWorkspaceLocation();
                    if (directory != null) {
                        Workspace.getInstance().newWorkspace(directory);
                        workspaceLoaded = true;
                    }
                } catch (WorkspaceException ex) {
                    workspaceLoaded = false;
                }
            } else if (choice==1){ //load workspace
                try {
                    File directory = LoadWorkspaceCommand.loadNewWorkspace();
                    if (directory != null && directory.exists()) {
                        Workspace.getInstance().load(directory);
                        workspaceLoaded = true;
                    }
                } catch (WorkspaceException ex) {
                    workspaceLoaded = false; 
                }
            } else { //exit
                System.exit(0); //TODO: define exit codes
            }
        }
    }
    
    private static void internationalize(){
        //FileChooser
        UIManager.put("FileChooser.lookInLabelText", BUNDLE.getString("FileChooser.lookInLabelText"));
        UIManager.put("FileChooser.saveInLabelText", BUNDLE.getString("FileChooser.saveInLabelText"));
        UIManager.put("FileChooser.fileNameLabelText", BUNDLE.getString("FileChooser.fileNameLabelText"));
        UIManager.put("FileChooser.filesOfTypeLabelText", BUNDLE.getString("FileChooser.filesOfTypeLabelText"));
        UIManager.put("FileChooser.upFolderToolTipText", BUNDLE.getString("FileChooser.upFolderToolTipText"));
        UIManager.put("FileChooser.upFolderAccessibleName", BUNDLE.getString("FileChooser.upFolderAccessibleName"));
        UIManager.put("FileChooser.homeFolderToolTipText", BUNDLE.getString("FileChooser.homeFolderToolTipText"));
        UIManager.put("FileChooser.homeFolderAccessibleName", BUNDLE.getString("FileChooser.homeFolderAccessibleName"));
        UIManager.put("FileChooser.newFolderToolTipText", BUNDLE.getString("FileChooser.newFolderToolTipText"));
        UIManager.put("FileChooser.newFolderAccessibleName", BUNDLE.getString("FileChooser.newFolderAccessibleName"));
        UIManager.put("FileChooser.newFolderActionLabelText", BUNDLE.getString("FileChooser.newFolderActionLabelText"));
        UIManager.put("FileChooser.newFolderButtonText", BUNDLE.getString("FileChooser.newFolderButtonText"));
        UIManager.put("FileChooser.listViewButtonToolTipText", BUNDLE.getString("FileChooser.listViewButtonToolTipText"));
        UIManager.put("FileChooser.listViewButtonAccessibleName", BUNDLE.getString("FileChooser.listViewButtonAccessibleName"));
        UIManager.put("FileChooser.listViewActionLabelText", BUNDLE.getString("FileChooser.listViewActionLabelText"));
        UIManager.put("FileChooser.detailsViewButtonToolTipText", BUNDLE.getString("FileChooser.detailsViewButtonToolTipText"));
        UIManager.put("FileChooser.detailsViewButtonAccessibleName", BUNDLE.getString("FileChooser.detailsViewButtonAccessibleName"));
        UIManager.put("FileChooser.detailsViewActionLabelText", BUNDLE.getString("FileChooser.detailsViewActionLabelText"));
        UIManager.put("FileChooser.refreshActionLabelText", BUNDLE.getString("FileChooser.refreshActionLabelText"));
        UIManager.put("FileChooser.viewMenuLabelText", BUNDLE.getString("FileChooser.viewMenuLabelText"));
        UIManager.put("FileChooser.fileNameHeaderText", BUNDLE.getString("FileChooser.fileNameHeaderText"));
        UIManager.put("FileChooser.fileSizeHeaderText", BUNDLE.getString("FileChooser.fileSizeHeaderText"));
        UIManager.put("FileChooser.fileTypeHeaderText", BUNDLE.getString("FileChooser.fileTypeHeaderText"));
        UIManager.put("FileChooser.fileDateHeaderText", BUNDLE.getString("FileChooser.fileDateHeaderText"));
        UIManager.put("FileChooser.fileAttrHeaderText", BUNDLE.getString("FileChooser.fileAttrHeaderText"));
        UIManager.put("FileChooser.cancelButtonText", BUNDLE.getString("FileChooser.cancelButtonText"));
        UIManager.put("FileChooser.openButtonText", BUNDLE.getString("FileChooser.openButtonText"));
        UIManager.put("FileChooser.saveButtonText", BUNDLE.getString("FileChooser.saveButtonText"));
        UIManager.put("FileChooser.newFolderErrorText", BUNDLE.getString("FileChooser.newFolderErrorText"));
        UIManager.put("FileChooser.newFolderExistsErrorText", BUNDLE.getString("FileChooser.newFolderExistsErrorText"));
        UIManager.put("FileChooser.newFolderPromptText", BUNDLE.getString("FileChooser.newFolderPromptText"));
        UIManager.put("FileChooser.untitledFolderName", BUNDLE.getString("FileChooser.untitledFolderName"));
        UIManager.put("FileChooser.newFolderTitleText", BUNDLE.getString("FileChooser.newFolderTitleText"));
        UIManager.put("FileChooser.saveButtonToolTipText", BUNDLE.getString("FileChooser.saveButtonToolTipText"));
        UIManager.put("FileChooser.openButtonToolTipText", BUNDLE.getString("FileChooser.openButtonToolTipText"));
        UIManager.put("FileChooser.cancelButtonToolTipText", BUNDLE.getString("FileChooser.cancelButtonToolTipText"));
        UIManager.put("FileChooser.chooseButtonText", BUNDLE.getString("FileChooser.chooseButtonText"));
    }
}
