/* DropList.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui;

import java.awt.Color;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.dnd.DropTargetAdapter;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.TooManyListenersException;
import javax.swing.BorderFactory;
import javax.swing.JList;
import javax.swing.JViewport;
import javax.swing.ListModel;
import javax.swing.Timer;


/**
 * Extension to {@link JList} which enables dropping at
 * positions between list elements and indicates the drop
 * position by means of a horizontal cursor.
 */
public class DropList extends JList {
    
    /**
     * Class which is responsible for auto scrolling.
     */
    private class AutoScroller extends Timer implements ActionListener{

        // NOTE: Java class Autoscroll is buggy, so we cannot use it here.
        // cf. Bug report #4407536
        
        // -1 = up  +1 = down
        private int direction;
        
        //
        public void setDirection (int direction) {
            this.direction = direction;
            restart ();
        }
        
        //
        public AutoScroller () {
            super (100, null);
            addActionListener (this);
        }

        //
        public void actionPerformed (ActionEvent e) {
            setCursorIndex (getCursorIndex() + direction);
        }
           
    }
    
    //
    private static final int AS_MARGIN = 10;
    
    //
    private void activateAutoScroll (Point location) {
        Container parent = getParent ();
        if (parent instanceof JViewport) {
            JViewport viewport = (JViewport)parent;
            Rectangle r = viewport.getViewRect ();
            if (location.y < r.y + AS_MARGIN)
                autoScroller.setDirection (-1);
            else if (location.y > r.y + r.height - AS_MARGIN)
                autoScroller.setDirection (+1);
        }
    }
    
    //
    private class MyDTL extends DropTargetAdapter {
        public void dragOver(DropTargetDragEvent dtde) {
            autoScroller.stop ();
            Point location = dtde.getLocation ();
            activateAutoScroll (location);
            setCursorIndex(location);
        }
        
        public void drop(DropTargetDropEvent dtde) {
            dtde.rejectDrop();
            setCursorIndex (-1);
            autoScroller.stop ();
        }

        public void dragEnter (DropTargetDragEvent dtde) {
            setCursorIndex(dtde.getLocation());
        }
        
        public void dragExit (java.awt.dnd.DropTargetEvent dte) {
            setCursorIndex(-1);
            autoScroller.stop ();
        }
    }        
    
    //
    private AutoScroller autoScroller;
    
    /**
     * Construct a list of this type for a given data model.
     */
    public DropList(ListModel dataModel) {
        super(dataModel);
        cursorIndex = -1;
        setBorder (BorderFactory.createEmptyBorder 
                (cursorMargin, 0, cursorMargin, 0));
        try {
            getDropTarget().addDropTargetListener(new MyDTL());
        } catch (TooManyListenersException ex) {
            // do nothing
        }
        autoScroller= new AutoScroller ();
    }
    
    
    /**
     * Extra vertical space to be left on top of the list and at the bottom
     * to make sure the cursor is fully visible when drawn at both extremes
     * of the list.
     */
    protected int cursorMargin = 3;
    
    /**
     * Draw a cursor at the given position.
     * @param g graphics context
     * @param x X-position of the cursor
     * @param y Y-position of the left side of the cursor
     * @param width Width of the cursor
     */
    protected void paintCursor(Graphics g, int x, int y, int width) {
        g.setColor(Color.BLACK);
        g.fillRect(x, y-1, x+width, 3);
    }
    
    //
    private int cursorIndex;
    
    /**
     * Return the position at which the cursor is drawn.
     */
    public int getCursorIndex() {
        return cursorIndex;
    }
    
    /**
     * Set the position at which the cursor should be drawn.
     * @param index Index of the list element in front of which the cursor should
     * be drawn. If out of range, the cursor will not be drawn.
     */
    public void setCursorIndex(int index) {
        if (index != cursorIndex) {
            this.cursorIndex = index;
            ensureCursorVisible ();
            repaint(); // TODO: more specific repaint
        }
    }
    
    /**
     * Set the cursor index according to the given location.
     */
    public void setCursorIndex(Point location) {
        int index = locationToIndex(location);
        if (index >= 0) {
            Rectangle bounds = getCellBounds(index, index);
            if (location.y - bounds.y > bounds.height/2)
                index ++;
            setCursorIndex(index);
        }
    }
    
    /**
     * Ensure cursor and the two elements
     * surrounding it, are visible.
     */
    private void ensureCursorVisible () {
        if (cursorIndex < 0)
            return;
        int maxIndex = getModel().getSize ();
        if (cursorIndex > maxIndex || maxIndex == 0)
            return;
        Rectangle r;
        if (cursorIndex == 0) {
            r = getCellBounds (0,0);
            r.y = 0;
        }
        else if (cursorIndex == maxIndex) {
            r = getCellBounds (cursorIndex-1,cursorIndex-1);
            r.height += cursorMargin;
        }
        else {
            r = getCellBounds (cursorIndex-1,cursorIndex);
        }
        scrollRectToVisible (r);
    }
    
    /**
     * Paint the cursor at the given position.
     */
    protected void paintCursor(Graphics g, int index) {
        if (index < 0 || index > getModel().getSize())
            return;
        if (index == 0) {
            Rectangle bounds = getCellBounds(0, 0);
            paintCursor(g, bounds.x, bounds.y, bounds.width);
            
        } else {
            Rectangle bounds = getCellBounds(index-1, index-1);
            paintCursor(g, bounds.x, bounds.y+bounds.height, bounds.width);
        }
    }
   
    
    // overrides JComponent
    @Override
    protected void paintComponent(Graphics g) {
        
        super.paintComponent(g);
        paintCursor(g, cursorIndex);
    }

 
    
   
}

