/* GraphViewer.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.help;

import be.ugent.caagt.swirl.dnd.DragHandler;
import be.ugent.caagt.swirl.dnd.LocalTransferHandler;

import java.awt.Dimension;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import javax.swing.JComponent;
import javax.swing.JLabel;

import org.grinvin.graphbrowser.GraphComponent;
import org.grinvin.list.GraphListElement;

/**
 * Component for displaying a graph in the {@link InlineGraphViewer}. The image can be dragged as a
 * {@link GraphListElement }.
 */
public class GraphViewer extends GraphComponent implements MouseMotionListener {
    
    // shared transfer handler
    private static final LocalTransferHandler TRANSFER_HANDLER;
    
    static {
        TRANSFER_HANDLER = new LocalTransferHandler();
        TRANSFER_HANDLER.setDragHandler(new GraphViewerDragHandler());
    }
    
    public GraphViewer(){
        setPreferredSize(new Dimension(200, 200));
        setMinimumSize(new Dimension(200, 200));
        setVerticalTextPosition(JLabel.BOTTOM);
        addMouseMotionListener(this);
        setTransferHandler(TRANSFER_HANDLER);
        setDragEnabled(true);
    }
    
    public GraphListElement[] getGraphListElement(){
        GraphListElement[] graph = new GraphListElement[1];
        graph[0] = getElement();
        return graph;
    }
    
    
    /*============================================================
     * DRAG SUPPORT
     *============================================================*/
    
    // implements MouseMotionListener
    public void mouseDragged(MouseEvent e) {
        if (dragEnabled) {
            TRANSFER_HANDLER.exportAsDrag(GraphViewer.this, e,
                    TRANSFER_HANDLER.COPY);
        }
    }
    
    // implements MouseMotionListener
    public void mouseMoved(MouseEvent e) {
        // do nothing
    }
    
    
    // drag handler for GraphViewer
    private static class GraphViewerDragHandler implements DragHandler {
        
        //
        private static final Class GRAPH_LIST_ELEMENT = GraphListElement.class;
        
        //
        public int getSourceActions(JComponent source) {
            return LocalTransferHandler.COPY;
        }
        
        //
        public void exportDone(JComponent source, JComponent target, Object[] objects, Class<?> type, int action) {
            // MOVE not supported
            return;
        }
        
        // copy to another type array
        public  Object[] getExportedObjects(JComponent source) {
            //TODO: do we have to copy? I think not
            Object[] values = ((GraphViewer)source).getGraphListElement();
            Object[] result = new Object[values.length];
            System.arraycopy(values, 0, result, 0, values.length);
            return result;
        }
        
        //
        public Class getExportedClass(JComponent source) {
            return GRAPH_LIST_ELEMENT;
        }
        
        //
        public boolean isTransferAllowed(JComponent source, JComponent target) {
            return true;
        }
    }
    
    //
    private boolean dragEnabled;
    
    /**
     * Indicate whether dragging on this component should be enabled.
     */
    public void setDragEnabled(boolean dragEnabled) {
        this.dragEnabled = dragEnabled;
    }
    
    /**
     * Is dragging on this component enabled?
     */
    public boolean isDragEnabled() {
        return this.dragEnabled;
    }
}
