/* InvariantComputerManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.factories.FactoryParameterException;
import org.grinvin.io.IOFormatException;
import org.grinvin.util.LocalClassLoader;
import org.grinvin.xml.XMLUtils;

import org.jdom.Element;

/**
 * Manages the registration of invariant computers for the InvariantManager
 */
class InvariantComputerManager {
    
    //
    private static final String LOGGER_NAME = "org.grinvin.invariants";
    
    //
    private static final Class<InvariantComputer> INVARIANT_COMPUTER_CLASS
            = InvariantComputer.class; // must come before SINGLETON!
    
    //
    private static final Class<InvariantComputerFactory> INVARIANT_COMPUTER_FACTORY_CLASS
            = InvariantComputerFactory.class; // must come before SINGLETON!
    
    // needed for call backs to getInvariant
    private final InvariantManager invariantManager;
    
    // needed for loading computer classes
    private final LocalClassLoader localClassLoader;
    
    
    /** Creates a new instance of InvariantComputerManager */
    public InvariantComputerManager(InvariantManager invariantManager,
            LocalClassLoader localClassLoader) throws
            IOFormatException {
        this.invariantManager = invariantManager;
        this.localClassLoader = localClassLoader;
        this.computers = new HashMap<Invariant, InvariantComputer>();
        this.computerFactories = new HashMap<String, InvariantComputerFactory>();
        
        // load the standard computers
        try {
            Element element = XMLUtils.loadFromClassPath("org/grinvin/invariants/computers/resources/computers.xml");
            if (element != null) {
                for (Object o : element.getChildren("package")) {
                    Element pkg = (Element)o;
                    loadComputers(pkg, pkg.getAttributeValue("name"));
                }
            }
        } catch (IOException ex) {
            Logger.getLogger(LOGGER_NAME).log(Level.WARNING, "Failed to load invariants list", ex);
        }
        
        // make the computers just loaded the default
        this.defaultComputers = computers;
        this.defaultFactories = computerFactories;
        
        this.computers = new HashMap<Invariant, InvariantComputer>();
        this.computerFactories = new HashMap<String, InvariantComputerFactory>();
    }
    
    /**
     * Maps an invariant identifier to the invariant computer currently used
     * to compute it.
     */
    private Map<Invariant, InvariantComputer> computers; // NOPMD
    
    // default set of computers always to be searched first
    private final Map<Invariant, InvariantComputer> defaultComputers;
    
    
    /**
     * Maps a generic id of a parametrized invariant to the invariant
     * computer currently used to compute it.
     */
    private Map<String, InvariantComputerFactory> computerFactories; // NOPMD
    
    // default set of computers always to be searched first
    private final Map<String, InvariantComputerFactory> defaultFactories;
    
    /**
     * Return the invariant computer which can be used to compute values for
     * the given invariant.
     */
    public InvariantComputer getInvariantComputerFor(Invariant invariant) {
        InvariantComputer result = defaultComputers.get(invariant);
        if (result == null)
            return computers.get(invariant);
        else
            return result;
    }
    
    /**
     * Return the computer factory that corresponds to the given invariant factory.
     */
    public InvariantComputerFactory getInvariantComputerFactoryFor(String id) {
        InvariantComputerFactory result = defaultFactories.get(id);
        if (result == null)
            return computerFactories.get(id);
        else
            return result;
    }
    
    /**
     * Return the InvariantComputer class for the given type.
     */
    private<T> Class<T> classForId(String id, Class<T> superType) throws IllegalInvariantComputerException {
        
        try {
            Class result = Class.forName(id, true, localClassLoader);
            if (superType.isAssignableFrom(result))
                return (Class<T>)result;
            else
                throw new IllegalInvariantComputerException("Computer (factory) not of type " + superType, id);
        } catch (ClassNotFoundException ex) {
            throw new IllegalInvariantComputerException("Unknown class", id, ex);
        }
    }
    
    /**
     * Register an instance of a given InvariantComputer class with the
     * manager.
     * @throws IllegalInvariantComputerException when the class does not
     * represent a legal invariant computer.
     */
    private void registerComputer(String className, String packageName) throws
            IllegalInvariantComputerException {
        // TODO: should register into defaultComputers
        registerComputer(
                classForId(packageName+"."+className, INVARIANT_COMPUTER_CLASS),
                "org.grinvin.invariants."+className);
    }
    
    //
    private void registerComputer(Class<InvariantComputer> clazz, String computerId) throws
            IllegalInvariantComputerException {
        try {
            InvariantComputer computer = clazz.newInstance();
            if (!computerId.equals(computer.getId()))
                throw new IllegalInvariantComputerException(
                        "Internal and external IDs differ", computerId);
            Invariant invariant =
                    invariantManager.getInvariant(computer.getInvariantId());
            computers.put(invariant, computer);
            // TODO: should check for duplicates
        } catch (InstantiationException ex) {
            throw new IllegalInvariantComputerException(
                    "Invalid invariant computer", computerId, ex);
        } catch (IllegalAccessException ex) {
            throw new IllegalInvariantComputerException(
                    "Invalid invariant computer", computerId, ex);
        } catch (UnknownInvariantException ex) {
            throw new IllegalInvariantComputerException(
                    "Computer for unknown invariant", computerId, ex);
        }
    }
    
    /**
     * Register an instance of a given InvariantComputerFactory class with the
     * manager.
     * @throws IllegalInvariantComputerException when the class does not
     * represent a legal invariant computer.
     */
    private void registerComputerFactory(String className, String packageName) throws
            IllegalInvariantComputerException {
        // TODO: should register into defaultFactories
        registerComputerFactory(
                classForId(packageName+"."+className, INVARIANT_COMPUTER_FACTORY_CLASS),
                "org.grinvin.invariants." + className);
    }
    
    //
    private void registerComputerFactory(Class<InvariantComputerFactory> clazz, String factoryId) throws
            IllegalInvariantComputerException {
        
        try {
            InvariantComputerFactory factory = clazz.newInstance();
            if (!factoryId.equals(factory.getId()))
                throw new IllegalInvariantComputerException(
                        "Internal and external IDs differ", factoryId);
            computerFactories.put(factory.getInvariantId(), factory);
            // TODO: should check for duplicates
        } catch (InstantiationException ex) {
            throw new IllegalInvariantComputerException(
                    "Invalid invariant computer factory", factoryId, ex);
        } catch (IllegalAccessException ex) {
            throw new IllegalInvariantComputerException(
                    "Invalid invariant computer factory", factoryId, ex);
        }
    }
    
    /**
     * Register a factory generated computer.
     * @param factory Factory with parameters set
     * @param invariant Corresponding factory generated invariant
     */
    public void registerComputerFromFactory(
            InvariantFactory factory,
            Invariant invariant) throws FactoryParameterException {
        InvariantComputer computer =
                getInvariantComputerFactoryFor(factory.getId()).
                createInvariantComputer(factory);
        computers.put(invariant, computer);
    }
    
    /**
     * Add a local invariant computer from file. If this is a computer
     * for a new invariant, information on this invariant is searched
     * for in the local class path. Can be used for simple invariants
     * and for invariant factories.
     * @throws IllegalInvariantComputerException when the file does not correspond
     * to a legal invariant computer Java class
     */
    public void loadInvariantComputer(File file) throws IllegalInvariantComputerException {
        try {
            Class clazz = localClassLoader.loadClassFromFile(file);
            String id = clazz.getName();
            if (INVARIANT_COMPUTER_CLASS.isAssignableFrom(clazz)) {
                registerComputer((Class<InvariantComputer>)clazz, id);
            } else if (INVARIANT_COMPUTER_FACTORY_CLASS.isAssignableFrom(clazz)) {
                registerComputerFactory((Class<InvariantComputerFactory>)clazz, id);
            } else {
                throw new IllegalInvariantComputerException(
                        "Not an invariant computer (factory)", id);
            }
        } catch (ClassNotFoundException ex) {
            throw new IllegalInvariantComputerException(
                    "Illegal class file for invariant computer (factory)", null, ex);
        } catch (LinkageError ex) {
            throw new IllegalInvariantComputerException(
                    "Illegal class file for invariant computer (factory)", null, ex);
        }
    }
    
    /**
     * Add information on (local) factories and computers
     * to the given JDOM-element for reasons of JDOM persistence.
     */
    public void saveComputers(Element parent) {
        for (String id: computerFactories.keySet())
            parent.addContent(new Element("factory").addContent(id));
        for (InvariantComputer computer: computers.values()) {
            String id = computer.getId();
            if (id.indexOf('?') < 0)
                parent.addContent(new Element("computer").addContent(id));
        }
    }
    
    /**
     * Load (local) factories and computers as listed in
     * the given JDOM element.
     */
    private void loadComputers(Element parent, String packageName) throws
            IOFormatException {
        for (Object obj : parent.getChildren()) {
            Element el = (Element)obj;
            String name = el.getName();
            try {
                if ("class".equals(name)) {
                    String computerId = el.getTextTrim();
                    registerComputer(computerId, packageName);
                } else if ("factory".equals(name)) {
                    String factoryId = el.getTextTrim();
                    registerComputerFactory(factoryId, packageName);
                } else {
                    throw new IOFormatException("Unexpected element <" + name + "> in XML file");
                }
            } catch (IllegalInvariantComputerException ex) {
                Logger.getLogger(LOGGER_NAME).warning(
                        "Skipping unknown invariant computer (factory) '" + el.getTextTrim() + "' in package '" + packageName +"'");
            }

        }
        
    }
    
}
