/* InvariantGroup.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import java.util.ArrayList;
import java.util.List;

/**
 * Groups a list of invariants, invariant factories or subgroups. 
 * The main purpose of an invariant group is to enable a tree-like display
 * of invariants and factories, in a GUI-component or as part of the
 * help system.
 */
public class InvariantGroup implements InvariantNode {
    
    //
    private final String name;

    public String getName() {
        return name;
    }
    
    //
    private final String insertionPoint;

    //
    public void accept(Visitor visitor) {
        visitor.visitGroup(this);
    }
    
    /**
     * Return the insertion point identifier for this
     * group, or null if the group does not serve as an insertion point.
     */
    public String getInsertionPoint() {
        return insertionPoint;
    }
    
    /** 
     * Construct an empty group with the given internationalized name (caption).
     * Initially this group has no children.
     * @param name Localized name for this group
     * @param insertionPoint Identifies this group as an insertion point. A value
     * of null indicates that the group does not serve as an insertion point.
     */
    public InvariantGroup(String name, String insertionPoint) {
        this.name = name;
        this.insertionPoint = insertionPoint;
        this.children = new ArrayList<InvariantNode> ();
    }
    
    //
    private final List<InvariantNode> children;

    public Iterable<InvariantNode> getChildren() {
        return this.children;
    }
    
    /**
     * Add a child to the group.
     */
    public void add (InvariantNode child) {
        children.add (child);
    }

}
