/* InvariantManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import java.io.File;
import java.io.IOException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.params.ParameterList;
import org.grinvin.util.LocalClassLoader;
import org.grinvin.xml.XMLUtils;

import org.jdom.Element;


/**
 * Keeps track of relations between invariants and invariant computers.
 * Handles the mapping between an invariant and its corresponding identifier, and between
 * invariants and their computer.<p>
 * This is a singleton class. The single shared object of this class can be retreived
 * using the method {@link #getInstance()}.
 */
public class InvariantManager {
    
    //
    private static final String LOGGER_NAME = "org.grinvin.invariants";
    
    //
    private static final InvariantManager SINGLETON = new InvariantManager();
    
    
    //
    private final InvariantComputerManager icm;
    
    /**
     * Maps an invariant identifier to the corresponding object of type invariant.
     */
    private final Map<String, Invariant> invariants;
    
    /**
     * Maps a generic identifier to the corresponding invariant factory.
     */
    private final Map<String, InvariantFactory> factories;
    
    // list of all ids of simple invariants and factories
    private final List<String> ids;
    
    //
    private final LocalClassLoader localClassLoader;
    
    /**
     * Local class loader which is used to retrieve invariants and
     * invariant computers.
     */
    public LocalClassLoader getLocalClassLoader() {
        return localClassLoader;
    }
    
    //
    private final Iterable<InvariantNode> standardInvariantNodes;
    
    /**
     * Return the 'standard' list of invariant nodes (mostly groups)
     * that come pre-installed with GrInvIn.
     */
    public Iterable<InvariantNode> getStandardInvariantNodes() {
        return standardInvariantNodes;
    }
    
    /**
     * Load an invariant and register it with this manager.
     * @param id Identifier for this invariant
     * @param path classpath relative path for the definition file of
     * this invariant.
     */
    public Invariant loadInvariant(String id, String path) throws UnknownInvariantException {
        Invariant inv = invariants.get(id);
        if (inv == null) {
            int pos = id.indexOf('?');
            String newId = id;
            if (pos >= 0) {
                Logger.getLogger(LOGGER_NAME).warning(
                        "Parametrized invariants cannot be loaded from file: " + id
                        );
                newId = id.substring(0, pos);
            }
            inv = new DefaultInvariant(newId, path, localClassLoader);
            invariants.put(newId, inv);
            ids.add(newId);
            fireNewInvariant(inv); // TODO: is this necessary?
        } else
            Logger.getLogger(LOGGER_NAME).warning(
                    "Definition of invariant '" + id + "' was already loaded"                    );
        return inv;
    }
    
    /**
     * Return the invariant with the given identifier.
     */
    public Invariant getInvariant(String id) throws UnknownInvariantException {
        Invariant inv = invariants.get(id);
        if (inv == null) {
            // split off parameter values
            int pos = id.indexOf('?');
            if (pos < 0) {
                // try to load invariant using identifier as path
                return loadInvariant(id, id.replace('.','/') + ".xml");
            } else {
                // create invariant from factory
                
                try {
                    String genericId = id.substring(0, pos);
                    String queryString = id.substring(pos+1);
                    InvariantFactory factory = getInvariantFactory(genericId);
                    ParameterList list = factory.getParameters();
                    factory.setParameterValues(list.parseQueryString(queryString));
                    inv = getInvariantForFactoryAux(factory);
                } catch (FactoryParameterException ex) {
                    throw new UnknownInvariantException(
                            "Invalid parameters for invariant", id, ex);
                } catch (FactoryException ex) {
                    throw new UnknownInvariantException(
                            "Invalid invariant", id, ex);
                }
            }
        }
        return inv;
    }
    
    /**
     * Load an invariant factory and register it with this manager.
     * @param id Identifier for this invariant factory
     * @param path classpath relative path for the definition file of
     * this invariant factory.
     */
    public InvariantFactory loadInvariantFactory(String id, String path) throws UnknownInvariantException{
        InvariantFactory factory = factories.get(id);
        if (factory == null) {
            factory = new DefaultInvariantFactory(id, path, localClassLoader);
            factories.put(id, factory);
            ids.add(id);
            fireNewInvariantFactory(factory);// TODO: is this necessary?
            
        } else
            Logger.getLogger(LOGGER_NAME).warning(
                    "Definition of invariant factory '" + id + "' was already loaded"
                    );
        return factory;
    }
    
    /**
     * Return the invariant factory with the given identifier.
     */
    public InvariantFactory getInvariantFactory(String id) throws UnknownInvariantException{
        InvariantFactory factory = factories.get(id);
        if (factory == null) {
            factory = loadInvariantFactory(id, id.replace('.','/') + ".xml");
        }
        return factory;
    }
    
    /**
     * Return the path of the definition file for the invariant or the factory
     * with the given ID. (Used by the help subsystem.)
     */
    public String getDefinitionPath(String id) {
        Invariant invariant = invariants.get(id);
        if (invariant != null)
            return invariant instanceof DefaultInvariant ? ((DefaultInvariant)invariant).getPath() : null;
        
        InvariantFactory factory = factories.get(id);
        if (factory != null)
            return factory instanceof DefaultInvariantFactory ? ((DefaultInvariantFactory)factory).getPath() : null;
        
        return null;
    }
    
    /**
     * Read a list of nodes from an XML-file. If the XML-file is not found
     * or does not parse correctly, a single group is returned which may serve
     * as a local insertion point.
     * @param path classpath relative path of the XML file
     * @param bundle Resource bundle used to resolve localizations for
     * group captions
     */
    private Iterable<InvariantNode> loadGroups(String path, ResourceBundle bundle) {
        try {
            Element element = XMLUtils.loadFromClassPath(path);
            if (element != null) {
                InvariantGroup dummy = new InvariantGroup("dummy", null);
                load(element, dummy, bundle);
                return dummy.getChildren();
            }
        } catch (IOException ex) {
            Logger.getLogger(LOGGER_NAME).log(
                    Level.WARNING, "Could not load invariants from file '"
                    + path + "'", ex);
        }
        // default if no invariants could be found
        List<InvariantNode> result = new ArrayList<InvariantNode> ();
        result.add(new InvariantGroup(bundle.getString("local"), "local"));
        return result;
    }
    
    /**
     * Add contents of element as children to the given group.
     */
    private void load(Element root, InvariantGroup parent, ResourceBundle bundle) {
        for (Object o : root.getChildren()) {
            Element element = (Element)o;
            String name = element.getName();
            if ("id".equals(name)) {
                String id = element.getTextTrim();
                try {
                    parent.add(getInvariant(id));
                } catch (UnknownInvariantException ex) {
                    Logger.getLogger(LOGGER_NAME).warning(
                            "Could not load unknown invariant '" + id +
                            "' from XML-file");
                }
            } else if ("factory".equals(name)) {
                String id = element.getTextTrim();
                try {
                    parent.add(getInvariantFactory(id));
                } catch (UnknownInvariantException ex) {
                    Logger.getLogger(LOGGER_NAME).warning(
                            "Could not load unknown  factory '" + id +
                            "' from XML-file");
                }
            } else  if ("group".equals(name)) {
                String caption = bundle.getString(element.getAttributeValue("i18n"));
                String insert = element.getAttributeValue("insert");
                InvariantGroup group = new InvariantGroup(caption, insert);
                load(element, group, bundle); // process children
                parent.add(group);
            } else if ("dir".equals(name)) {
                String dirName = element.getAttributeValue("path");
                for (Object obj : element.getChildren()) {
                    Element child = (Element)obj;
                    String childName = child.getName();
                    String childId = child.getTextTrim();
                    String path = dirName + "/" +
                            childId.substring(childId.lastIndexOf('.')+1) +
                            ".xml";
                    try {
                        if ("id".equals(childName)) {
                            Invariant invariant = loadInvariant(childId, path);
                            parent.add(invariant);
                        } else if ("factory".equals(childName)) {
                            InvariantFactory factory = loadInvariantFactory(childId, path);
                            parent.add(factory);
                        }
                    } catch (UnknownInvariantException ex) {
                        Logger.getLogger(LOGGER_NAME).warning(
                                "Skipping unknown invariant (factory) '" + childId +
                                "' with path '" + path +"'");
                    }
                }
            }
        }
    }
    
    /**
     * Return the invariant computer which can be used to compute values for
     * the given invariant or null when no computer exists
     * @param invariant the {@link Invariant} for which you want a computer
     * @return the {@link InvariantComputer} that can compute the given
     * {@link Invariant} or {@code null} if no computer is available.
     */
    public InvariantComputer getInvariantComputerFor(Invariant invariant) {
        return icm.getInvariantComputerFor(invariant);
    }
    
    /**
     * Return the computer factory that corresponds to the given invariant factory.
     */
    public InvariantComputerFactory getInvariantComputerFactoryFor(String id) {
        return icm.getInvariantComputerFactoryFor(id);
    }
    
    /** Creates a new instance of InvariantManager */
    private InvariantManager() {
        
        try {
            
            this.invariants = new HashMap<String, Invariant>();
            this.factories = new HashMap<String, InvariantFactory>();
            this.ids = new ArrayList<String> ();
            
            this.localClassLoader = new LocalClassLoader();
            
            Iterable<InvariantNode> invariantNodes;
            try {
                invariantNodes = loadGroups(
                        "org/grinvin/invariants/definitions/resources/invariants.xml",
                        ResourceBundle.getBundle("org.grinvin.invariants.definitions.resources.invariants")
                        );
            } catch (MissingResourceException ex) {
                Logger.getLogger(LOGGER_NAME).log(Level.WARNING, "Failed to load invariants", ex);
                invariantNodes = new ArrayList<InvariantNode>();
            }            
            this.standardInvariantNodes = invariantNodes;
            
            this.icm = new InvariantComputerManager(this, localClassLoader);
        
        } catch (Exception ex) {
            throw new RuntimeException(
                    "Could not initialize invariant manager", ex);
        }
    }
    
    /**
     * Get the singleton instance of this class
     * @return the single shared instance of this class
     */
    public static InvariantManager getInstance() {
        return SINGLETON;
    }
    
    public Map<String, InvariantFactory> getFactories() {
        return factories;
    }
    
    /**
     * Return a list of all ids for both factories and parameterless invariants.
     */
    public List<String> getIds() {
        return ids;
    }
    
    /**
     * Return the invariant which is computed by the given computer.
     * Equivalent to <code>getInvariant (computer.getInvariantId())</code>
     * but does not throw an exception.
     */
    public Invariant getInvariantForComputer(InvariantComputer computer) {
        String id = computer.getInvariantId();
        Invariant result = invariants.get(id);
        assert result != null : "Invariant computer with unknown invariant";
        return result;
    }
    
// same as getInvariantForFactory but does not check the cache first
    private Invariant getInvariantForFactoryAux(InvariantFactory factory) throws
            FactoryParameterException {
        Invariant result = factory.createInvariant();
        icm.registerComputerFromFactory(factory, result);
        invariants.put(result.getId(), result);
        return result;
    }
    
    /**
     * Return the invariant for the given factory.
     * Equivalent to <code>getInvariant (factory.getInvariantId())</code>
     * but does not throw an UnknownInvariantException.
     * @throws FactoryParameterException when the factory parameters
     * were not set in an appropriate manner before this method was called.
     */
    public Invariant getInvariantForFactory(InvariantFactory factory) throws
            FactoryParameterException, FactoryException {
        String id = factory.getInvariantId();
        Invariant result = invariants.get(id);
        if (result == null)
            result = getInvariantForFactoryAux(factory);
        return result;
    }
    
    
    /**
     * Add a local invariant computer from file. If this is a computer
     * for a new invariant, information on this invariant is searched
     * for in the local class path. Can be used for simple invariants
     * and for invariant factories.
     * @throws IllegalInvariantComputerException when the file does not correspond
     * to a legal invariant computer Java class
     */
    public void loadInvariantComputer(File file) throws IllegalInvariantComputerException {
        icm.loadInvariantComputer(file);
    }
    
//
    private final List<InvariantManagerListener> listeners
            = new ArrayList<InvariantManagerListener> ();
    
    /**
     * Add a listener which will be notified whenever a new invariant
     * or factory is introduced into the system. As with Swing listeners,
     * the listener most recently registered shall always be notified first.
     */
    public void addListener(InvariantManagerListener listener) {
        listeners.add(listener);
    }
    
//
    private void fireNewInvariant(Invariant invariant) {
        for (int i=listeners.size()-1; i >= 0; i--)
            listeners.get(i).newInvariant(invariant);
    }
    
//
    private void fireNewInvariantFactory(InvariantFactory factory) {
        for (int i=listeners.size()-1; i >= 0; i--)
            listeners.get(i).newInvariantFactory(factory);
    }
    
}

