/* AbstractInvariantCComputer.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers;

import org.grinvin.GraphBundleView;
import org.grinvin.GraphView;

/**
 * Abstract superclass for invariant computers that are written in C. Provides
 * a method to convert the graph to multicode.
 */
public abstract class AbstractInvariantCComputer extends AbstractInvariantComputer {
    
    /**
     * Converts the graph to multicode-representation. It can then be passed on to 
     * the C-function as an array of jint's (i.e. longs in C).
     */
    protected int[] graphToMulticode(GraphView graph) {
        int numberOfVertices = graph.getNumberOfVertices();
        int[] multicode = new int[graph.getNumberOfEdges() + numberOfVertices];
        int pos=0;
        multicode[pos++]=numberOfVertices;
        for(int i = 0; i < numberOfVertices - 1; i++){
            for(int j = i + 1; j < numberOfVertices; j++)
                if(graph.areAdjacent(graph.getVertex(i),graph.getVertex(j)))
                    multicode[pos++]=j;
            multicode[pos++]=0;
        }
        return multicode;
    }
    
    protected int[] graphToGRAPHAndADJAZENZ(GraphBundleView graphbundle, int maxVertices, int emptyCode) {
        int numberOfVertices = graphbundle.getGraph().getNumberOfVertices();
        int[][] adjlist = graphbundle.adjacencyList();
        int[] graphArray = new int[(maxVertices+1)*(maxVertices+1)];
        graphArray[0] = numberOfVertices;
        for(int i = 0; i < adjlist.length; i++){
            for(int j = 0; j < adjlist[i].length; j++)
                graphArray[(i+1)*maxVertices+j] = adjlist[i][j]+1; //+1 because vertices are numbered from 1 to n
            graphArray[(i+1)*maxVertices+adjlist[i].length] = emptyCode;
            graphArray[(maxVertices+1)*maxVertices + i + 1] = adjlist[i].length;
        }
        return graphArray;
    }
}
