/* EmbeddingLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import org.grinvin.Embedding;
import org.grinvin.GraphView;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * Loads an <tt>embedding.xml</tt>-section from an input stream.
 */
public class EmbeddingLoader {
    
    //
    private Embedding embedding;
    
    
    /** Creates a loader for embeddings.
     * @param embedding object which shall hold the resulting graph embedding.
     * The peer of this object should already be initialized. 
     */
    public EmbeddingLoader(Embedding embedding) {
        this.embedding = embedding;
    }
    
    /**
     * Construct the embedding for the current graph from the given JDOM element.
     */
    public void toEmbedding(Element element) throws IOFormatException {
        if (! "embedding".equals(element.getName()))
            throw new IOFormatException("Expected <embedding> element");
        int dimension = Integer.parseInt(element.getAttributeValue("dimension"));
        embedding.setDimension (dimension);
        GraphView graph = embedding.getPeer ();
        for (Object obj: element.getChildren("coordinates")) {
            Element c = (Element)obj;
            int id = Integer.parseInt(c.getAttributeValue("id"));
            double[] coordinates = new double[dimension];
            List dlist = c.getChildren("double");
            for (int i=0; i < dimension; i++) {
                Element d = (Element)dlist.get(i);
                coordinates[i] = Double.parseDouble(d.getAttributeValue("value"));
            }
            embedding.setCoordinates(graph.getVertex(id), coordinates);
        }        
    }
     
    /**
     * Load the embedding from the input stream.
     */
    public void load (InputStream input) throws IOException {
        try {
            Document document = new SAXBuilder().build(input);
            toEmbedding (document.getRootElement());
        } catch (JDOMException ex) {
            throw new IOFormatException ("XML parse error", ex);
        }
    }
    
}
