/* GraphBundleLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import org.grinvin.Embedding;
import org.grinvin.Graph;
import org.grinvin.GraphBundle;
import org.grinvin.gui.icons.DefaultGraphIconFactory;
import org.grinvin.gui.icons.GraphIconFactory;
import org.grinvin.util.InternationalizedProperties;

/**
 * Loads a {@link GraphBundle} from a zip file.
 */
public class GraphBundleLoader {
    
    //
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    //
    private GraphBundle bundle;
    
    //
    private SectionLoader sloader;
    
    // meta-information stored in the file
    private Properties meta;
    
    /**
     * Default constructor.
     */
    private GraphBundleLoader(GraphBundle bundle, SectionLoader sloader) {
        this.bundle = bundle;
        this.sloader = sloader;
    }
    
    /**
     * Get the entry with the given name.
     */
    private InputStream openEntry (String name) throws IOException {
        InputStream in = sloader.openSection(name);
        if (in == null)
            throw new IOFormatException("Expected section: " + name);
        return in;
    }
    
    /**
     * Load the meta-information for the graph.
     */
    private void loadMeta() throws IOException {
        meta = new Properties ();
        InputStream in = openEntry ("meta-info.xml");
        meta.loadFromXML(in);
        in.close ();
    }
    
    /**
     * Load the graph properties.
     */
    private void loadProperties() throws IOException {
        InputStream in = openEntry ("resources.xml");
        InternationalizedProperties props = new InternationalizedProperties();
        props.load(in);
        bundle.setProperties(props);
        in.close();
    }
      
    /**
     * Load the abstract graph.
     */
    private void loadGraph() throws IOException {
        InputStream in = openEntry("graph.xml");
        Graph graph = bundle.createGraph();
        GraphLoader loader = new GraphLoader(graph);
        loader.load(in);
        in.close();
    }
    
    /**
     * Add the embedding with the given index as the next embedding in the bundle.
     */
    private void addEmbedding(int index) throws IOException {
        InputStream in = openEntry("embedding_"+index+".xml");
        Embedding embedding = bundle.createEmbedding();
        EmbeddingLoader loader =
                new EmbeddingLoader(embedding);
        loader.load(in);
        in.close();
    }
    
    //
    private void loadInvariantValues() throws IOException {
        InputStream in;
        try {
            in = openEntry("invariantvalues.xml");
        } catch (IOFormatException ex) {
            // the given bundle does not contain invariant values
            //TODO: log failure somewhere
            return;
        }
        InvariantValuesLoader loader = new InvariantValuesLoader(bundle);
        loader.load(in);
        in.close();
    }
    
    /**
     * Load the bundle and close the inputstream.
     */
    private void load() throws IOException {
        loadProperties();
        loadGraph();
        loadMeta();
        String countStr = meta.getProperty("nrOfEmbeddings", "2");
        try {
            int count = Integer.parseInt(countStr);
            for (int i=0; i < count; i++)
                addEmbedding(i);
        } catch (NumberFormatException nfex) {
            throw new IOFormatException ("number of embeddings is not an integer", nfex);
        }
        initGraphIconFactory ();
        loadInvariantValues();
    }
    
    /**
     * Initialize the graph icon factory.
     */
    private void initGraphIconFactory() {
        String name = meta.getProperty("graphIconFactory");
        GraphIconFactory gif = DefaultGraphIconFactory.getInstance();
        if (name != null) {
            try {
                Class<?> clazz = Class.forName(name);
                Method method = clazz.getMethod("getInstance");
                gif = (GraphIconFactory)method.invoke(null); // static method
                bundle.setGraphIconFactory(gif);
            } catch (Exception ex) {
                LOGGER.log (Level.WARNING, "unable to instantiate graph icon factory", ex);
            }
        }
        bundle.setGraphIconFactory(gif);
    }
    
    /**
     * Load the bundle from the given file.
     * @param bundle Empty bundle which will hold the result of the load operation.
     * @param file File which contains the graph bundle in Zip-format
     */
    public static void load(GraphBundle bundle, File file)
        throws IOException {
        ZipFile zip = new ZipFile (file);
        GraphBundleLoader loader 
            = new GraphBundleLoader(bundle, new ZipFileSectionLoader (zip));
        loader.load();
        zip.close ();
    }
    
    /**
     * Load the bundle from the given input stream.
     * @param bundle Empty bundle which will hold the result of the load operation.
     * @param in Input stream which contains the graph bundle in Zip-format
     */
    public static void load(GraphBundle bundle, InputStream in)
        throws IOException {
        ZipInputStream zip = new ZipInputStream (in);
        GraphBundleLoader loader 
            = new GraphBundleLoader(bundle, new ZipInputStreamSectionLoader (zip));
        loader.load();
        in.close ();
    }
    
    /**
     * Load the bundle from the given directory. The directory should
     * contain the uncompressed contents of a ZIP-archive.<p>
     * <b>Note:</b> The preferred representation of graph bundles on disk
     * is by means of compressed ZIP-archives. This method is provided
     * primarily for debugging purposes.
     *
     * @param bundle Empty bundle which will hold the result of the load operation.
     * @param dir Directory which contains the graph bundle as
     * uncompressed contents of a ZIP-archive.
     */
    public static void loadFromDirectory (GraphBundle bundle, File dir)
        throws IOException {
        
        GraphBundleLoader loader 
            = new GraphBundleLoader(bundle, new DirectorySectionLoader (dir));
        loader.load();
    }
}
