/* GraphBundleSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.grinvin.EmbeddingView;
import org.grinvin.GraphBundleView;
import org.grinvin.GraphView;
import org.grinvin.gui.icons.DefaultGraphIconFactory;
import org.grinvin.gui.icons.GraphIconFactory;

/**
 * Saves a {@link GraphBundleView} onto an output stream.
 */
public class GraphBundleSaver {
    
    //
    private GraphBundleView bundle;
    
    //
    private ZipOutputStream zip;
    
    /**
     * Default constructor.
     */
    private GraphBundleSaver (GraphBundleView bundle, ZipOutputStream zip) {
        this.bundle = bundle;
        this.zip = zip;
    }
    
    //
    private static final String ZIPFILE_COMMENT =
            "Graph bundle -- Saved by org.grinvin.io.GraphBundleSaver";
    
    /**
     * Save the meta-information for the graph.
     */
    private void saveMeta() throws IOException {
        Properties meta = new Properties ();
        meta.setProperty("version", "1.0");
        meta.setProperty("nrOfEmbeddings", "" + bundle.getEmbeddingCount());
        GraphIconFactory gif = bundle.getGraphIconFactory();
        if (gif != null && ! (gif instanceof DefaultGraphIconFactory))
            meta.setProperty("graphIconFactory", gif.getClass().getName());
        ZipEntry entry = new ZipEntry ("meta-info.xml");
        zip.putNextEntry (entry);
        meta.storeToXML (zip, null);
        zip.closeEntry ();
    }
    
    /**
     * Save the graph properties.
     */
    private void saveProperties () throws IOException {
        ZipEntry entry = new ZipEntry ("resources.xml");
        zip.putNextEntry (entry);
        bundle.getProperties().save (zip);
        zip.closeEntry ();
    }
   
    /**
     * Save the abstract graph.
     */
    private void saveGraph () throws IOException {
        GraphView graph = bundle.getGraph ();
        GraphSaver saver = new GraphSaver (graph);
        ZipEntry entry = new ZipEntry ("graph.xml");        
        zip.putNextEntry (entry);
        saver.save (zip);
        zip.closeEntry ();
    }
    
    /**
     * Save the embedding with the given index.
     */
    private void saveEmbedding (int index) throws IOException {
        EmbeddingView embedding = bundle.getEmbedding (index);
        EmbeddingSaver saver = new EmbeddingSaver (embedding);
        String name = "embedding_" + index + ".xml";
        ZipEntry entry = new ZipEntry (name);        
        zip.putNextEntry (entry);
        saver.save (zip);
        zip.closeEntry ();
    }
    
    //
    private void saveInvariantValues() throws IOException {
        ZipEntry entry = new ZipEntry("invariantvalues.xml");
        zip.putNextEntry(entry);
        InvariantValuesSaver saver = new InvariantValuesSaver(bundle.getInvariantValues());
        saver.save(zip);
        zip.closeEntry();
    }
    
    /**
     * Save the bundle and close the outputstream.
     */
    private void save () throws IOException {
        zip.setComment (ZIPFILE_COMMENT);
        saveProperties();
        saveGraph ();
        saveMeta ();
        for (int i=0; i < bundle.getEmbeddingCount (); i++)
            saveEmbedding (i);
        saveInvariantValues();
        zip.finish ();
        zip.close ();
    }
    
    /**
     * Save the bundle onto the output stream.
     * @param bundle Bundle which should be saved
     * @param out Output stream to which the bundle should be written
     */
    public static void save (GraphBundleView bundle, OutputStream out)
        throws IOException {
        ZipOutputStream zip = new ZipOutputStream (out);
        GraphBundleSaver saver = new GraphBundleSaver (bundle, zip);
        saver.save ();
    }
    
}
