/* GraphListSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.GraphURI;
import org.grinvin.invariants.Invariant;
import org.grinvin.list.GraphInvariantList;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;
import org.grinvin.list.InvariantList;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * Helper methods for saving graph list elements, lists of graphs
 * and lists of invariants to disk.
 */
public final class GraphListSaver {
    
    //
    private static final Logger LOGGER = Logger.getLogger ("org.grinvin.io");
    
    // do not instantiate
    private GraphListSaver () {}
    
    /**
     * Save a list of invariants as an XML file. Creates a file with contents
     * of the form
     * <pre>
     *    &lt;invariants&gt;
     *        &lt;invariant invariantId="..."/&gt;
     *        ...
     *    &lt;/invariants&gt;
     * </pre>
     * @param list List(model) of invariants to be stored
     * @param file File into which these invariants should be stored
     */
    public static void save(InvariantList list, File file) throws IOException {
        Element element = new Element ("invariants");
        for (Invariant inv: list) 
            element.addContent(
                    new Element("invariant").setAttribute("invariantId", inv.getId())
                    );
        new XMLOutputter(Format.getPrettyFormat()).output (
                new Document(element), new FileOutputStream(file)
                );
        // TODO: make XMLOutputter a package wide singleton object?
    }
    
    /**
     * Save a list of graphs as an XML file. The XML file will only contain URL-references
     * to the corresponding graph bundles, and not the graph bundle contents. These should
     * be saved separately.<p>
     * If the parameter <tt>ilist</tt> is not null, also the invariants in that
     * list are stored into the XML file.
     * Creates a file with contents of the form
     * <pre>
     *    &lt;graphlistelements name="..."&gt;
     *        &lt;graphlistelement graphURI="..."/&gt;
     *        ...
     *        &lt;invariant invariantId="..."/&gt;
     *        ...
     *    &lt;/graphlistelements&gt;
     * </pre>
     * <b>Note:</b> Graphs without an URI ar note saved
     * @param list Graph list to be stored
     * @param ilist Optional invariant list to be stored into the same file (can be null)
     * @param file File into which this list should be stored
     */
    public static void save(GraphList list, InvariantList ilist, 
            File file) throws IOException {
        Element element = new Element("graphlistelements");
        String name = list.getName();
        if (name != null) // editor list has no name
            element.setAttribute("graphlistName", name);
        for(GraphListElement e : list) {
            URI uri = e.getURI();
            if (uri != null)
                element.addContent(
                    new Element("graphlistelement").setAttribute(
                        "graphURI", uri.toString())
                    );
                
        }
        if (ilist != null) {
            for (Invariant inv: ilist)
                element.addContent(
                    new Element("invariant").setAttribute("invariantId", inv.getId())
                    );
        }
      new XMLOutputter(Format.getPrettyFormat()).output (
                new Document(element), new FileOutputStream(file)
                );
    }
    
    /**
     * Convenience method which saves a graph list of type 
     * {@link GraphInvariantList}. Calls {@link #save(GraphList,InvariantList,File)}
     * with the appropriate arguments.
     */
    public static void save (GraphInvariantList list, File file) throws IOException {
        save (list.getGraphList(), list.getInvariantList(), file);
    }

    /**
     * Save a (session) graph list into a (workspace) directory. Uses the uri
     * of the graph list to determine the destination file name. If no URI was 
     * assigned, a new URI is created. Note that the graph list elements themselves
     * need to be saved separately.<p>
     * @throws IllegalArgumentException if the URI of the list is non null and
     * not a session URI
     * @see #saveIntoWorkspace(GraphListElement gle, File directory)
     */
    public static void saveIntoWorkspace (GraphList list, InvariantList ilist,
            File directory) throws IOException {
        URI uri = list.getURI ();
        if (! GraphURI.isSession(uri))
            throw new IllegalArgumentException ("Can only save session lists into the workspace");
        if (uri != null) {
            File file = new File(directory, uri.getSchemeSpecificPart());
            if (list.isDirty() || !file.exists() || (ilist != null && ilist.isDirty())) {
                file.delete();
                save(list, ilist, file);
            }
        } else {
            File file = File.createTempFile("GIO", ".graphlist", directory);
            try {
                list.setURI(GraphURI.createSession(file.getName()));
            } catch (URISyntaxException e) {
                assert false : "Unexpected URISyntaxException";
            }
            save (list, ilist, file);
        }
    }
    
    /**
     * Save a (session) graph list element into a (workspace) directory. Uses the uri
     * of the graph to determine the destination file name. If no URI was 
     * assigned, a new URI is created.<p>
     * Only saves graphs with a session URI and which are dirty or for which the
     * corresponding file does not yet exist. The 'dirty' flag of the graph list
     * element is cleared after it is saved.
     */
    public static void saveIntoWorkspace(GraphListElement gle,
            File directory) throws IOException {
        URI uri = gle.getURI();
        if (!GraphURI.isSession(uri)) {
            if (GraphURI.isGlobal(uri)) {
                cacheInvariants(gle);
            }
            return;
        }
        if (!gle.gotGraph())
            return;
        if (uri == null) {
            File file = File.createTempFile("GIO", ".gph", directory);
            GraphBundleSaver.save(gle.getBundle(), new FileOutputStream(file));
            try {
                gle.setURI(GraphURI.createSession(file.getName()));
            } catch (URISyntaxException e) {
                assert false : "Unexpected URISyntaxException";
            }
        } else {
            File file = new File(directory, uri.getSchemeSpecificPart());
            if (gle.isDirty() || ! file.exists())
                GraphBundleSaver.save(gle.getBundle(), new FileOutputStream(file));
        }
        gle.setDirty(false);
    }
    
    public static void cacheInvariants(GraphListElement gle) {
        URI uri = gle.getURI();
        InvariantValuesSaver saver = new InvariantValuesSaver(gle.getBundle().getInvariantValues());
        checkCacheDir();
        
        String cachedFilename;
        String raw = uri.getRawSchemeSpecificPart();
        int pos = raw.indexOf('?');
        if (pos >= 0)
            cachedFilename = raw.substring(0, pos) + "-" + raw.substring(pos+1).replace('&', '_').replace('=','_') + ".xml";
        else
            cachedFilename = raw + "-";
        
        File cachedFile = new File(GrinvinPreferences.INSTANCE.getStringPreference(Preference.GRINVIN_CACHE_DIR) + "/" + cachedFilename);
        try {
            saver.save(new FileOutputStream(cachedFile));
        } catch (FileNotFoundException ex) {
            LOGGER.log(Level.FINE, "Unable to load cached invariant values", ex);
            //TODO: handle, it is ok to just ignore this for now, the values will be recomputed
        } catch (IOException ex) {
            LOGGER.log(Level.FINE, "Unable to load cached invariant values", ex);
            //TODO: handle, it is ok to just ignore this for now, the values will be recomputed
        }
    }

    //
    private static void checkCacheDir() {
        File cacheDir = new File(GrinvinPreferences.INSTANCE.getStringPreference(Preference.GRINVIN_CACHE_DIR));
        if (cacheDir.exists()) {
            if (!cacheDir.isDirectory()) {
                throw new RuntimeException("Cache location is not a directory: " + cacheDir);
            }
        } else {
            if (!cacheDir.mkdir()) {
                LOGGER.log(Level.WARNING, "Could not create cache directory: " + cacheDir);
            }
        }
    }
    
}
