/* FactoryList.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list;

import be.ugent.caagt.swirl.dnd.DragHandler;
import be.ugent.caagt.swirl.dnd.LocalTransferHandler;
import be.ugent.caagt.swirl.lists.CellList;
import be.ugent.caagt.swirl.lists.DefaultCellListCellRenderer;
import be.ugent.caagt.swirl.lists.TypedListModel;

import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;
import javax.swing.JComponent;
import javax.swing.ListSelectionModel;

import org.grinvin.GraphURIType;
import org.grinvin.factories.graphs.GraphFactory;
import org.grinvin.gui.icons.SelectUnselectIcon;
import org.grinvin.gui.icons.SvgIconManager;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;

/**
 * Cell list component with entries of type {@link GraphFactory}. Provides drag support
 * for factories. Dropping is not allowed.
 */
public class FactoryList extends CellList {
    
    // shared transfer handler
    private static final LocalTransferHandler TRANSFER_HANDLER;
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle("org.grinvin.list.resources");
    
    static {
        TRANSFER_HANDLER = new LocalTransferHandler();
        TRANSFER_HANDLER.setDragHandler(new FactoryListDragHandler());
    }
    
    public FactoryList(TypedListModel<GraphFactory> model) {
        super(model, GrinvinPreferences.INSTANCE.getIntPreference(Preference.GRAPHICON_WIDTH), GrinvinPreferences.INSTANCE.getIntPreference(Preference.GRAPHICON_HEIGHT), 4);
        setTransferHandler(TRANSFER_HANDLER);
        setDragEnabled(true);
        getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        setToolTipText(BUNDLE.getString("Factories.name"));
        setCellRenderer(new FactoryListCellRenderer());
    }
    
    // drag handler for lists of type FactoryList
    private static class FactoryListDragHandler implements DragHandler {
        
        //
        private static final Class FACTORY = GraphFactory.class;
        
        //
        public int getSourceActions(JComponent source) {
            return LocalTransferHandler.COPY;
        }
        
        //
        public void exportDone(JComponent source, JComponent target, Object[] objects, Class<?> type, int action) {
            // MOVE not supported
            return;
        }
        
        // copy to another type array
        public  Object[] getExportedObjects(JComponent source) {
            //TODO: do we have to copy? I think not
            Object[] values = ((FactoryList)source).getSelectedValues();
            Object[] result = new Object[values.length];
            System.arraycopy(values, 0, result, 0, values.length);
            return result;
        }
        
        //
        public Class getExportedClass(JComponent source) {
            return FACTORY;
        }
        
        //
        public boolean isTransferAllowed(JComponent source, JComponent target) {
            return true;
        }
    }
    
    public class FactoryListCellRenderer extends DefaultCellListCellRenderer {
        
        //TODO: remove duplicate code with GraphListCellRenderer
        
        //
        private class BundleSizePair {
            
            //
            public GraphFactory factory;
            
            //
            public int size;
            
            //
            public BundleSizePair(GraphFactory factory, int size) {
                this.factory = factory;
                this.size = size;
            }
            
            //
            public boolean equals(Object obj) {
                if (obj instanceof BundleSizePair) {
                    BundleSizePair bsp = (BundleSizePair)obj;
                    return bsp.size == size && bsp.factory.equals(factory) ;
                } else
                    return false;
            }
            
            //
            public int hashCode() {
                return 137251 * size + factory.hashCode();
            }
        }
        
        /**
         * Map which maps bundle-size pairs to icons.
         */
        private final Map<BundleSizePair, SelectUnselectIcon> cache;
        
        /**
         * Create a graph list cell renderer that delegates to the given graph
         * renderer.
         */
        public FactoryListCellRenderer() {
            this.cache = new HashMap<BundleSizePair,SelectUnselectIcon>();
        }
        
        /**
         * Retrieve the graph icon for the given graph bundle and size
         */
        protected SelectUnselectIcon getIcon(GraphFactory factory, int cellWidth, int cellHeight) {
            int size = Math.min(cellWidth, cellHeight-15) - 10;
            BundleSizePair bsp = new BundleSizePair(factory, size);
            SelectUnselectIcon icon = cache.get(bsp);
            if (icon == null) {
                icon = SvgIconManager.getInstance().createSelectUnselectIcon(factory.getIconPath(), factory.getSelectedIconPath(), size);
                cache.put(bsp, icon);
            }
            return icon;
        }
        
        // overrides DefaultCellListCellRenderer
        @Override protected void prepare(CellList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            GraphFactory element = (GraphFactory)value;
            SelectUnselectIcon icon = getIcon(element, list.getCellWidth(), list.getCellHeight());
            icon.setSelected(isSelected);
            setIcon(icon);
            setText(element.getName());
            setIconBackground(GraphURIType.FACTORY.getIconBackgroundColor());
        }
        
        // overrides  DefaultCellListCellRenderer
        @Override public String getToolTipText(CellList list, Object value, int index) {
            GraphFactory element = (GraphFactory)value;
            return element.getDescription();
        }
        
    }
    
}
