/* GraphInvariantTableModel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list;

import java.util.ArrayList;
import java.util.List;
import javax.swing.event.ListDataEvent;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;
import org.grinvin.invariants.Invariant;

import org.grinvin.invariants.InvariantValue;

/**
 * {@link TableModel} to be used in {@link org.grinvin.graphbrowser.GraphTableComponent}. This class is
 * only used to represent the model for this component. A data structure that
 * combines both a {@link GraphListModel} and an {@link InvariantListModel} is
 * {@link GraphInvariantListModel}.
 */
public class GraphInvariantTableModel implements TableModel {
    
    //
    private GraphInvariantListModel graphInvariantList;
    
    //
    private List<TableModelListener> listeners;
    
    //
    public GraphInvariantTableModel(GraphInvariantListModel graphInvariantList) {
        this.graphInvariantList = graphInvariantList;
        this.listeners = new ArrayList<TableModelListener>();
        
        //TODO: maybe listen to GraphInvariantListModel instead
        graphInvariantList.getGraphListModel().addGraphListModelContentsListener(new GraphListModelDelegateListener(this));
        graphInvariantList.getInvariantListModel().addInvariantListModelListener(new InvariantListModelDelegateListener(this));
    }
    
    //
    public GraphInvariantListModel getGraphInvariantListModel() {
        return graphInvariantList;
    }
    
    private GraphList getGraphList() {
        return graphInvariantList.getGraphList();
    }
    
    private InvariantList getInvariantList() {
        return graphInvariantList.getInvariantList();
    }
    
    /***** implement TableModel *****/
    
    //
    public void addTableModelListener(TableModelListener listener) {
        listeners.add(listener);
    }
    
    //
    public Class getColumnClass(int col) {
        return InvariantValue.class;
    }
    
    //column 0 contains the name of the graph
    public int getColumnCount() {
        return getInvariantList().size();
    }
    
    //
    public String getColumnName(int column) {
        return getInvariantList().get(column).getName();
    }
    
    //
    public int getRowCount() {
        return getGraphList().size();
    }
    
    //
    public Object getValueAt(int row, int column) {
        return getGraphList().get(row).getInvariantNow(getInvariantList().get(column));
    }
    
    //
    public boolean isCellEditable(int row, int col) {
        return false;
    }
    
    //
    public void removeTableModelListener(TableModelListener listener) {
        listeners.remove(listener);
    }
    
    //
    public void setValueAt(Object value, int row, int col) {
    }
    
    /***** implement TableModel *****/
    
    //
    public void removeColumn(final int column) {
        if (getColumnCount() > column) {
            getInvariantList().remove(column);
            //TODO: keep changecount
            //changeCount++;
            fireColumnRemoved(column, column);
        }
    }
    
    //
    private void fireRowAdded(int index0, int index1) {
        for (TableModelListener listener : listeners) {
            listener.tableChanged(new TableModelEvent(this, index0, index1, TableModelEvent.ALL_COLUMNS, TableModelEvent.INSERT));
        }
    }
    
    //
    private void fireRowRemoved(int index0, int index1) {
        for (TableModelListener listener : listeners) {
            listener.tableChanged(new TableModelEvent(this, index0, index1, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
        }
    }
    
    //
    private void fireRowChanged(int index0, int index1) {
        for (TableModelListener listener : listeners) {
            listener.tableChanged(new TableModelEvent(this, index0, index1, TableModelEvent.ALL_COLUMNS, TableModelEvent.UPDATE));
        }
    }
    
    //
    private void fireColumnAdded(int index0, int index1) {
        for (TableModelListener listener : listeners) {
            listener.tableChanged(new TableModelEvent(this, TableModelEvent.HEADER_ROW));
        }
    }
    
    //
    private void fireColumnRemoved(int index0, int index1) {
        for (TableModelListener listener : listeners) {
            listener.tableChanged(new TableModelEvent(this, TableModelEvent.HEADER_ROW));
        }
    }
    
    //
    private void fireColumnChanged(int index0, int index1) {
        for (TableModelListener listener : listeners) {
            listener.tableChanged(new TableModelEvent(this, TableModelEvent.HEADER_ROW));
        }
    }

    //
    private void fireCellChanged(int row, int column) {
        for (TableModelListener listener : listeners) {
            listener.tableChanged(new TableModelEvent(this, row, row, column, TableModelEvent.UPDATE));
        }
    }
    
    //
    private class GraphListModelDelegateListener implements GraphListModelContentsListener {
        
        //
        private GraphInvariantTableModel tableModel;
        
        //
        public GraphListModelDelegateListener(GraphInvariantTableModel tableModel) {
            this.tableModel = tableModel;
        }
        
        public void intervalAdded(ListDataEvent e) {
            tableModel.fireRowAdded(e.getIndex0(), e.getIndex1());
        }
        
        public void intervalRemoved(ListDataEvent e) {
            tableModel.fireRowRemoved(e.getIndex0(), e.getIndex1());
        }
        
        public void contentsChanged(ListDataEvent e) {
            tableModel.fireRowChanged(e.getIndex0(), e.getIndex1());
        }
        
        public void graphListModelNameChanged(GraphListModel graphListModel) {
            //ignore, should be handled by the enclosing window
        }

        public void graphInvariantComputed(int position, Invariant invariant) {
            tableModel.fireCellChanged(position, getInvariantList().indexOf(invariant));
        }
        
    }
    
    //
    private class InvariantListModelDelegateListener implements InvariantListModelListener {
        
        //
        private GraphInvariantTableModel tableModel;
        
        //
        public InvariantListModelDelegateListener(GraphInvariantTableModel tableModel) {
            this.tableModel = tableModel;
        }
        
        public void intervalAdded(ListDataEvent e) {
            tableModel.fireColumnAdded(e.getIndex0(), e.getIndex1());
        }
        
        public void intervalRemoved(ListDataEvent e) {
            tableModel.fireColumnRemoved(e.getIndex0(), e.getIndex1());
        }
        
        public void contentsChanged(ListDataEvent e) {
            tableModel.fireColumnChanged(e.getIndex0(), e.getIndex1());
        }
        
    }
    
}
