/* GraphListCellRenderer.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list;

import be.ugent.caagt.swirl.lists.CellList;
import be.ugent.caagt.swirl.lists.DefaultCellListCellRenderer;

import java.util.HashMap;
import java.util.Map;

import org.grinvin.GraphBundleView;
import org.grinvin.GraphURI;
import org.grinvin.gui.icons.GraphIcon;

/**
 * Cell renderer for cell lists with elements
 * of type {@link GraphListElement}. Keeps a cache of the icons which it displays.
 *
 */
public class GraphListCellRenderer extends DefaultCellListCellRenderer {
    
    // TODO: should probably become a singleton object, now the cache
    // is repeated when different lists are used
    
    //
    private static class BundleSizePair {
        //
        public GraphBundleView bundle;
        
        //
        public int size;
        
        //
        public BundleSizePair(GraphBundleView bundle, int size) {
            this.bundle = bundle;
            this.size = size;
        }
        
        //
        public boolean equals(Object obj) {
            if (obj instanceof BundleSizePair) {
                BundleSizePair bsp = (BundleSizePair)obj;
                return bsp.size == size && bsp.bundle.equals(bundle) ;
            } else
                return false;
        }
        
        //
        public int hashCode() {
            return 137251 * size + bundle.hashCode();
        }
    }
    
    /**
     * Map which maps bundle-size pairs to icons.
     */
    private final Map<BundleSizePair, GraphIcon> cache;
    
    /**
     * Create a graph list cell renderer that delegates to the given graph
     * renderer.
     */
    public GraphListCellRenderer() {
        this.cache = new HashMap<BundleSizePair,GraphIcon> ();
    }
      
    
    /**
     * Retrieve the graph icon for the given graph bundle and size
     */
    protected GraphIcon getIcon(GraphBundleView bundle, int cellWidth, int cellHeight) {
        int size = Math.min(cellWidth, cellHeight-15) - 10;
        BundleSizePair bsp = new BundleSizePair(bundle, size);
        GraphIcon icon = cache.get(bsp);
        if (icon == null) {
            icon = bundle.getGraphIconFactory().createIcon(bundle, size);
            cache.put(bsp, icon);
        }
        return icon;
    }
    
    // overrides DefaultCellListCellRenderer
    @Override protected void prepare(CellList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
        GraphListElement element = (GraphListElement)value;
        GraphBundleView bundle = element.getBundle();
        GraphIcon icon = getIcon(bundle, list.getCellWidth(), list.getCellHeight());
        icon.setSelected(isSelected);
        setIcon(icon);
        String caption = bundle.getName();
        setText(caption);
        setIconBackground(GraphURI.getType(element.getURI()).getIconBackgroundColor());
    }
    
    // overrides  DefaultCellListCellRenderer
    @Override public String getToolTipText(CellList list, Object value, int index) {
        GraphListElement element = (GraphListElement)value;
        GraphBundleView bundle = element.getBundle();
        return bundle.getDescription();
    }
    
}
