/* GraphPropertiesTableModel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.ResourceBundle;
import java.util.SortedSet;
import java.util.TreeSet;
import javax.swing.table.AbstractTableModel;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantValue;

/**
 * Table model for the table of properties of a graph.
 */
public class GraphPropertiesTableModel extends AbstractTableModel implements GraphListElementListener {
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle("org.grinvin.list.resources");
    
    //
    private GraphListElement element;
    
    //
    private List<Invariant> invariants;
    
    //
    private List<GraphPropertiesTableModelListener> listeners;
    
    //
    private class InvariantComparator implements Comparator<Invariant> {
        public int compare(Invariant inv1, Invariant inv2) {
            return inv1.getName().compareTo(inv2.getName());
        }
    }
    
    /**
     * Creates the list of invariants from the current
     * graph list element.
     */
    private void initInvariants() {
        invariants.clear();
        if (element != null) {
            SortedSet<Invariant> set = new TreeSet<Invariant>(new InvariantComparator());
            set.addAll(element.getBundle().getInvariants());
            invariants.addAll(set);
        }
        fireTableDataChanged();
    }
    
    /**
     * Create a model for the given graph list element and the given
     * list of invariants.
     */
    public GraphPropertiesTableModel(GraphListElement element) {
        invariants = new ArrayList<Invariant>();
        listeners = new ArrayList<GraphPropertiesTableModelListener>();
        setElement(element);
    }
    
    /**
     * Use a different graph list element.
     */
    public void setElement(GraphListElement newElement) {
        if (element == newElement)
            return;
        
        if (element != null)
            element.removeGraphListElementListener(this);
        
        this.element = newElement;
        if (element != null) {
            element.addGraphListElementListener(this);
        }
        fireGraphListElementChanged();
        initInvariants();
    }
    
    /**
     * Has a fixed number of columns.
     */
    public int getColumnCount() {
        return 2;
    }
    
    //
    public Class getColumnClass(int col) {
        if (col == 0) {
            return Invariant.class;
        } else {
            return InvariantValue.class;
        }
    }

    /**
     * The value in the first column is the invariant,
     * the value in the second column is its value.
     */
    public Object getValueAt(int row, int column) {
        if(row >= invariants.size())
            return null;
        Invariant invar = invariants.get(row);
        if (column == 0)
            return invar;
        else if (element != null)
            return element.getInvariantNow(invar);
        else
            return null;
    }
    
    /**
     * Column names are null.
     */
    public String getColumnName(int column) {
        if (column == 0) {
            return BUNDLE.getString("graphproperties.invariant");
        } else if (column == 1) {
            return BUNDLE.getString("graphproperties.value");
        } else {
            assert false : "invalid column count: " + column;
            return " ";
        }
    }
    
    //
    public int getRowCount() {
        return invariants.size();
    }
    
    //
    public GraphListElement getGraphListElement() {
        return element;
    }

    //
    public void addGraphPropertiesTableModelListener(GraphPropertiesTableModelListener listener) {
        listeners.add(listener);
    }
    
    //
    public void removeGraphPropertiesTableModelListener(GraphPropertiesTableModelListener listener) {
        listeners.remove(listener);
    }

    //
    public void fireGraphListElementChanged() {
        for(GraphPropertiesTableModelListener listener : listeners) {
            listener.graphListElementChanged();
        }
    }

    // implements GraphListElementListener
    public void graphListElementURIChanged(GraphListElement element) {
        //ignore
    }

    // implements GraphListElementListener
    public void graphListElementNameChanged(GraphListElement element) {
        //ignore
    }

    // implements GraphListElementListener
    public void graphListElementInvariantComputed(GraphListElement element, Invariant invariant) {
        initInvariants();
    }

    // implements GraphListElementListener
    public void graphListElementGraphChanged(GraphListElement element) {
        //ignore
    }

}
