/* GraphViewPanel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.util.Observable;
import java.util.Observer;
import javax.swing.JPanel;

import org.grinvin.Edge;
import org.grinvin.EmbeddingListener;
import org.grinvin.EmbeddingView;
import org.grinvin.GraphListener;
import org.grinvin.GraphURIType;
import org.grinvin.GraphView;
import org.grinvin.Vertex;
import org.grinvin.gred.GraphContext;
import org.grinvin.gred.Guides;
import org.grinvin.gred.guides.NullGuides;
import org.grinvin.render.Renderer;

/**
 * Provides a view of the Graph.
 * TODO: clean this up, refactor common code with GraphPanel
 */
public class GraphViewPanel extends JPanel implements ComponentListener, EmbeddingListener, GraphListener, Observer {

    /**
     * Panel background color.
     */
    protected Color backgroundColor = GraphURIType.GRAPH_SESSION.getIconBackgroundColor();
    
    
    //
    protected EmbeddingView embedding;
    
    //
    protected GraphView graph;
    
    //
    protected Renderer renderer;
    
    //
    protected GraphContext context;
    
    //
    protected double scale;

    //
    private static final Guides DEFAULT_GUIDES = new NullGuides ();
    
    /**
     * Current {@link Guides} object for this panel.
     */
    protected Guides guides;
    
    
    /**
     * Create a new graph panel with given peer. The panel is given a preferred size
     * large enough to display the coordinate range (-1.1,-1.1)-(1.1,1.1).
     * @param embedding Embedding displayed in this panel. The peer of this embedding
     * should be of type GraphModel.
     * @param renderer Renderer for this panel.
     * @param context Graph context for this panel.
     * @param scale Number of pixels corresponding to a unit length in the embedding.
     */
    public GraphViewPanel(EmbeddingView embedding, Renderer renderer, GraphContext context, double scale) {
        super();
        
        initEmbedding(embedding);
        this.renderer = renderer;
        
        this.context = context;
        context.addObserver(this);
        
        this.scale = scale;
        
        this.guides = DEFAULT_GUIDES; // null guides
        
        setBackground(backgroundColor);
        addComponentListener(this);
        
        int prefSize = (int)(scale * 13 / 5);
        setPreferredSize(new Dimension(prefSize, prefSize));
        
        //this.undoManager = new UndoManager();
        
        //setTransferHandler(new DropTransferHandler());
    }
    
    //
    private void initEmbedding(EmbeddingView embedding) {
        this.embedding = embedding;
        if (embedding.getDimension() != 2) {
            throw new IllegalArgumentException
                ("Cannot display embeddings of dimension other than 2");
        }

        //TODO: add to EmbeddingView: addEmbeddingListener
        //embedding.addEmbeddingListener(this);
        
        this.graph = embedding.getPeer();
/*
        if (this.graph instanceof GraphModel) {
            ((GraphModel)this.graph).addGraphListener(this);
        } else
            throw new IllegalArgumentException("Embedding should have peer of type GraphModel");
 */
    }
    
    
    /* ============================================================
     * PAINT
     * ============================================================ */
    
    /**
     * Paints the graph on the panel using the current renderer.
     * Paints background, edges and then vertices.
     */
    @Override protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D)g.create(); // TODO: regression test for forgetting
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
            RenderingHints.VALUE_ANTIALIAS_ON);
        g2.translate(getWidth()/2,getHeight()/2);
        paintBackground(g2);
        paintEdges(g2);
        paintVertices(g2);
    }
    
    /**
     * Paint a background overlay.
     * See {@link #paintComponent} for the order in which the various
     * paint methods are executed.
     * <p>This implementation is delegates to the current guides object.
     */
    protected void paintBackground(Graphics2D g2) {
        guides.paint (g2, scale, getWidth()/scale/2, getHeight()/scale/2);
    }
    
    /**
     * Paint all edges of the graph using the current renderer.<p>
     * See {@link #paintComponent} for the order in which the various
     * paint methods are executed.
     */
    protected void paintEdges(Graphics2D g2) {
        for(Edge e : graph.edges()) {
            double[] firstCoordinates = embedding.getCoordinates(e.getFirstEndpoint());
            double[] secondCoordinates = embedding.getCoordinates(e.getSecondEndpoint());
            /* Zoom */
            firstCoordinates[0] *= scale;
            firstCoordinates[1] *= -scale;
            secondCoordinates[0] *= scale;
            secondCoordinates[1] *= -scale;
            renderer.paint(e, firstCoordinates, secondCoordinates,
                context.isRollOver(e), context.isSelected(e), g2);
        }
    }
    
    /**
     * Paint all vertices of the graph using the current vertex renderer.<p>
     * See {@link #paintComponent} for the order in which the various
     * paint methods are executed.
     */
    protected void paintVertices(Graphics2D g2) {
        for (Vertex v : graph.vertices()) {
            double[] coordinates = embedding.getCoordinates(v);
            /* Zoom */
            coordinates[0] *= scale;
            coordinates[1] *= -scale;
            renderer.paint(v, coordinates,
                context.isRollOver(v), context.isSelected(v), g2);
        }
    }
        
    /* ============================================================
     * CONTEXT LISTENER
     * ============================================================ */
    
    // implements Observer
    public void update(Observable o, Object arg) {
        repaint();
    }
    
    /* ============================================================
     * COMPONENT LISTENER
     * ============================================================ */
    
    // implements ComponentListener
    public void componentShown(ComponentEvent e) {
    }
    
    /**
     * Recenters the drawing when the component changes size.
     */
    public void componentResized(ComponentEvent e) {
        repaint();
    }
    
    // implements ComponentListener
    public void componentMoved(ComponentEvent e) {
    }
    
    // implements ComponentListener
    public void componentHidden(ComponentEvent e) {
    }

    /* ============================================================
     * EMBEDDING LISTENER
     * ============================================================ */
    
    // implements EmbeddingListener
    public void vertexCoordinatesChanged(Vertex vertex) {
        repaint(); // TODO: only vertex area
    }
    
    // implements EmbeddingListener
    public void dimensionChanged(int oldDimension) {
        if (embedding.getDimension() != 2) {
            throw new IllegalStateException
                ("Cannot display embeddings of dimension other than 2");
        }
    }
    
    // implements EmbeddingListener
    public void embeddingChanged() {
        graph = embedding.getPeer();
        repaint();
    }

    /* ============================================================
     * GRAPH LISTENER
     * ============================================================ */
    
    // implements GraphListener
    public void vertexChanged(Vertex vertex) {
        repaint(); // TODO: only vertex area
    }
    
    // implements GraphListener
    public void vertexAdded(Vertex vertex) {
        repaint(); // TODO: only vertex area
    }
    
    // implements GraphListener
    public void vertexRemoved(Vertex vertex) {
        repaint(); // TODO: only vertex and adjacent edges
    }
    
    // implements GraphListener
    public void vertexRestored(Vertex vertex) {
        repaint(); // TODO: only vertex
    }
    
    // implements GraphListener
    public void edgeRemoved(Edge edge) {
        repaint(); // TODO: only edge and adjacent vertices
    }
    
    // implements GraphListener
    public void edgeRestored(Edge edge) {
        repaint(); // TODO: only edge and adjacent vertices
    }
    
    // implements GraphListener
    public void edgeChanged(Edge edge) {
        repaint(); // TODO: only edge area
    }
    
    // implements GraphListener
    public void edgeAdded(Edge edge) {
        repaint(); // TODO: only edge area
    }
    
    // implements GraphListener
    public void graphChanged() {
        repaint();
    }


}
