/* InvariantTreeModel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list;

import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantFactory;
import org.grinvin.invariants.InvariantGroup;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.InvariantNode;

/**
 * Tree model containing the tree of supported invariants.
 */
public class InvariantTreeModel extends DefaultTreeModel {
    
    // TODO: differentiate between invariants with a known computer and others
    
    //
    private final String name;
    
    // maps keys to nodes
    private final Map<String,MutableTreeNode> insertionPoints;
    
    /**
     * Return a model for the tree of invariants that corresponds
     * to the given list of nodes.
     */
    public InvariantTreeModel(Iterable<InvariantNode> list) {
        super(null);
        this.insertionPoints = new HashMap<String,MutableTreeNode> ();
        this.name = ResourceBundle.getBundle("org.grinvin.invariants.resources").getString("window.title");
        MutableTreeNode root = new DefaultMutableTreeNode(null, true);
        InvariantNode.Visitor visitor = new MyVisitor(root);
        for (InvariantNode node : list) {
            node.accept(visitor);
        }
        setRoot(root);
    }
    
    public String getName() {
        return name;
    }
    
    /**
     * Return the node that corresponds to the given named insertion point.
     */
    public MutableTreeNode getInsertionPoint(String key) {
        return insertionPoints.get(key);
    }
    
    //
    private static final InvariantTreeModel STANDARD_TREE
            = new InvariantTreeModel(InvariantManager.getInstance().getStandardInvariantNodes());
    
    
    /**
     * Return the tree model for the standard invariants that come with GrInvIn.
     */
    public static InvariantTreeModel getStandardTree () {
        return STANDARD_TREE;
    }    
    
    /**
     * Visitor which adds a visited invariant node as a child to a given
     * parent tree node.
     */
    private class MyVisitor implements InvariantNode.Visitor {
        
        //
        private final MutableTreeNode root;
        
        MyVisitor(MutableTreeNode root) {
            this.root = root;
        }
        
        public void visitInvariant(Invariant invariant) {
            root.insert(new DefaultMutableTreeNode(invariant, false), root.getChildCount());
        }
        
        public void visitFactory(InvariantFactory factory) {
            root.insert(new DefaultMutableTreeNode(factory, false), root.getChildCount());
        }
        
        public void visitGroup(InvariantGroup group) {
            String insert = group.getInsertionPoint();
            MutableTreeNode parent = new DefaultMutableTreeNode(group.getName(), true);
            if (insert != null)
                insertionPoints.put(insert, parent);
            MyVisitor childVisitor = new MyVisitor(parent);
            for (InvariantNode node : group.getChildren())
                node.accept(childVisitor);
            root.insert(parent, root.getChildCount());
        }
        
    }
    
}
