/* ImportGraphBundle.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list.commands;

import be.ugent.caagt.swirl.commands.AttributedCommandManager;

import java.io.File;
import java.io.IOException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;
import org.grinvin.DefaultGraphBundle;
import org.grinvin.GraphBundle;

import org.grinvin.graphbrowser.GraphBrowserMultiPanel;
import org.grinvin.io.Graph6Loader;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;
import org.grinvin.list.GraphListElementManager;

import org.pietschy.command.CommandManager;
import org.pietschy.command.file.AbstractFileOpenCommand;
import org.pietschy.command.file.ExtensionFileFilter;

/**
 * Command to import a (some) GraphBundle(s) and add it (them) to the current
 * GraphList.
 */
public class ImportGraphBundle extends AbstractFileOpenCommand {
    
    //
    private static final Logger LOGGER = Logger.getLogger ("org.grinvin.io");
    
    //
    private static final String KEY = GraphBrowserMultiPanel.class.getName();
    
    //
    private static final FileFilter GRAPHLIST_FILTER = new ExtensionFileFilter("gph", "GrInvIn graph bundle (*.gph)");

    //
    private static final FileFilter G6_FILTER = new ExtensionFileFilter("g6", "Graph6 graph (*.g6)");

    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle ("org.grinvin.list.commands.resources");
    
    //
    private GraphBrowserMultiPanel panel;
    
    //
    public ImportGraphBundle(CommandManager commandManager, String id) {
        super(commandManager, id, new FileFilter[] {GRAPHLIST_FILTER, G6_FILTER});
        this.setMultiselectionEnabled(true);
        this.panel = (GraphBrowserMultiPanel)((AttributedCommandManager)commandManager).getAttribute(KEY);
    }
    
    // implements performOpen
    public void performOpen(File[] files) {
        GraphList list = panel.getGraphList();
        for(File file : files) {
            if (GRAPHLIST_FILTER.accept(file)) {
                GraphListElement graphlistelement = GraphListElementManager.getInstance().getGraphListElement(file.toURI());
                if(!list.contains(graphlistelement))
                    list.add(graphlistelement);
                else
                    JOptionPane.showMessageDialog(panel, BUNDLE.getString("GraphList.duplicate"), BUNDLE.getString("GraphList.warning.title"), JOptionPane.DEFAULT_OPTION);
            } else if (G6_FILTER.accept(file)) {
                GraphBundle bundle = new DefaultGraphBundle();
                try {
                    Graph6Loader.readSingleGraph(bundle, file);
                    GraphListElement graphlistelement = GraphListElementManager.getInstance().createGraphListElement(bundle);
                    list.add(graphlistelement);
                } catch (IOException ex) {
                    LOGGER.log(Level.INFO, "Failed to import graph", ex);
                }
                
            }
        }
    }
    
}
