/* LocalClassLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import java.net.MalformedURLException;
import java.net.URL;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

/**
 * Class loader which loads classes from a <i>local classpath</i>, i.e., a
 * set of directories on the local file system.<p>
 * To load a class two different methods can be used:
 * <ul>
 * <li>Using {@link #loadClass} which is the standard method to retrieve a
 * class with a given name.</li>
 * <li>Using {@link #loadClassFromFile} to retrieve a class from a given
 * class file.</li>
 * </ul>
 */
public class LocalClassLoader extends ClassLoader {
    
    //
    private static final Logger LOGGER
            = Logger.getLogger ("org.grinvin.util");
    // TODO: implement findResources(...)
    
    /**
     * List of directories searched when loading the class. I.e., the local class
     * path.
     */
    private List<File> localClasspath;
    
    /**
     * Return the local class path, i.e., the directories
     * which are currently used to load classes from.<p>
     * Note that this method returns an iterable and not a list. This return
     * value is intended to be used in a for each loop:
     * <pre>
     *    for (File dir : myClassLoader().getLocalClasspath()) {
     *       // do something with dir
     *    }
     * </pre>
     */
    public Iterable<File> getLocalClasspath () {
        return localClasspath; // TODO: should disallow removes
    }
    
    //
    
    
    /**
     * Add a directory to the local class path. No action is taken
     * when the directory already belongs to the local class path.
     * @throws IllegalArgumentException when {@code dir} is not a directory.
     */
    public void addDirectory (File dir) {
        if (!dir.isDirectory ())
            throw new IllegalArgumentException ("Only directories are allowed");
        // TODO: support jars
        else if (!localClasspath.contains (dir))
            localClasspath.add (dir);
    }
    
    /**
     * Add a base directory which is derived from the given class name
     * which is known to reside in the given file.
     */
    public void addDirectory (File file, String name) {
        int pos = name.lastIndexOf ('.') + 1;
        if (file.getName ().equalsIgnoreCase (name.substring (pos)+".class")) {
            File dir = file.getParentFile ();
            while (pos > 0) {
                name = name.substring (0, pos-1);
                pos = name.lastIndexOf ('.') + 1;
                if (dir != null && ! dir.getName ().equalsIgnoreCase (name.substring (pos)))
                    return; // could not guess
                dir = dir.getParentFile ();
            }
            if (dir != null)
                addDirectory (dir);
        }
    }
    
    /** Creates a new instance of LocalClassLoader */
    public LocalClassLoader () {
        super(LocalClassLoader.class.getClassLoader());
        localClasspath = new ArrayList<File> ();
    }
    
    protected Class<?> findClass (String name) throws ClassNotFoundException {
        byte[] data = loadClassData (name);
        return defineClass (name, data, 0, data.length);
    }
    
    /**
     * Load a class file as binary data.
     * @throws ClassNotFoundException when an error occurred while loading the file
     */
    private byte[] loadClassFile (File file) throws ClassNotFoundException {
        long l = file.length ();
        if (l > Integer.MAX_VALUE)
            throw new RuntimeException ("Class file \'" + file + "\' is too large");
        int len = (int) l;
        byte[] result = new byte[len];
        try {
            InputStream is = new FileInputStream (file);
            try {
                int offs = 0;
                int size = is.read (result, offs, len);
                while (size > 0) {
                    offs += size;
                    len -= size;
                    size = is.read (result, offs, len);
                }
                return result;
            } finally {
                if (is != null)
                    is.close ();
            }
        } catch (IOException ex) {
            throw new ClassNotFoundException ("Error reading class file", ex);
        }
    }
    
    /**
     * Load a class as binary data.
     * @throws ClassNotFoundException when the class file could not be found
     * in the current list of directories or when an input-output
     * error occurred while loading the file
     */
    private byte[] loadClassData (String name) throws ClassNotFoundException {
        String path = name.replace ('.', File.separatorChar);
        for (File dir : localClasspath) {
            File file = new File (dir,path+".class");
            if (file.exists ())
                return loadClassFile (file);
        }
        throw new ClassNotFoundException ("Class '" + name + "' not found in local classpath");
    }
    
    protected URL findResource (String name) {
        for (File dir : localClasspath) {
            File file = new File (dir,name);
            if (file.exists ()) {
                try {
                    return file.toURI ().toURL ();
                } catch (MalformedURLException ex) {
                    // presently ignored
                }
            }
        }
        return null; // resource not found
    }
    
    /**
     * Load a class from a given class file. Apart from loading the file,
     * the system also tries to guess what is the directory for the base
     * package of this file and adds it to the local class path. This should
     * enable resolution of other local classes referred to by this class.
     */
    public Class<?> loadClassFromFile (File file) throws ClassNotFoundException {
        byte[] data = loadClassFile (file);
        try {
            Class<?> clazz = defineClass (null, data, 0, data.length);
            resolveClass (clazz);
            addDirectory (file, clazz.getName ());
            return clazz;
        } catch (LinkageError le) {
            LOGGER.warning ("Linkage error:" + le);
            throw le;
        }
    }
}
