/* DefaultWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.awt.Window;
import java.io.File;
import java.io.IOException;

import org.grinvin.io.Bounds;
import org.grinvin.io.IOFormatException;

import org.jdom.DataConversionException;
import org.jdom.Element;

/**
 * Common super class of all wrappers that manage a single window of 
 * type <tt>W</tt>.
 */
public abstract class DefaultWindowWrapper<W extends Window> implements Wrapper<W> {
       
    /**
     * {@inheritDoc}<p>
     * This implementation disposes of the associated window.
     */
    public void dispose(W window) {
        window.dispose ();
    }

    //
    public abstract W create();

    /**
     * General purpose routine which can be used to set bounds
     * and visbility of any window.
     */
    protected static void loadWindowProperties (Element element, Window window) throws IOFormatException {
        Bounds bounds = new Bounds(element);
        window.setLocation(bounds.location);
        window.setSize(bounds.size);
        try {
            window.setVisible (
                    element.getAttribute("isvisible").getBooleanValue() );
        } catch (DataConversionException ex) {
            throw new IOFormatException("Incorrect boolean attribute value", ex);
        }
    }
    
    /**
     * General purpose routine which can be used to save the
     * bounds and visibility of any window.
     */
    protected static void saveWindowProperties (Element element, Window window) {
        // TODO: use getBounds and make Bounds obsolete
        Bounds bounds = new Bounds(window.getLocation(), window.getSize());
        element.addContent(bounds.toElement());
        element.setAttribute("isvisible", "" + window.isVisible());
    }
    
    /**
     * {@inheritDoc}<p>This implementation creates the window using
     * {@link #create} and sets window bounds and visibility
     * from the information in the workspace XML-file.
     */
    public W load(Element element, File directory) throws IOException {
        W window = create ();
        loadWindowProperties(element, window);
        return window;
    }
    
    /**
     * Create a JDOM-element that contains the version information.
     * Further attributes and contents should be added later.
     */
    protected final Element baseElement() {
        Element result = new Element(getElementName());
        int major = getMajorVersion();
        if (major > 0)
            result.setAttribute("version", major+"."+getMinorVersion());
        return result;
        
    }
    
    /**
     * {@inheritDoc}<p>This implementation saves the information obtained
     * from the method {@link #baseElement} and {@link #saveWindowProperties} into <tt>parent</tt> and leaves
     * the directory alone.
     */
    public void save(W window, Element parent, File directory) throws IOException {
        Element result = baseElement ();
        saveWindowProperties (result, window);
        parent.addContent(result);
    }
    
    /**
     * This format has version 1.0
     */
    public int getMajorVersion () {
        return 1;
    }
    
    /**
     * This format has version 1.0
     */
    public int getMinorVersion () {
        return 0;
    }
    

}
