/* GrinvinWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;


import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.grinvin.graphbrowser.GraphBrowserWindow;

import org.grinvin.grinvin.GrinvinWindow;
import org.grinvin.io.IOFormatException;
import org.jdom.DataConversionException;
import org.jdom.Element;

/**
 * Wrapper for the main GrInvIn window.
 */
public class GrinvinWindowWrapper extends DefaultWindowWrapper<GrinvinWindow> {
    
    //
    private final Map<String,DefaultWindowWrapper> wrapperMap;
    
    //
    private final GraphBrowserWindowWrapper wrapper;
    
    //
    public GrinvinWindowWrapper() {
        wrapperMap = new HashMap<String,DefaultWindowWrapper> ();
        //wrapperMap.put ("AvailableInvariantsList", new InvariantTreeWindowWrapper ());
        //wrapperMap.put ("FactoryWindow",           new FactoryWindowWrapper ());
        wrapperMap.put("ConjecturingWindow",      new ConjecturingWindowWrapper());
        wrapperMap.put("EditorWindow",            new EditorWindowWrapper());
        wrapperMap.put("LoggingWindow",           new LoggingWindowWrapper());
        
        this.wrapper = new GraphBrowserWindowWrapper();
        //listsWindowWrapper = new GraphListsWindowWrapper ();
    }
    
    public String getElementName() {
        return "grinvinWindow";
    }
    
    /**
     * Create all top level windows
     * and add them to the menus of the main window.
     */
    @Override
    public GrinvinWindow create() {
        GrinvinWindow window = new GrinvinWindow();
        
        // initialize standard windows
        for (Map.Entry<String, DefaultWindowWrapper> entry : wrapperMap.entrySet()) {
            window.setWindow(entry.getKey(), entry.getValue().create());
        }
        
        window.loadGraphListWindowsInMenu();
        
        //
        //window.setGraphListsWindow(listsWindowWrapper.create());
        window.setVisible(true);
        return window;
    }
    
    /**
     * Dispose of this window and all dependent windows.
     */
    @Override
    public void dispose(GrinvinWindow window) {
        
        // dispose of all standard windows
        for (Map.Entry<String, DefaultWindowWrapper> entry : wrapperMap.entrySet()) {
            entry.getValue().dispose(window.getWindow(entry.getKey()));
        }
        
        //listsWindowWrapper.dispose(window.getGraphListsWindow());
        
        for (GraphBrowserWindow w: window.getGraphListsModel().getListModel())
            wrapper.dispose(w);
        
        super.dispose(window);
    }
    
    /**
     * Load this window and all dependent windows.
     */
    @Override
    public GrinvinWindow load(Element element, File directory) throws IOException {
        // create the window
        GrinvinWindow window = new GrinvinWindow();
        
        // load standard windows
        for (Map.Entry<String, DefaultWindowWrapper> entry : wrapperMap.entrySet()) {
            window.setWindow(entry.getKey(), entry.getValue().load(
                    element.getChild(entry.getValue().getElementName()),
                    directory
                    ));
        }
        
        // load lists window
        /*window.setGraphListsWindow(
                listsWindowWrapper.load(
                element.getChild(listsWindowWrapper.getElementName()),
                directory ));
         */
        
        // load graph lists
        String childElementName = wrapper.getElementName();
        for (Object obj : element.getChildren(childElementName))
            window.getGraphListsModel().getListModel().add(wrapper.load((Element)obj, directory));
        
        //handle old workspace
        Element oldElement = element.getChild("listsWindow");
        if(oldElement!=null)
            for (Object obj : oldElement.getChildren(childElementName))
                window.getGraphListsModel().getListModel().add(wrapper.load((Element)obj, directory));
        window.loadGraphListWindowsInMenu();
        
        try {
            int seqno = element.getAttribute("listSeqNum").getIntValue();
            window.getGraphListsModel().setListSequenceNumber(seqno);
        } catch (DataConversionException ex) {
            throw new IOFormatException("Incorrect integer attribute value", ex);
        } catch (NullPointerException ex) {
            //handle old workspace
            window.getGraphListsModel().setListSequenceNumber(1);
        }
        
        
        // make window visible
        loadWindowProperties(element, window);
        
        // select tab
        String selectedTab = element.getAttributeValue("tab");
        if(selectedTab==null)
            selectedTab = "invariants";
        try{
            window.setSelectedTab(selectedTab);
        } catch (IllegalArgumentException ex){
            //do nothing: just select first tab
        }
        return window;
    }
    
    /**
     * Save this window and all dependent windows.
     */
    @Override
    public void save(GrinvinWindow window, Element parent, File directory) throws IOException {
        Element result = baseElement();
        saveWindowProperties(result, window);
        
        // save tab
        result.setAttribute("tab", window.getSelectedTab());
        
        // save standard windows
        for (Map.Entry<String, DefaultWindowWrapper> entry : wrapperMap.entrySet()) {
            entry.getValue().save(window.getWindow(entry.getKey()),
                    result, directory );
        }
        
        // save lists window
        //listsWindowWrapper.save(window.getGraphListsWindow(), result, directory);
        
        // save graph lists
        result.setAttribute("listSeqNum", ""+window.getGraphListsModel().getListSequenceNumber());
        for (GraphBrowserWindow w: window.getGraphListsModel().getListModel()) {
            wrapper.save(w, result, directory);
        }
        
        parent.addContent(result);
    }
}
