/* StateStack.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.engine.apengine;

/**
 * Abstract class for handling state in the tree generators.
 */
public abstract class StateStack<E> {
    
        //
        private E[] contents;
        
        //
        protected int size;
        
        // location of the top of the stack
        protected int top;
        
        /**
         * Create a new StateStack.
         */
        public StateStack() {
            this(1);
        }
        
        /**
         * Create a new StateStack.
         * @param initSize the initial size of the stack
         */
        private StateStack(int initSize) {
            size = initSize;
            contents = createArray(size);
            // create stack objects once for the entire computation
            for(int i=0; i < size;i++) {
                contents[i] = emptyState();
            }
            top = 0;
        }
        
        //
        protected abstract E emptyState();
        
        //
        protected abstract E[] createArray(int size);
        
        /**
         * Extend the size of the stack.
         */
        protected void extend() {
            int newSize = size*2;
            E[] newContents = createArray(newSize);
            System.arraycopy(contents, 0, newContents, 0, size);
            for(int i=size; i < newSize;i++) {
                newContents[i] = emptyState();
            }
            contents = newContents;
            size = newSize;
        }

        //
        public final E pop() {
            top--;
            if (top >= 0)
                return contents[top];
            else
                return null;
        }
        
        //
        public final E peek() {
            if (top >= 0)
                return contents[top];
            else
                return null;
        }
        
        //
        public final boolean empty() {
            return top == 0;
        }
    
}
