/* HelpURLConnection.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.help;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Locale;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.grinvin.factories.FactoryManager;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.xml.XMLUtils;

import org.jdom.transform.JDOMSource;

class HelpURLConnection extends URLConnection{
    
    private final String css;
    private final String id;
    private final String xmlPath;
    
    private boolean connected;
    
    private final Source xsltSource;
    private final TransformerFactory transFact;
    private final PipedInputStream in;
    private final PipedOutputStream out;
    private final Result result;
    
    HelpURLConnection(URL url) throws IOException {
        super(url);
        xsltSource = new StreamSource(HelpURLConnection.class.getResourceAsStream("/org/grinvin/help/" + url.getProtocol() + ".xslt"));
        this.transFact = TransformerFactory.newInstance();
        transFact.setURIResolver(new HelpURIResolver());
        
        this.id = url.getPath();
        if ("graphfactory".equals(url.getProtocol())) {
            String base = FactoryManager.getGraphFactoryBroker().get(id).getClass().getName();
            this.xmlPath = base.replace('.','/') + ".xml";
        } else if ("generator".equals(url.getProtocol())) {
            String base = FactoryManager.getGeneratorBroker().get(id).getClass().getName();
            this.xmlPath = base.replace('.','/') + ".xml";
        }
        else
            this.xmlPath = InvariantManager.getInstance().getDefinitionPath(id);
        
        this.css = HelpURLConnection.class.getResource("/org/grinvin/helpsets/style.css").toString();
        this.in = new PipedInputStream();
        this.out = new PipedOutputStream(in);
        this.result = new StreamResult(out);
        
    }
    
    synchronized public void connect() throws IOException {
        
        
        final Source xmlSource = new JDOMSource (XMLUtils.loadFromClassPath(xmlPath));
        // cannot use StreamSource because of DTD location      
        
        try {
            // TODO: please do this off line
            final Transformer trans = transFact.newTransformer(xsltSource);
            
            trans.setParameter("locale", Locale.getDefault().toString());
            trans.setParameter("th.name", HelpManager.getResourceBundle().getString("th.name"));
            trans.setParameter("th.description", HelpManager.getResourceBundle().getString("th.description"));
            trans.setParameter("css", css);
            trans.setParameter("factory.id", id);
            
            new Thread(
                    new Runnable(){
                public void run(){
                    try {
                        trans.transform(xmlSource, result);
                        out.close();
                    } catch (TransformerException ex) {
                        //do nothing?
                    } catch (IOException ex) {
                        //do nothing?
                    }
                }
            }
            ).start();
        } catch (TransformerException e) {
            throw new IOException("Could not connect:" + e); 
            // TODO: Java 6.0 allows e as second argument
        }
        connected=true;
    }
    
    
    synchronized public InputStream getInputStream() throws IOException {
        if(!connected)
            connect();
        return in;
    }
    
    public String getContentType() {
        return "text/html";
    }
}
