/* VertexConnectivityFlow.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * Invariant computer which returns the vertex-connectivity using flows.
 */
public class VertexConnectivityFlow extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        int[][] adjlist = bundle.adjacencyList();
        if(adjlist.length<2)
            return new IntegerValue(0, this);
        
        //first calculate minimum degree (i.e. upperbound for vertex-connectivity)
        int mindeg = Integer.MAX_VALUE;
        for (int[] neighbours : adjlist)
            if (neighbours.length < mindeg)
                mindeg = neighbours.length;  
        
        if(mindeg==adjlist.length-1)
            //complete graph
            return new IntegerValue(mindeg, this);
        
        int minimumCutSize = mindeg;
        for(int i = 0; i < minimumCutSize + 1; i++)
            for(int j = i; j < adjlist.length; j++)
                minimumCutSize = findMaxFlowInSTNetwork(adjlist, i, j, minimumCutSize);
        
        
        return new IntegerValue(minimumCutSize, this);
    }
    
    //returns the minimum of the maxflow of the st-network and satisfied
    private int findMaxFlowInSTNetwork(int[][] adjlist, int source, int target, int satisfied){
        //construct directed graph
        int[][] directedGraph = new int[2*adjlist.length][2*adjlist.length];
        for(int i=0; i<adjlist.length; i++){
            for(int neighbour : adjlist[i])
                directedGraph[i + adjlist.length][neighbour]++;
            directedGraph[i][i + adjlist.length]++;
        }
        int pathCount = 0;
        while(findPath(adjlist, source + adjlist.length, target, directedGraph, new boolean[2*adjlist.length]) && pathCount < satisfied)
            pathCount++;
        return pathCount;
    }
    
    //try to find a path from the current vertex to the target
    private boolean findPath(int[][] adjlist, int currentVertex, int target, int[][] directedGraph, boolean[] currentPath) {
        currentPath[currentVertex]=true;
        if(currentVertex==target)
            return true;
        
        if(currentVertex < adjlist.length){ //in-vertex
            
            for(int nextVertex : adjlist[currentVertex]){
                nextVertex+= adjlist.length;
                if(directedGraph[currentVertex][nextVertex]>0 && !currentPath[nextVertex]){
                    directedGraph[currentVertex][nextVertex]--;
                    directedGraph[nextVertex][currentVertex]++;
                    if(findPath(adjlist, nextVertex, target, directedGraph, currentPath)){
                        return true;
                    } else {
                        directedGraph[currentVertex][nextVertex]++;
                        directedGraph[nextVertex][currentVertex]--;
                    }
                }
            }           
            int nextVertex = currentVertex + adjlist.length;
            if(directedGraph[currentVertex][nextVertex]>0 && !currentPath[nextVertex]){
                directedGraph[currentVertex][nextVertex]--;
                directedGraph[nextVertex][currentVertex]++;
                if(findPath(adjlist, nextVertex, target, directedGraph, currentPath)){
                    return true;
                } else {
                    directedGraph[currentVertex][nextVertex]++;
                    directedGraph[nextVertex][currentVertex]--;
                }
            }     
            
        } else { //out-vertex
            
            for(int nextVertex : adjlist[currentVertex - adjlist.length]){
                if(directedGraph[currentVertex][nextVertex]>0 && !currentPath[nextVertex]){
                    directedGraph[currentVertex][nextVertex]--;
                    directedGraph[nextVertex][currentVertex]++;
                    if(findPath(adjlist, nextVertex, target, directedGraph, currentPath)){
                        return true;
                    } else {
                        directedGraph[currentVertex][nextVertex]++;
                        directedGraph[nextVertex][currentVertex]--;
                    }
                }
            }           
            int nextVertex = currentVertex - adjlist.length;
            if(directedGraph[currentVertex][nextVertex]>0 && !currentPath[nextVertex]){
                directedGraph[currentVertex][nextVertex]--;
                directedGraph[nextVertex][currentVertex]++;
                if(findPath(adjlist, nextVertex, target, directedGraph, currentPath)){
                    return true;
                } else {
                    directedGraph[currentVertex][nextVertex]++;
                    directedGraph[nextVertex][currentVertex]--;
                }
            }     
            
        }
   
        currentPath[currentVertex]=false;
        return false;
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.VertexConnectivity";
    }
    
}
