/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.test.remoting.configuration;

import junit.framework.TestCase;
import org.jboss.logging.Logger;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.InvokerRegistry;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.ClientInvoker;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.multiplex.MultiplexClientInvoker;
import org.jboss.remoting.transport.socket.SocketClientInvoker;

import javax.management.MBeanServer;
import java.util.HashMap;
import java.util.Map;


/**
 * This class tests to insure that when <code>Client.invoke()</code> recreates the client invoker, it
 * passes the configuration map to <code>InvokerRegistry</code>.  See JIRA issue JBREM-335 (this is old).
 * New Jira issue is JBREM-479.
 * <p/>
 *
 * @author <a href="mailto:ron.sigal@jboss.com">Ron Sigal</a>
 * @version $Revision$
 *          <p/>
 *          Copyright (c) 2006
 *          </p>
 */
public class ClientConfigurationMapTestCase extends TestCase
{
   protected static final Logger log = Logger.getLogger(ClientConfigurationMapTestCase.class);

   public void testMultiplex() throws Throwable
   {
      //Start a server
      Connector serverConnector = new Connector();
      String locatorURI = "multiplex://localhost:9088";
      InvokerLocator serverLocator = new InvokerLocator(locatorURI);
      locatorURI = serverLocator.getLocatorURI();
      serverConnector.setInvokerLocator(locatorURI);
      serverConnector.create();
      SimpleServerInvocationHandler invocationHandler = new SimpleServerInvocationHandler();
      serverConnector.addInvocationHandler("test", invocationHandler);
      serverConnector.start();

      //Create a client       
      Map configuration = new HashMap();
      configuration.put(SocketClientInvoker.SO_TIMEOUT_FLAG, "123456");
      configuration.put("foobar", "barfoo");
      InvokerLocator clientLocator = new InvokerLocator(locatorURI + "/?junk"); // Don't get LocalInvoker
      Client client = new Client(clientLocator, configuration);
      client.connect();
      log.info("Created client");

      // should NOT get same client invoker as the one for client above since config passed is not the same
      MultiplexClientInvoker clientInvoker1 = (MultiplexClientInvoker) InvokerRegistry.createClientInvoker(clientLocator);
      assertEquals(MultiplexClientInvoker.SO_TIMEOUT_DEFAULT, clientInvoker1.getTimeout());
      assertNotSame(client.getInvoker(), clientInvoker1);

      // should get the same client invoker as the one for client above since config passed *is* the same
      MultiplexClientInvoker clientInvoker2 = (MultiplexClientInvoker) InvokerRegistry.createClientInvoker(clientLocator, configuration);
      assertEquals(123456, clientInvoker2.getTimeout());
      assertSame(client.getInvoker(), clientInvoker2);

      Map configuration2 = new HashMap();
      configuration2.put(SocketClientInvoker.SO_TIMEOUT_FLAG, "123456");
      // should *not* get the same client invoker as the one for client above since config passed is not the same
      MultiplexClientInvoker clientInvoker3 = (MultiplexClientInvoker) InvokerRegistry.createClientInvoker(clientLocator, configuration2);
      assertEquals(123456, clientInvoker3.getTimeout());
      assertNotSame(client.getInvoker(), clientInvoker3);

      // now to test ref counting in invoker registry
      ClientInvoker originalInvoker = client.getInvoker();
      // this should remove one of the reference counts to original invoker
      InvokerRegistry.destroyClientInvoker(clientLocator, configuration);

      // now will check to make sure gets original invoker again after removing one reference to it

      // should get the same client invoker as the one for client above since config passed *is* the same
      clientInvoker2 = (MultiplexClientInvoker) InvokerRegistry.createClientInvoker(clientLocator, configuration);
      assertEquals(123456, clientInvoker2.getTimeout());
      assertSame(client.getInvoker(), clientInvoker2);
      // this should remove one of the reference counts to original invoker
      InvokerRegistry.destroyClientInvoker(clientLocator, configuration);
      // this should remove the other reference, making ref count = 0 and destroying the client invoker;
      client.disconnect();

      // should *not* get the same client invoker as the one for client above since one above should have been destroyed
      clientInvoker2 = (MultiplexClientInvoker) InvokerRegistry.createClientInvoker(clientLocator, configuration);
      assertEquals(123456, clientInvoker2.getTimeout());
      assertNotSame(originalInvoker, clientInvoker2);

      client.disconnect();
      serverConnector.stop();
   }

   class SimpleServerInvocationHandler implements ServerInvocationHandler
   {
      public void addListener(InvokerCallbackHandler callbackHandler)
      {
      }

      public Object invoke(InvocationRequest invocation) throws Throwable
      {
         return "Sausages";
      }

      public void removeListener(InvokerCallbackHandler callbackHandler)
      {
      }

      public void setInvoker(ServerInvoker invoker)
      {
      }

      public void setMBeanServer(MBeanServer server)
      {
      }
   }
}
