package jdepend.framework;

import java.io.*;
import java.util.*;

/**
 * The <code>ClassFileParserTest</code> is a <code>TestCase</code> 
 * for the <code>ClassFileParser</code> class.
 * 
 * @author <b>Mike Clark</b> (mike@clarkware.com)
 * @author Clarkware Consulting, Inc.
 */
 
public class ClassFileParserTest extends JDependTestCase {

	private ClassFileParser parser;
	private String baseClassDir;
	
	
	public ClassFileParserTest(String name) {
		super(name);
	}

	protected void setUp() {
		super.setUp();
		PackageFilter filter = new PackageFilter(new ArrayList());
		parser = new ClassFileParser(filter);
		
		baseClassDir = getClassesDir() + 
			"jdepend" + File.separator + 
        	"framework" + File.separator;
	}

	protected void tearDown() {
	    super.tearDown();
	}
    
    public void testParseInvalidClassFile() {
	
        File f = new File(getSrcDir() +
            "jdepend" + File.separator + 
            "framework" + File.separator + "JDepend.java");
            
        try {
        
            parser.parse(f);
            fail("Invalid class file: Should raise IOException");
        
        } catch(IOException success)  {
        }
    }

	public void testParseInterfaceClass() throws IOException {

		JavaClass clazz = 
			parser.parse(new File(baseClassDir + "MockInterface.class"));

		assertTrue(clazz.isAbstract());

		assertEquals("jdepend.framework.MockInterface", 
        	clazz.getName());
            
        assertEquals("MockInterface.java", clazz.getSourceFile());

		Collection imports = clazz.getImportedPackages();
		assertEquals(6, imports.size());
		
		assertTrue(imports.contains(new JavaPackage("java.math")));
		assertTrue(imports.contains(new JavaPackage("java.text")));
		assertTrue(imports.contains(new JavaPackage("java.lang")));
		assertTrue(imports.contains(new JavaPackage("java.io")));
		assertTrue(imports.contains(new JavaPackage("java.rmi")));
		assertTrue(imports.contains(new JavaPackage("java.util")));
	}

	public void testParseAbstractClass() throws IOException {
	
		JavaClass clazz = 
			parser.parse(new File(baseClassDir + "MockAbstractClass.class"));

		assertTrue(clazz.isAbstract());

		assertEquals("jdepend.framework.MockAbstractClass", 
        	clazz.getName());
            
        assertEquals("MockAbstractClass.java", clazz.getSourceFile());

		Collection imports = clazz.getImportedPackages();
		assertEquals(7, imports.size());
		
		assertTrue(imports.contains(new JavaPackage("java.math")));
		assertTrue(imports.contains(new JavaPackage("java.text")));
		assertTrue(imports.contains(new JavaPackage("java.lang")));
		assertTrue(imports.contains(new JavaPackage("java.lang.reflect")));
		assertTrue(imports.contains(new JavaPackage("java.io")));
		assertTrue(imports.contains(new JavaPackage("java.rmi")));
		assertTrue(imports.contains(new JavaPackage("java.util")));
	}

	public void testParseConcreteClass() throws IOException {

		JavaClass clazz = 
			parser.parse(new File(baseClassDir + "MockConcreteClass.class"));

		assertFalse(clazz.isAbstract());

		assertEquals("jdepend.framework.MockConcreteClass",
        	clazz.getName());
            
        assertEquals("MockConcreteClass.java", clazz.getSourceFile());

		Collection imports = clazz.getImportedPackages();
		assertEquals(9, imports.size());
		
		assertTrue(imports.contains(new JavaPackage("java.net")));
		assertTrue(imports.contains(new JavaPackage("java.text")));
		assertTrue(imports.contains(new JavaPackage("java.sql")));
		assertTrue(imports.contains(new JavaPackage("java.lang")));
		assertTrue(imports.contains(new JavaPackage("java.io")));
		assertTrue(imports.contains(new JavaPackage("java.rmi")));
		assertTrue(imports.contains(new JavaPackage("java.util")));
		assertTrue(imports.contains(new JavaPackage("java.util.jar")));
		assertTrue(imports.contains(new JavaPackage("java.math")));	
	}
    
    public void testParseInnerClass() throws IOException {

        JavaClass clazz = 
            parser.parse(new File(baseClassDir + "MockConcreteClass$MockInnerClass.class"));

        assertFalse(clazz.isAbstract());

        assertEquals("jdepend.framework.MockConcreteClass$MockInnerClass",
            clazz.getName());
            
        assertEquals("MockConcreteClass.java", clazz.getSourceFile());

        Collection imports = clazz.getImportedPackages();
        assertEquals(1, imports.size());
            
        assertTrue(imports.contains(new JavaPackage("java.lang"))); 
        
    }

    public void testParsePackageClass() throws IOException {

        JavaClass clazz = 
            parser.parse(new File(baseClassDir + "MockPackageClass.class"));

        assertFalse(clazz.isAbstract());

        assertEquals("jdepend.framework.MockPackageClass",
            clazz.getName());
            
        assertEquals("MockConcreteClass.java", clazz.getSourceFile());

        Collection imports = clazz.getImportedPackages();
        assertEquals(1, imports.size());
            
        assertTrue(imports.contains(new JavaPackage("java.lang"))); 
        
    }
    
	public static void main(String args[]) {
		junit.textui.TestRunner.run(ClassFileParserTest.class);
	}
}
