// license:GPL-2.0+
// copyright-holders:Felipe Sanches
/***************************************************************************
  This driver covers only the Operator Panel of the BancTec 91690 Document Processor equipment

  Author: Felipe Sanches <juca@members.fsf.org>

  Maintainence Manual: <https://garoa.net.br/w/images/PAINEL_BANCTEC_91690.PDF>
*/

#include "emu.h"
#include "cpu/mcs51/mcs51.h"
#include "cpu/m6800/m6800.h"
#include "includes/banctec.h"
#include "video/mc6845.h"

static ADDRESS_MAP_START( banctec_mem , AS_PROGRAM, 8, banctec_state )
	AM_RANGE(0x0000, 0x07ff) AM_ROM
	AM_RANGE(0x0800, 0xffff) AM_RAM /* Probably wrong. Must be verified on pcb! */
ADDRESS_MAP_END

static ADDRESS_MAP_START( banctec_mcu_mem , AS_PROGRAM, 8, banctec_state )
	AM_RANGE(0x0000, 0x1fff) AM_ROM
	AM_RANGE(0x2000, 0x7fff) AM_RAM /* Probably wrong. Must be verified on pcb! */
	AM_RANGE(0x8000, 0xffff) AM_RAM AM_SHARE("videoram") /* Probably wrong. Must be verified on pcb! */
ADDRESS_MAP_END

void banctec_state::machine_reset()
{
}


/****************************
* Video/Character functions *
****************************/

TILE_GET_INFO_MEMBER(banctec_state::get_bg_tile_info)
{
	UINT8 *videoram = m_videoram;
	int code = videoram[tile_index];
	int color = 1;

	SET_TILE_INFO_MEMBER(0, code, color, 0);
}

void banctec_state::video_start()
{
	m_bg_tilemap = &machine().tilemap().create(m_gfxdecode, tilemap_get_info_delegate(FUNC(banctec_state::get_bg_tile_info),this), TILEMAP_SCAN_ROWS, 8, 8, 40, 25);
}

UINT32 banctec_state::screen_update_banctec(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	m_bg_tilemap->draw(screen, bitmap, cliprect, 0, 0);

	return 0;
}

/* ROCKWELL 6545 - Transparent Memory Addressing */

MC6845_ON_UPDATE_ADDR_CHANGED(banctec_state::crtc_addr)
{
	/* What is this function meant to do ? */
}

/******************************
* Graphics Decode Information *
******************************/

const gfx_layout banctec_gfx_layout =
{
	8, 8,               /* 8x8 characters */
	256,                /* 256 characters */
	1,                  /* 1 bits per pixel */
	{0},                /* no bitplanes; 1 bit per pixel */
	{0, 1, 2, 3, 4, 5, 6, 7},
	{0 * 256*8, 1 * 256*8, 2 * 256*8, 3 * 256*8, 4 * 256*8, 5 * 256*8, 6 * 256*8, 7 * 256*8},
	8                 /* size of one char */
};

static GFXDECODE_START( banctec )
			GFXDECODE_ENTRY( "gfx", 0x00000, banctec_gfx_layout, 0, 2 )
GFXDECODE_END

static MACHINE_CONFIG_START( banctec, banctec_state )
	/* basic machine hardware */

	MCFG_CPU_ADD("maincpu", I80C31, XTAL_11_0592MHz)
	MCFG_CPU_PROGRAM_MAP(banctec_mem)

	MCFG_CPU_ADD("mcu", M6803, 4000000)     /* Actual MCU is a Motorola 6803 and the clock frequency is still unknown */
	MCFG_CPU_PROGRAM_MAP(banctec_mcu_mem)

// The video signal is generated by a R6545EAP character generator chip
// The U20 EPROM holds the image data for the character set.

	// video hardware
	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_REFRESH_RATE(60)
	MCFG_SCREEN_SIZE((52+1)*8, (31+1)*8)
	MCFG_SCREEN_VISIBLE_AREA(0*8, 40*8-1, 0*8, 25*8-1)
	MCFG_SCREEN_UPDATE_DRIVER(banctec_state, screen_update_banctec)
	MCFG_SCREEN_PALETTE("palette")

	MCFG_GFXDECODE_ADD("gfxdecode", "palette", banctec)
	MCFG_PALETTE_ADD("palette", 2)
//  MCFG_PALETTE_INIT_OWNER(banctec_state, banctec)

	MCFG_MC6845_ADD("crtc", R6545_1, "screen", XTAL_20MHz) /* (?) */
	MCFG_MC6845_SHOW_BORDER_AREA(false)
	MCFG_MC6845_CHAR_WIDTH(8)
	MCFG_MC6845_ADDR_CHANGED_CB(banctec_state, crtc_addr)

MACHINE_CONFIG_END

ROM_START(banctec)
	ROM_REGION(0x800,"maincpu",0)
	ROM_LOAD("banctec_eseries_panel_opnl.u20", 0x000, 0x800, CRC(c2ab9c06) SHA1(a296589034f656790ad5ffbce028dd846a40cf03))

	ROM_REGION(0x2000,"mcu",0)
	ROM_LOAD("banctec_eseries_panel.u8", 0x0000, 0x2000, CRC(f3335e0a) SHA1(5ca45fdcb7ef45a65c28c79abfa9ebb7a8a06619))

	ROM_REGION(0x1000,"gfx",0)
	ROM_LOAD("banctec_eseries_panel.u20", 0x0000, 0x1000, CRC(5b6ecec9) SHA1(35aff8f965bce77205e3a43d71e39097585091a7))
ROM_END

/***************************************************************************

  Game driver(s)

***************************************************************************/

/*    YEAR  NAME      PARENT    COMPAT  MACHINE   INPUT     INIT    MONITOR COMPANY   FULLNAME */
CONS( 1989, banctec, 0,        0,      banctec, 0, driver_device, 0,       "DALE Electronics",  "BancTec ESeries Panel", MACHINE_NOT_WORKING | MACHINE_NO_SOUND)
