/*
 *   dspitcb.c -- Routines that provide all ITCB related functionality
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <stdio.h>
#include <string.h>
#include "dspmgr.h"

#include "dspquery.h"                  // CH01 get NameToXXX prototypes
#include "dspgpc.h"                    // Include function prototypes
#include "dspitcb.h"                   // Include function prototypes
#include "dspmem.h"                    // Include function prototypes
#include "dspstruc.h"                  // Include function prototypes
#include "dspglist.h"                  // Include CLL handling
#include "dspactiv.h"                  // Include Activate Task routines
#include "dspbios.h"                   // Include Bios routines
#include "dspfxutl.h"                  // Include Bios routines
#include "dspparse.h"                  // String compare routine
#include "dspload.h"
#include "dspfree.h"
#include "dspstcmp.h"                  // Function Prototypes

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ProcessITCBCL                                           */
/*                                                                          */
/* FUNCTION: This routines will traverse the ITCB connection list of a      */
/*           module, connect the Primary and Secondary ITCBs, and also      */
/*           fill in the Primary and Secondary ITCB data structure pointers.*/
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD prmod - Ptr to module. Connect all ITCB's in this module.   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: ITCBCL deleted for the module since its no longer needed.  */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ProcessITCBCL(PRMOD prmod)

{
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   PRITCBCL   pritcbcl;
   PRTSK      prPrimarytask;
   PRTSK      prSecondarytask;

   /*************************************************************************/
   /* Traverse the ITCB Connection list for this module.                    */
   /*************************************************************************/

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspitcb::ProcessITCBCL entry prmod %x\n",(int)prmod);
   while (prmod->MOD_pritcbclTail != NULL)/* list empty?                    */
      {
      pritcbcl = prmod->MOD_pritcbclTail;

      /**********************************************************************/
      /* for the PRIMARY itcb ....                                          */
      /* If the module name is 'BIOS' set prPrimarytask to NULL,            */
      /* else ignore module name, search module for task.                   */
      /**********************************************************************/

      if ((mwstrcmpnull(pritcbcl->ITCBCL_pszITCBPModName, "BIOS\0") == 0))
         prPrimarytask = NULL;
      else {
         if (pritcbcl->ITCBCL_pszITCBPTaskName == NULL)
            ulRC = DSP_NAME_NOT_FOUND;
         else {
            ulRC = NameToTaskHandle((HMOD)prmod,
               pritcbcl->ITCBCL_pszITCBPTaskName, (HTASK *)&prPrimarytask);
         }
         if (ulRC != DSP_NOERROR) {
	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspitcb::ProcessITCBCL error ulRC %lx Primary ITBC: %s\n",ulRC,
               pritcbcl->ITCBCL_pszITCBPTaskName);
            return (ulRC);
         }
      }

      /**********************************************************************/
      /* for the SECONDARY itcb ....                                        */
      /* If the module name is 'BIOS' set prSecondarytask to NULL,          */
      /* else ignore module name, search module for task.                   */
      /**********************************************************************/

      if ((mwstrcmpnull(pritcbcl->ITCBCL_pszITCBSModName, "BIOS\0") == 0))
         prSecondarytask = NULL;
      else {
         if (pritcbcl->ITCBCL_pszITCBSTaskName == NULL)
            ulRC = DSP_NAME_NOT_FOUND;
         else {
            ulRC = NameToTaskHandle((HMOD)prmod,
               pritcbcl->ITCBCL_pszITCBSTaskName, (HTASK *)&prSecondarytask);
         }
         if (ulRC != DSP_NOERROR) {
	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspitcb::ProcessITCBCL error ulRC %lx Secondary ITCB: %s\n", ulRC,
               pritcbcl->ITCBCL_pszITCBSTaskName);
            return (ulRC);
         }
      }

      /**********************************************************************/
      /* make the connection and Delete the ITCBCL.                         */
      /**********************************************************************/

      if ((ulRC = ConnectITCB(prPrimarytask, pritcbcl->ITCBCL_pszITCBPName,
         prSecondarytask, pritcbcl->ITCBCL_pszITCBSName)) != DSP_NOERROR)
         return (ulRC);

      if ((ulRC = RemoveITCBCL(prmod, pritcbcl)) != DSP_NOERROR)
         return (ulRC);
   }                                   /* end while                         */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspitcb::ProcessITCBCL exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveITCBCL                                            */
/*                                                                          */
/* FUNCTION: This routines will delete One ITCB Control Block from the      */
/*           Connection list.                                               */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD   prmod - Pointer to the Module with the Connection list    */
/*        PRITCBCL pritcbcl - Pointer to ITCBCL node to delete.             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveITCBCL(PRMOD prmod,PRITCBCL pritcbcl)

{
   RC         ulRC;                    // Error return code
   PRITCBCL   pritcbclPrev;            // Prev
   PRITCBCL   pritcbclCur;

   /*************************************************************************/
   /* Check to see if there are any ITCBCL blocks to process.               */
   /*************************************************************************/

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspitcb::RemoveITCBCL entry prmod %x\n",(int)prmod);
   if (prmod->MOD_pritcbclTail == NULL)
      return (DSP_NOERROR);

   /*************************************************************************/
   /* Remove.... takes the ITCBCL off the linked-list and recovers          */
   /* the structures memory.  So find the right pointers for                */
   /* manipulation; Namely the previous to the current ITCBCL               */
   /*************************************************************************/

   pritcbclPrev = prmod->MOD_pritcbclTail;
   pritcbclCur = pritcbclPrev->ITCBCL_pritcbclNext;

   while (pritcbclCur != pritcbcl) {
      pritcbclPrev = pritcbclCur;
      pritcbclCur = pritcbclCur->ITCBCL_pritcbclNext;
   }                                   /* endwhile                          */
   ulRC = RemoveRITCBConnList(&(prmod->MOD_pritcbclTail), pritcbcl,
      pritcbclPrev);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspitcb::RemoveITCBCL exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ConnectITCB                                             */
/*                                                                          */
/* FUNCTION: This routines will connect two ITCBs.  This includes setting   */
/*           both connectors to the same buffer and                         */
/*           filling in the Primary and Secondary ITCB data structure ptrs. */
/*                                                                          */
/* INPUT:                                                                   */
/*        HTASK  hPrimaryTask - Handle of Primary Task for ITCB.            */
/*        PSZ   pszPrimary   - Name of Primary ITCB.                        */
/*        HTASK  hSecondaryTask  - Handle of Secondary Task for ITCB.       */
/*        PSZ   pszSecondary    - Name of Secondary ITCB.                   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ConnectITCB(PRTSK prtskPrimary,PSZ pszPrimary,PRTSK prtskSecondary,PSZ
                pszSecondary)

{
   RC         ulRC=DSP_NOERROR;        // Error return code. CH03 Initialized.
   PRITCB     pritcbPrimary;           // Pointer to owner ITCB
   PRITCB     pritcbSecondary;         // Pointer to Secondary ITCB
   PRITCB     pritcbNode;              // General Pointer to a ITCB Node
   PRBIOT     prbiot;                  // PRBIOT
   USHORT     usITCBAddr;              // Address of ITCB

   /************************************************************************/
   /*  CH02 If either prtsk is not NULL it better be valid!  MTS #2675     */
   /************************************************************************/
   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspitcb::ConnectITCB entry prtskPrimary %x pszPrimary %s prtskSecondary %x pszSecondary %s\n",
	   (int)prtskPrimary,pszPrimary,(int)prtskSecondary,pszSecondary);

   if ((prtskSecondary!=NULL) && (prtskSecondary->TSK_achValid!=TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH02 Chk before using MTS #2675 */

   if ((prtskPrimary!=NULL) && (prtskPrimary->TSK_achValid != TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH02 Chk before using MTS #2675 */

   /********************************************************
    * If either the Owner or User Tasks are NULL.
    * Call ConnectorToBIOS() to find the bios task
    * Call GetBIOSTask to Load the BIOS task.
    *******************************************************                 */
   /*  If both User and Owner handles are null, ie. BIOS tasks, then return */
   /*  DSP_INV_HANDLE because we don't know which DSP to load them on.      */
   /*  This also prevents a trap which occurred when attempting to access a */
   /*  task structure with a null prtskSecondary pointer                    */

   if (prtskPrimary == NULL) {
      if (prtskSecondary == NULL)
         ulRC = DSP_INV_HANDLE;
      else {
         ulRC = ConnectorToBIOS(pszPrimary, pg->prmgrTail, &prbiot);
         if (ulRC == DSP_NOERROR) {
            ulRC = GetBIOSTask(prbiot->BT_pszFilename, prbiot->BT_pszTaskname,
               prtskSecondary->TSK_prdsp->DSP_prmodOS, /* CH03 */
               prtskSecondary->TSK_prdsp,
               &prtskPrimary);
         }
      }
   }

   else if (prtskSecondary == NULL) { /* CH03 Added "else" clause */
      ulRC = ConnectorToBIOS(pszSecondary, pg->prmgrTail, &prbiot);
      if (ulRC == DSP_NOERROR) {
         ulRC = GetBIOSTask(prbiot->BT_pszFilename, prbiot->BT_pszTaskname,
            prtskPrimary->TSK_prdsp->DSP_prmodOS, /* CH03 */
            prtskPrimary->TSK_prdsp,
            &prtskSecondary);
      }
   }

   if (ulRC != DSP_NOERROR) /* CH03 Combined several tests into this one */
      return (ulRC);

   if (prtskPrimary->TSK_prdsp != prtskSecondary->TSK_prdsp)
      return (DSP_INV_ITCB_CONNECT);

   if ((ulRC=CheckDSPStatus(prtskPrimary->TSK_prdsp)))
      return (ulRC);  /* CH04 Check for card unplugged */

   /*************************************************************************/
   /* Verify that there are Data Segments attached to the                   */
   /* tasks.                                                                */
   /* Verify that there are ITCBs attached to Segments.                     */
   /* The ITCBs are in the Data Segments.                                   */
   /*************************************************************************/

   if ((prtskPrimary->TSK_prsegDTail == NULL) ||
      (prtskSecondary->TSK_prsegDTail == NULL))
      return (DSP_NO_DATA);

   /*************************************************************************/
   /* Find the ITCBs within the Tasks. Search  ITCBs until                  */
   /* a match is found or the entire list has been checked.                 */
   /* Start at Tail->Next and go thru Tail.                                 */
   /* If a match is found, bFound will be set to TRUE and                   */
   /* the loop will end with pritcbNode pointing to the                     */
   /* correct ITCB.                                                         */
   /*************************************************************************/
   /*************************************************************************/
   /* Find the Primary ITCB.                                                */
   /*************************************************************************/

   if ((ulRC = FindITCB(prtskPrimary, pszPrimary, &pritcbNode)) != DSP_NOERROR
      )
      return (ulRC);

   if ((pritcbNode->ITCB_usflgKind&ITCB_KINDMASK) == ITCB_PRIMARY)
      pritcbPrimary = pritcbNode;
   else
      return (DSP_INV_ITCB_TYPE);

   /*************************************************************************/
   /* Find the Secondary ITCB.                                              */
   /*************************************************************************/

   if ((ulRC = FindITCB(prtskSecondary, pszSecondary, &pritcbNode)) !=
      DSP_NOERROR)
      return (ulRC);

   if ((pritcbNode->ITCB_usflgKind&ITCB_KINDMASK) == ITCB_SECONDARY)
      pritcbSecondary = pritcbNode;
   else
      return (DSP_INV_ITCB_TYPE);

   /*************************************************************************/
   /* Make sure that the ITCB is not connected (ITCB_pritcbPrimary == NULL) */
   /*************************************************************************/

   if (pritcbSecondary->ITCB_pritcbPrimary == NULL) { /* CH01 DEC           */

      /**********************************************************************/
      /* Make sure the Primary buffer is >= the secondary buffer            */
      /**********************************************************************/
      if (pritcbPrimary->ITCB_ulSize < pritcbSecondary->ITCB_ulSize)
         return (DSP_INV_ITCB_CONNECT);

      /**********************************************************************/
      /* If the protocols are OK, have to find the Offset                   */
      /* of the Label in the Primary.                                       */
      /* Then have to find the Offset to the label in the                   */
      /* Secondary.                                                         */
      /* Then have to write the Primary's address into the                  */
      /* secondary.                                                         */
      /**********************************************************************/

      pritcbSecondary->ITCB_prmemVariable = pritcbPrimary->ITCB_prmemVariable;

      /**********************************************************************/
      /*  Write the address of the Primary ITCB in the Secondary            */
      /*  segment.                                                          */
      /**********************************************************************/

      if (ulRC == DSP_NOERROR) {
         usITCBAddr = (USHORT)
            (pritcbPrimary->ITCB_prseg->SEG_prmem->MEM_ulSegAddr+
            pritcbPrimary->ITCB_ulOffset);

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspitcb::ConnectITCB");
         PRINT_DEBUG(EMBEDFILEPTR, "  prtskPrimary (0x%X)", prtskPrimary);
         PRINT_DEBUG(EMBEDFILEPTR, "  pszPrimary \"%s\"\n", pszPrimary);
         PRINT_DEBUG(EMBEDFILEPTR, "  prtskSecondary (0x%X)", prtskSecondary);
         PRINT_DEBUG(EMBEDFILEPTR, "  pszSecondary \"%s\"\n", pszSecondary);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "WRITE23: addr of Primary ITCB in Secondary segment",
              EOF, 1L,
              pritcbSecondary->ITCB_prseg->SEG_prtsk->TSK_prdsp,
              pritcbSecondary->ITCB_prseg->SEG_prmem->MEM_ulSegAddr+
              pritcbSecondary->ITCB_ulOffset, 1L, (PVOID)&usITCBAddr);

         ulRC = DspBIOS_D_Write(pritcbSecondary->ITCB_prseg->SEG_prtsk->
            TSK_prdsp, pritcbSecondary->ITCB_prseg->SEG_prmem->MEM_ulSegAddr+
            pritcbSecondary->ITCB_ulOffset, 1L, (PVOID)&usITCBAddr);

         /*******************************************************************/
         /* If the write was successful:                                    */
         /* Increment the Primary Use Count.                                */
         /* Point the Secondary to  Primary for disc purposes.              */
         /* Add the Secondary to the Primary's Secondary List.              */
         /*******************************************************************/

         if (ulRC == DSP_NOERROR) {
            pritcbPrimary->ITCB_usUseCount += 1;
            pritcbSecondary->ITCB_pritcbPrimary = pritcbPrimary;

            ulRC = CLLInsTail((PVOID)pritcbSecondary, (PVOID *)
               &(pritcbSecondary->ITCB_pritcbSecListNext), (PVOID *)&(
               pritcbPrimary->ITCB_pritcbSecListNext->ITCB_pritcbSecListNext),
               (PVOID *)&(pritcbPrimary->ITCB_pritcbSecListNext));

         }                             /* endif                             */
      }                                /* endif                             */
   }
   else
      ulRC = DSP_ITCB_CONNECTED;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspitcb::ConnectITCB exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DisconnectITCB                                          */
/*                                                                          */
/* FUNCTION: This routines will disconnect two ITCB.                        */
/*           This includes verifying, and if necessary, setting the Primary */
/*           Tasks to Standby mode.                                         */
/*                                                                          */
/* INPUT:                                                                   */
/*     HTASK  hPrimaryTask - Handle of Primary Task for ITCB.               */
/*     PSZ   pszPrimary   - Name of Primary ITCB.                           */
/*     HTASK  hSecondaryTask  - Handle of Secondary Task for ITCB.          */
/*     PSZ   pszSecondary    - Name of Secondary ITCB.                      */
/*     USHORT usInhibit - Indicate whether Freeing of BIOS tasks is allowed */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: Primary and Secondary Tasks set to Standby Mode.           */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC DisconnectITCB(PRTSK prtskPrimary,PSZ pszPrimary,PRTSK prtskSecondary,PSZ
                   pszSecondary,USHORT usInhibit)

{
   RC         ulRC=DSP_NOERROR;        // Error return code
   PRITCB     pritcbPrimary;           // Pointer to owner ITCB
   PRITCB     pritcbSecondary;         // Pointer to Secondary ITCB
   PRITCB     pritcbNode;              // General Pointer to a ITCB Node
   PRITCB     pritcbPrev;              // General Pointer to a ITCB Node
   PRBIOT     prbiot;
   RC         ulRC_user = 0;           // Freed User Task RC
   RC         ulRC_owner = 0;          // Freed Owner Task RC
   USHORT     usNullBuff = 0;

   /************************************************************************/
   /*  CH01 If either prtsk is not NULL it better be valid!  MTS #2675     */
   /************************************************************************/

   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspitcb::DisconnectITCB entry prtskPrimary %x pszPrimary %s prtskSecondary %x pszSecondary %s\n",
	   (int)prtskPrimary,pszPrimary,(int)prtskSecondary,pszSecondary);

   if ((prtskSecondary!=NULL) && (prtskSecondary->TSK_achValid!=TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH02 Chk before using MTS #2675 */

   if ((prtskPrimary!=NULL) && (prtskPrimary->TSK_achValid != TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH02 Chk before using MTS #2675 */

   /*******************************************************
    * Check the pointers for the Tasks for validity.
    * If either of the Tasks comes in set to NULL,
    * this must be a disconnect for a BIOS task.
    * Find the BIOS Task that is used in the disconnect.
    *******************************************************                 */
   /*  If both User and Owner handles are null, ie. BIOS tasks, then return */
   /*  DSP_INV_HANDLE because we don't know which DSP they are loaded on.   */
   /*  This also prevents a trap which occurred when attempting to access a */
   /*  task structure with a null prtskSecondary pointer                    */

   if (prtskPrimary == NULL) {
      if (prtskSecondary == NULL)
         ulRC = DSP_INV_HANDLE;
      else {
         ulRC = ConnectorToBIOS(pszPrimary, pg->prmgrTail, &prbiot);
         if (ulRC == DSP_NOERROR) {
            ulRC = NameToTaskHandle(prtskSecondary->TSK_prdsp->DSP_prmodOS,
                                    prbiot->BT_pszTaskname,
                                    &prtskPrimary);  /* CH02 */
            if (ulRC) 
	      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspitcb::DisconnectITCB error ulRC %lx Bios Task: %s\n",ulRC, prbiot->BT_pszTaskname);
         }
      }
   }

   else if (prtskSecondary == NULL) { /* CH02 Added "else" clause */
      ulRC = ConnectorToBIOS(pszSecondary, pg->prmgrTail, &prbiot);
      if (ulRC == DSP_NOERROR) {
         ulRC = NameToTaskHandle(prtskPrimary->TSK_prdsp->DSP_prmodOS,
                                 prbiot->BT_pszTaskname,
                                 &prtskSecondary); /* CH02 */
            if (ulRC)
	      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspitcb::DisconnectITCB error ulRC %lx Bios Task: %s\n", ulRC,prbiot->BT_pszTaskname);
      }
   }

   if (ulRC != DSP_NOERROR) /* CH02 Combined several tests into this one */
      return (ulRC);

   /*************************************************************************/
   /* Verify that there are Data Segments attached to the                   */
   /* tasks.                                                                */
   /* Verify that there are ITCBs attached to Segments.                     */
   /* The ITCBs are in the Data Segments.                                   */
   /*************************************************************************/

   if ((prtskPrimary->TSK_prsegDTail == NULL) ||
      (prtskSecondary->TSK_prsegDTail == NULL))
      return (DSP_NO_DATA);

   /*************************************************************************/
   /* Find the ITCBs within the Tasks. Search ITCBs until                   */
   /* a match is found or the entire list has been checked.                 */
   /* Start at Tail->Next and go thru Tail.                                 */
   /* If a match is found, bFound will be set to TRUE and                   */
   /* the loop will end with pritcbNode pointing to the                     */
   /* correct ITCB.                                                         */
   /*************************************************************************/
   /*************************************************************************/
   /* Find the Primary ITCB.                                                */
   /*************************************************************************/

   if ((ulRC = FindITCB(prtskPrimary, pszPrimary, &pritcbNode)) != DSP_NOERROR
      )
      return (ulRC);

   if ((pritcbNode->ITCB_usflgKind&ITCB_KINDMASK) == ITCB_PRIMARY)
      pritcbPrimary = pritcbNode;
   else
      return (DSP_INV_ITCB_TYPE);

   if ((ulRC=CheckDSPStatus(prtskPrimary->TSK_prdsp)))
      return (ulRC);  /* CH03 Check for card unplugged */

   /*************************************************************************/
   /* Find the Secondary ITCB.                                              */
   /*************************************************************************/

   if ((ulRC = FindITCB(prtskSecondary, pszSecondary, &pritcbNode)) !=
      DSP_NOERROR)
      return (ulRC);

   if ((pritcbNode->ITCB_usflgKind&ITCB_KINDMASK) == ITCB_SECONDARY)
      pritcbSecondary = pritcbNode;
   else
      return (DSP_INV_ITCB_TYPE);

   /*************************************************************************/
   /* Set the Secondary ITCB buffer pointer to NULL                         */
   /* Verify that Secondary and the Primary are connected.                  */
   /* buffer.  Remove Secondary from Primarys SecondaryList                 */
   /* and decrement the UseCount.                                           */
   /*************************************************************************/

   pritcbSecondary->ITCB_prmemVariable = NULL;

   if (pritcbSecondary->ITCB_pritcbPrimary == pritcbPrimary) {
      pritcbSecondary->ITCB_pritcbPrimary = NULL;

    /* Find the Previous so you can delete the Secondary from ...SecondaryList*/

      pritcbPrev = pritcbPrimary->ITCB_pritcbSecListNext;/* Tail of List    */
      pritcbNode = pritcbPrev->ITCB_pritcbSecListNext;/* Current Node       */
      while (pritcbNode != pritcbSecondary) {
         pritcbPrev = pritcbNode;
         pritcbNode = pritcbNode->ITCB_pritcbSecListNext;
      }                                /* endwhile                          */
      CLLDelNode((PVOID *)&(pritcbSecondary->ITCB_pritcbSecListNext), (PVOID
         *)&(pritcbPrev->ITCB_pritcbSecListNext), (PVOID)pritcbPrev, (PVOID *)
         &(pritcbPrimary->ITCB_pritcbSecListNext));

      /**********************************************************************/
      /* Write a NULL to the Secondary's ITCB Pointer.                      */
      /**********************************************************************/

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspitcb::ConnectITCB");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtskPrimary (0x%X)", prtskPrimary);
      PRINT_DEBUG(EMBEDFILEPTR, "  pszPrimary \"%s\"\n", pszPrimary);
      PRINT_DEBUG(EMBEDFILEPTR, "  prtskSecondary (0x%X)", prtskSecondary);
      PRINT_DEBUG(EMBEDFILEPTR, "  pszSecondary \"%s\"", pszSecondary);
      PRINT_DEBUG(EMBEDFILEPTR, "  usInhibit (0x%X)\n", usInhibit);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "WRITE24:  NULL to the Secondary's ITCB Pointer", EOF, 1L,
           pritcbSecondary->ITCB_prseg->SEG_prtsk->TSK_prdsp,
           pritcbSecondary->ITCB_prseg->SEG_prmem->MEM_ulSegAddr+
           pritcbSecondary->ITCB_ulOffset, 1L, (PVOID)&usNullBuff);

      ulRC = DspBIOS_D_Write(pritcbSecondary->ITCB_prseg->SEG_prtsk->TSK_prdsp
         , pritcbSecondary->ITCB_prseg->SEG_prmem->MEM_ulSegAddr+
         pritcbSecondary->ITCB_ulOffset, 1L, (PVOID)&usNullBuff);

      pritcbPrimary->ITCB_usUseCount -= 1;
   }
   else
      return (DSP_ITCB_NOTCONNECTED);

   /********************************************************                */
   /* Call Check Connections to check the connections                       */
   /* for the BIOS tasks.                                                   */
   /* If no connections are left, Free the BIOS Task.                       */
   /********************************************************                */

   if (((prtskPrimary->TSK_usflgType&TSK_BIOSMASK) == TSK_BIOS) && ((usInhibit
      &DONT_FREE_OWNER_BIOS) == 0))
      {
      ulRC = CheckConnections(prtskPrimary);
      if (ulRC == DSP_NO_CONNECTIONS) {
         ulRC = FreeTask(prtskPrimary);
         if (ulRC == DSP_NOERROR)
            ulRC_owner = DSP_OWNER_TASK_FREE;
         else
            return (ulRC);
      }                                /* endif                             */
   }
   if (((prtskSecondary->TSK_usflgType&TSK_BIOSMASK) == TSK_BIOS) &&
      ((usInhibit&DONT_FREE_USER_BIOS) == 0))
      {
      ulRC = CheckConnections(prtskSecondary);
      if (ulRC == DSP_NO_CONNECTIONS) {
         ulRC = FreeTask(prtskSecondary);
         if (ulRC == DSP_NOERROR)
            ulRC_user = DSP_USER_TASK_FREE;
         else
            return (ulRC);
      }                                /* endif                             */
   }

   /******************************************************************      */
   /* removed the bitwise exclusive OR of ulRC_user with ulRC_owmer         */
   /* which was causing an erroneous ulRC of 3, "DSP_FILE_NOT_FOUND"        */

   if (ulRC == DSP_NOERROR) {
      if (ulRC_user && ulRC_owner)
         ulRC = DSP_OW_US_TASK_FREE;
      else
         if (ulRC_user)

            ulRC = DSP_USER_TASK_FREE;
         else
            if (ulRC_owner)

               ulRC = DSP_OWNER_TASK_FREE;
   }


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspitcb::DisconnectITCB exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FindITCB                                                */
/*                                                                          */
/* FUNCTION: This routines will take a task and ITCB name and return the    */
/*           ITCB.                                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK prtsk      - Task                                           */
/*        PSZ   pszitcb     - Name of ITCB                                  */
/*        PRITCB *ppritcb    - ITCB node if found                           */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error or not found.                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FindITCB(PRTSK prtsk,PSZ pszitcb,PRITCB *ppritcb)
{
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   PRITCB     pritcb=0;
   BOOL       bFound = FALSE;          /* Assume FALSE                      */

   /*********************************************************               */
   /* Find the ITCB from the tasks list                                     */
   /*********************************************************               */
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspitcb::FindITCB entry prtsk %x pszitcb %s\n",(int)prtsk,pszitcb);
   if (prtsk->TSK_pritcbTail != NULL) {

      pritcb = prtsk->TSK_pritcbTail;
      do {
         pritcb = pritcb->ITCB_pritcbNext;

         if (mwstrcmpnull(pszitcb, pritcb->ITCB_pszName) == 0)
            bFound = TRUE;

      }  while ((bFound == FALSE) && (pritcb != prtsk->TSK_pritcbTail));
                                       /* enddo                             */
   }                                   /* endif                             */
   if (bFound == TRUE)
      *ppritcb = pritcb;
   else {
      ulRC = DSP_NAME_NOT_FOUND;
      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspitcb::FindITCB error ulRC %lx ITCB: %s\n", ulRC, pszitcb); 
   }

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspitcb::FindITCB exit ulRC %lx\n",ulRC);
   return (ulRC);
}
