/* ============================================================
 * File  : streamconfig.cpp
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2004-03-08
 * Description : mythtv compatibility parts copied from mythnews
 *               0.13 plugin source, mythnews copyright Renchi Raju
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#include <iostream>
#include <qlayout.h>
#include <qcursor.h>
#include <qregexp.h>
#include <qtextedit.h>
#include <qtextstream.h>

#include "streamconfig.h"


using namespace std;
/*

  Needs cleanup

  StreamTunedConfig: main config window with "stream" and "storage" tabs
    StreamConfig: stream configuration handler (stream tab)
      StorageGroup: StreamConfig goupBox with storage controls
      StreamEditGroup: StreamConfig groupBox with stream property edit boxes
   (StorageConfig: storage configuration handler (storage tab) see storageconfig.h)

*/


// ------------------- Storage Select group box --------------------

StorageGroup::StorageGroup(char* title, QWidget* parent, ReposStorage *storageStorage,
                           StreamStorage *streamStorage) : QVGroupBox(title, parent)
{
    myParent = parent;
    storages = storageStorage;
    streams  = streamStorage;

    if ( !streams )
    {
       cerr << "stream storage not initialized" << endl;
       exit(-1);
    }

    if ( !storages )
    {
       cerr << "repository storage not initialized" << endl;
       exit(-1);
    }

    activeStorage = streams->getStorageName();

    rollback = false;   // select and load previous storage after storage select fail
    rollbackReason = "";

    setPalette(parent->palette());
    setFont(parent->font());
    setBackgroundOrigin(QWidget::WindowOrigin);

    buildGroup();

    connect (
               streams,
               SIGNAL( storageEvent(int, int, bool ) ),
               this,
               SLOT  ( slotStreamStorageEvent(int, int, bool ) )
             );

    connect (
               storages,
               SIGNAL( storageEvent(int, int, bool ) ),
               this,
               SLOT  ( slotStorageEvent(int, int, bool ) )
             );

    connect (
               storages,
             SIGNAL(recordInserted(ChangedRecord*)),
             this,
             SLOT(slotRecordInserted(ChangedRecord*))
            );

    connect (
               storages,
             SIGNAL(recordUpdated(ChangedRecord*)),
             this,
             SLOT(slotRecordUpdated(ChangedRecord*))
            );

    connect (
               storages,
             SIGNAL(recordRemoved(ChangedRecord*)),
             this,
             SLOT(slotRecordRemoved(ChangedRecord*))
            );

    // set storage handling group fields
    if ( streams->getAccessType() != StreamStorage::none )
    {
      for (int i = 0; i < storageBox->count(); i++)
        if (storageBox->text(i) == streams->getStorageName() )
            storageBox->setCurrentItem(i);

      if ( streams->getSynchronized() )
      {
         synchronized(true);
         loadedStorageLabel->setText( "loaded: " + streams->getStorageName() );
      }
    }

}

void StorageGroup::fillStorageBox()
{
    ValueList values;

    storages->resetRecordList();
    storageBox->clear();

    while ( storages->getNextRecord(values) )
    {
       storageBox->insertItem( values[r_name] );
    }

    storageBox->insertItem( streams->getDefaultName() );

}

void StorageGroup::buildGroup()
{
    QFontMetrics metric( font() );
    int maxHeight = metric.height() + 2;

    setTitle( " storage handling " );

    syncLabel = new QLabel(this);
    syncLabel->setBackgroundOrigin(QWidget::WindowOrigin);
    syncLabel->setPaletteForegroundColor( black );
    syncLabel->setMaximumHeight(maxHeight);
    syncLabel->setText("connected storage:");

    storageBox = new QComboBox(this);
    storageBox->setMaximumHeight(maxHeight);

    fillStorageBox();

    loadedStorageLabel = new QLabel(this);
    loadedStorageLabel->setBackgroundOrigin(QWidget::WindowOrigin);
    loadedStorageLabel->setFrameStyle(QFrame::Box + QFrame::Sunken);
    loadedStorageLabel->setMaximumHeight(maxHeight);

    loadButton      = new ConfigButton("&Load storage", this);
    overwriteButton = new ConfigButton("&Overwrite storage", this);

    connect(storageBox,      SIGNAL(activated(const QString&) ), this,
                             SLOT(slotBoxActivated(const QString&)) );
    connect(loadButton,      SIGNAL(clicked()), this, SLOT(slotLoadClicked()));
    connect(overwriteButton, SIGNAL(clicked()), this, SLOT(slotOverwriteClicked()));

}

StorageGroup::~StorageGroup()
{

}

void StorageGroup::synchronized(bool sync)
{
  if (!sync)
  {
    loadButton->setEnabled(true);
    overwriteButton->setEnabled(true);
    syncLabel->setPaletteForegroundColor( blue );
    syncLabel->setText("connected storage not synchronized");
  }
    else
  {
    loadButton->setEnabled(false);
    overwriteButton->setEnabled(false);
    syncLabel->setPaletteForegroundColor( black );
    syncLabel->setText("connected storage:");
  }
}

void StorageGroup::reportMessage( const QString& message, bool bad)
{
   StreamConfig *streamConfig = dynamic_cast<StreamConfig*>(myParent);

   if (streamConfig)
     streamConfig->reportMessage(message, bad);
}


void StorageGroup::slotBoxActivated(const QString& selectedText)
{
    if ( selectedText == streams->getStorageName() ) return;

    ValueList values(6);
    QString error;

    // select storage if found
    if (  storages->getStorageValuesByName(values, selectedText) )
    {
       //streams = new StreamStorage();
       //streams->selectStorage(ident_storagegroup, values);
       storages->openStorage(streams, ident_storagegroup, values, error);

    }
       else 
    {
      reportMessage(error, true);
      if (streams->getDefaultName() == selectedText)
              streams->selectDefaultDb(ident_storagegroup);
    }
}

void StorageGroup::slotLoadClicked()
{
   QString error;
   if ( !streams->loadList(ident_storagegroup, error) )
     reportMessage(error, true);
}

void StorageGroup::slotOverwriteClicked()
{
   QString error;
   if ( !streams->storeList(ident_storagegroup, error) )
     reportMessage(error, true);
}

void StorageGroup::slotStorageEvent(int ident, int eventType, bool error)
{
  if ( error )
  {
    if ( ident == ident_storagegroup )
      reportMessage(storages->getLastError(), true);
    return;
  }

  switch (eventType)
  {
    case ReposStorage::selected:
    break;
    case ReposStorage::loaded:
         fillStorageBox();
    break;
  }
}

void StorageGroup::slotRecordInserted(ChangedRecord *rec)
{
  if ( rec->error ) return;

  storageBox->insertItem( rec->values[r_name] );
}

void StorageGroup::slotRecordUpdated(ChangedRecord *rec)
{
  if ( rec->error ) return;

  for (int i = 0; i < storageBox->count(); i++)
    if (storageBox->text(i) == streams->getStorageName() )
    {
        storageBox->setCurrentText( rec->values[r_name] );
    }
}

void StorageGroup::slotRecordRemoved(ChangedRecord *rec)
{
  if ( rec->error ) return;

  bool found = false;
  fillStorageBox();

  for (int i = 0; i < storageBox->count(); i++)
    if (storageBox->text(i) == streams->getStorageName() )
    {
      storageBox->setCurrentItem(i);
      found = true;
    }

  if ( !found && storageBox->count() > 0 )
    slotBoxActivated( storageBox->text(0) );

  // last storage deleted
}

void StorageGroup::slotStreamStorageEvent(int ident, int eventType, bool error)
{

  if ( error )
  {
    if (ident == ident_storagegroup)
    {
      reportMessage( streams->getLastError(), true );
      //reload previous storage
      if (!rollback)
      {
         rollback = true;
         rollbackReason = streams->getLastError();
         slotBoxActivated(activeStorage);
      }
         else rollback = false; // prevent loop

    }
    return;
  }

  ValueList values;
  ValueList oldStorageValues;
  ValueList newValues;
  QString err;

  bool oldStorage, newStorage;

  switch (eventType)
  {
    case StreamStorage::selected:
      //cout << "storage selected" << endl;
      // update connected and disconnected storages
      oldStorage = storages->getStorageValuesByName(oldStorageValues, activeStorage);
      newStorage = storages->getStorageValuesByName( values, streams->getStorageName() );

      if ( activeStorage != streams->getStorageName() )
      if ( oldStorage )
      {
          //cout << "found old: " << activeStorage << endl;

          newValues = oldStorageValues;
          newValues[r_selected] = "-";

          // condition should not fail
          if ( oldStorageValues[r_selected] != "-" )
            if ( !storages->updateRecord(ident_storagegroup, oldStorageValues, newValues, err) )
              reportMessage(err, true);
      }
        else
          cerr << "warning: unknown previous connected storage" << endl;

      
      if ( activeStorage != streams->getStorageName() )
      if ( newStorage )
      {
          // cout << "found new: " << streams->getStorageName() << endl;

          newValues = values;
          newValues[r_selected] = "*";
          if ( values[r_selected] != "*" )
            if ( !storages->updateRecord(ident_storagegroup, values, newValues, err) )
              reportMessage(err, true);
      }
        else
          cerr << "error: unknown previous connected storage" << endl;

      synchronized(false);

      // handle external storage change / reload after error
      for (int i = 0; i < storageBox->count(); i++)
        if (storageBox->text(i) == streams->getStorageName() )
            storageBox->setCurrentItem(i);

      activeStorage = storageBox->currentText();

      // auto load storage values on rollback
      if (rollback)
        slotLoadClicked();

    break;

    case StreamStorage::loaded:

       synchronized(true);
       loadedStorageLabel->setText( "loaded: " + streams->getStorageName() );
       if ( streams->getStorageName() != activeStorage )
         cerr << "error: activeStorage value not equal to storage name" << endl;

       if (rollback)
       {
         // overwrite "OK" in error field loaded during succesful rollback
         reportMessage( rollbackReason, true );
         rollback = false;
       }

    break;

  }
}

// ------------------- Stream Edit group box --------------------

class StreamEditGroup : public EditGroup
{

  public:
    StreamEditGroup(char* title, StreamConfig* parent) : EditGroup(title, parent)
    {
        boxAddTitle  = "Add stream";
        boxEditTitle = "Edit stream";
        boxIdleTitle = "Add stream";

        streamConfig = parent;
        buildGroup(5);

        QLabel *label;
        label = getLabel("label0");
        if (label) label->setText("Stream Folder");
        label = getLabel("label1");
        if (label) label->setText("Stream Name");
        label = getLabel("label2");
        if (label) label->setText("Stream Url");
        label = getLabel("label3");
        if (label) label->setText("Stream Description");
        label = getLabel("label4");
        if (label) label->setText("Stream Handler");
    }

    void initFields(QListViewItem* item)
    {
      StationItem* station = dynamic_cast<StationItem*>(item);
      if ( station )
      {
        MythLineEdit *editbox;
        editbox = getLineEdit("edit0");
        if (editbox) editbox->setText(station->folder);
        editbox = getLineEdit("edit1");
        if (editbox) editbox->setText(station->name);
        editbox = getLineEdit("edit2");
        if (editbox) editbox->setText(station->url);
        editbox = getLineEdit("edit3");
        if (editbox) editbox->setText(station->descr);
        editbox = getLineEdit("edit4");
        if (editbox) editbox->setText(station->handler);

        linkedItem = item;
        setStatus(edit);
      }

      FolderItem* folderItem = dynamic_cast<FolderItem*>(item);
      if (folderItem)
      {
        MythLineEdit *editbox;
        editbox = getLineEdit("edit0");
        if (editbox) editbox->setText(folderItem->folder);
        editbox = getLineEdit("edit1");
        if (editbox) editbox->setText("");
        editbox = getLineEdit("edit2");
        if (editbox) editbox->setText("");
        editbox = getLineEdit("edit3");
        if (editbox) editbox->setText("");
        editbox = getLineEdit("edit4");
        if (editbox) editbox->setText("");

        setStatus(add);
      }

      if (!folderItem && !station)
      {
        setStatus(idle);
      }
    }

    void commitFields()
    {
      MythLineEdit *edit1, *edit2, *edit3, *edit4, *edit5;
      edit1 = getLineEdit("edit0");
      edit2 = getLineEdit("edit1");
      edit3 = getLineEdit("edit2");
      edit4 = getLineEdit("edit3");
      edit5 = getLineEdit("edit4");

      if ( status == add | status == idle )
      {
        streamConfig->createStation(edit1->text(), edit2->text(),
                                edit3->text(), edit4->text(), edit5->text());
      }
      else if ( status == edit && dynamic_cast<StationItem*>(linkedItem))
      {
        streamConfig->updateStation(dynamic_cast<StationItem*>(linkedItem),
                                edit1->text(), edit2->text(), edit3->text(), edit4->text(), edit5->text() );
      }
    }

    void removeLinked()
    {
      if (status == edit && dynamic_cast<StationItem*>(linkedItem))
        streamConfig->removeStation(dynamic_cast<StationItem*>(linkedItem));
    }

    private:
      StreamConfig *streamConfig;

};

Readme::Readme(BaseParent *parent, const char *name) : BaseWidget(parent, name)
{
  QVBoxLayout *vbox = new QVBoxLayout(this, (int)(10));

  QFont f("monospace");
  f.setPointSize( font().pointSize() );

  QTextEdit *textEdit = new QTextEdit(this, name);
  textEdit->setFont(f);
  textEdit->setReadOnly(true);
  textEdit->setBackgroundOrigin(QWidget::WindowOrigin);
  textEdit->setFrameStyle(QFrame::Box + QFrame::Raised);
  vbox->addWidget(textEdit);

  QFile file( PREFIX"/share/"SUBPATH"/README" );
  if ( file.open( IO_ReadOnly ) )
  {
      QTextStream stream( &file );
      textEdit->setText( stream.read() );
  }
  file.close();
}

/*
StreamSettings::StreamSettings(BaseParent *parent, const char *name) : BaseWidget(parent, name)
{
  QVBoxLayout *vbox = new QVBoxLayout(this, (int)(10));

  QFont f("monospace");
  f.setPointSize( font().pointSize() );

  MythLineEdit *editBox;
  editBox = new MythLineEdit(this, "download directory");
  editBox->setWidth(300);
  editBox->setFont(f);
  editBox->setBackgroundOrigin(QWidget::WindowOrigin);
  editBox->setFrameStyle(QFrame::Box + QFrame::Raised);

}

*/

//--------------------- Tab widget------------------------------

StreamTunedConfig::StreamTunedConfig(QWidget* /*parent*/, const char *name,
                                     StreamStorage *storage, ReposStorage *storageBin)
                 : QTabWidget(0, name)
{

   setCaption(name);
   resize( 630, 450 );
   setMargin(2);

   streamConfig   = new StreamConfig ((BaseParent*)this, name, storage, storageBin);
   storageConfig  = new StorageConfig((BaseParent*)this, name, storage, storageBin);
   //recordConfig   = new RecordConfig((BaseParent*)this, name);
   Readme *readme = new Readme((BaseParent*)this, name);
   //StreamSettings *settings = new StreamSettings((BaseParent*)this, name);

   addTab(streamConfig , "streams");
   addTab(storageConfig, "storage");
   //addTab(recordConfig, "recording");
   addTab(readme, "readme");
   //addTab(settings, "settings");
   setCaption(name);
}

StreamTunedConfig::~StreamTunedConfig() {}

// --------------------- Main class ------------------------------

StreamConfig::StreamConfig(BaseParent *parent, const char *name,
                           StreamStorage *storage, ReposStorage *storageBin)
            : BaseWidget(parent, name)
{

    streamStorage  = storage;
    storageStorage = storageBin;

    connect (
              streamStorage,
              SIGNAL(recordInserted(ChangedRecord*)),
              this,
              SLOT(slotRecordInserted(ChangedRecord*))
            );

    connect (
              streamStorage,
              SIGNAL(recordUpdated(ChangedRecord*)),
              this,
              SLOT(slotRecordUpdated(ChangedRecord*))
            );

    connect (
              streamStorage,
              SIGNAL(recordRemoved(ChangedRecord*)),
              this,
              SLOT(slotRecordRemoved(ChangedRecord*))
            );

    connect (
              streamStorage,
              SIGNAL( storageEvent(int, int, bool ) ),
              this,
              SLOT  ( slotStorageEvent(int, int, bool ) )
            );

    setupView();

    if ( !getStationsFromDB() )
      reportMessage("cannot read selected stream storage", true);

    setCursor(QCursor(Qt::ArrowCursor));

    // didn't need this w. mythtv 0.13, check...
    QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));

    show();
}

StreamConfig::~StreamConfig()
{
    QApplication::restoreOverrideCursor();
}

void StreamConfig::reportMessage(const QString& error, bool bad)
{
  errorLabel->setText(error);
  if ( bad )
    errorLabel->setPaletteForegroundColor( red );
  else
    errorLabel->setPaletteForegroundColor( black );
}

void StreamConfig::loadDefaultStations() {}

bool StreamConfig::getStationsFromDB()
{
    // Fill station picklist
    QString folder;
    QString name;
    QString url;
    QString descr;
    QString handler;

    streamStorage->resetRecordList();
    m_leftView->clear();

    ValueList values(5);

    while ( streamStorage->getNextRecord(values) )
    {
      if (values.count() == 5)
        initStation(values[0], values[1], values[2], values[3], values[4]);
    }

    return true;
}


void StreamConfig::setupView()
{
    QFontMetrics metric( font() );
    int maxHeight = metric.height() + 2;

    int hmult = 1;
    QVBoxLayout *vbox = new QVBoxLayout(this, (int)(hmult*10));

    errorLabel = new QLabel(this);
    errorLabel->setBackgroundOrigin(QWidget::WindowOrigin);
    errorLabel->setFrameStyle(QFrame::Box + QFrame::Raised);
    //errorLabel->setMargin(int(hmult*4));
    errorLabel->setText(" ");
    errorLabel->setMaximumHeight(maxHeight);
    vbox->addWidget(errorLabel);
    // List Views

    QHBoxLayout *hbox = new QHBoxLayout(vbox);

    m_leftView = new QListView(this);
    //m_leftView->header()->hide();
    m_leftView->addColumn("Stations");
    m_leftView->setRootIsDecorated(true);
    hbox->addWidget(m_leftView, 1);

    //QVBoxLayout *detailvbox = new QVBoxLayout(hbox, 5);
    QVBoxLayout *detailvbox = new QVBoxLayout();
    hbox->addLayout(detailvbox, 1);
// Edit boxes

    streamEditGroup = new StreamEditGroup("Edit webcast", this);
    detailvbox->addWidget(streamEditGroup);

    storageGroup = new StorageGroup("Storage handling", this, storageStorage, streamStorage);
    detailvbox->addWidget(storageGroup);

    connect(m_leftView, SIGNAL(selectionChanged(QListViewItem*)),
            this, SLOT(itemSelected(QListViewItem*)));
}

//----------------------- item manipulation helpers ---------------------------

void StreamConfig::assignFolder(StationItem* station, const QString& folder)
{
    FolderItem* newFolder;
    FolderItem* oldFolder;

    oldFolder = dynamic_cast<FolderItem*>(station->parent());
    oldFolder->takeItem(station);

    QListViewItem* item = m_leftView->findItem(folder, 0, Qt::ExactMatch);
    if (!item)
    {
      newFolder = new FolderItem(m_leftView, folder);
      newFolder->setOpen(true);
    }
      else newFolder = dynamic_cast<FolderItem*>(item);

    newFolder->insertItem(station);
    if (oldFolder->childCount() == 0)
      delete(oldFolder);
}

FolderItem* StreamConfig::getFolderItem(QString folder)
{
    QListViewItem* item = m_leftView->firstChild();

    while ( item && item->text(0) != folder )
      item = item->nextSibling();

    return dynamic_cast<FolderItem*>(item);
}

StationItem* StreamConfig::getStationItem(FolderItem* folder, QString station)
{
    if (!folder) return 0;

    QListViewItem* item = folder->firstChild();

    while ( item && item->text(0) != station )
      item = item->nextSibling();

    return dynamic_cast<StationItem*>(item);
}

FolderItem* StreamConfig::forceFolder(const QString& folder)
{
    FolderItem* newFolder = getFolderItem(folder);

    if (!newFolder)
    {
      newFolder = new FolderItem(m_leftView, folder);
      newFolder->setOpen(true);
    }
    return newFolder;
}


//----------------------- item manipulation functions ---------------------------

void StreamConfig::updateStation(StationItem* station, const QString& folder, const QString& name,
                                    const QString& url, const QString& descr, const QString& handler)
{
  QString error;
    ValueList values(5);
    values[0] = folder;
    values[1] = name;
    values[2] = url;
    values[3] = descr;
    values[4] = handler;

    ValueList oldValues(5);
    oldValues[0] = station->folder;
    oldValues[1] = station->name;
    oldValues[2] = station->url;
    oldValues[3] = station->descr;
    oldValues[4] = station->handler;

  if ( station &&
       !streamStorage->updateRecord(ident_streamconfig, oldValues, values, error)
     )
       reportMessage(error, true);
}

StationItem* StreamConfig::initStation(const QString& folder, const QString& name,
                                         const QString& url, const QString& descr, const QString& handler)
{

  //if (folder == "" || name == "" || url == "")
  //  return 0;

  FolderItem* myFolder = forceFolder(folder);
  StationItem* station = new StationItem(myFolder, name);

  station->folder = folder;
  station->name   = name;
  station->url    = url;
  station->descr  = descr;
  station->handler  = handler;

  m_leftView->setSelected(station, true);

  return station;
}

void StreamConfig::createStation(const QString& folder, const QString& name,
                                    const QString& url, const QString& descr, const QString& handler)
{
  QString error;

  ValueList values(5);
  values[0] = folder;
  values[1] = name;
  values[2] = url;
  values[3] = descr;
  values[4] = handler;
  if ( !streamStorage->insertRecord(ident_streamconfig, values, error) )
    reportMessage(error, true);
}


void StreamConfig::removeStation(StationItem* station)
{
  // if not in storage: fine.
    QString error;

    ValueList values(5);
    values[0] = station->folder;
    values[1] = station->name;
    values[2] = station->url;
    values[3] = station->descr;
    values[4] = station->handler;

    if ( station &&
         !streamStorage->removeRecord(ident_streamconfig, values, error) )
      reportMessage( error, true );
}



//----------------------- storage callback ---------------------------

void StreamConfig::slotStorageEvent(int /*ident*/, int eventType, bool error)
{
  QString errorMsg;

  if (!error && eventType == ReposStorage::loaded )
    getStationsFromDB();
}

void StreamConfig::slotRecordInserted(ChangedRecord *rec)
{
    StationItem* station;

    if (rec->error)
    {
       if ( rec->ident == ident_streamconfig )
         reportMessage( streamStorage->getLastError(), true );
       return;
    }

    station = initStation(rec->values[0], rec->values[1], rec->values[2], rec->values[3], rec->values[4]);

    m_leftView->setSelected(station, true);
}

void StreamConfig::slotRecordUpdated(ChangedRecord *rec)
{
    StationItem* station = 0;

    if (rec->error)
    {
      if ( rec->ident == ident_streamconfig )
        reportMessage( streamStorage->getLastError(), true );
      return;
    }

    FolderItem* folder = getFolderItem(rec->oldValues[0]);
    if (folder)
      station = getStationItem(folder, rec->oldValues[1]);

    if (!station)
    {
      cerr << "cannot find folder,name: " << rec->oldValues[0]
           << "" << rec->oldValues[1] << endl;
      return;
    }

    if (station->folder != rec->values[0])
      assignFolder(station, rec->values[0]);

    station->folder = rec->values[0];
    station->name   = rec->values[1];
    station->url    = rec->values[2];
    station->descr  = rec->values[3];
    station->handler = rec->values[4];
    station->setText(0, rec->values[1]);

    m_leftView->setSelected(station, true);
}

void StreamConfig::slotRecordRemoved(ChangedRecord *rec)
{
    StationItem* station = 0;

    if (rec->error)
    {
      if ( rec->ident == ident_streamconfig )
        reportMessage( streamStorage->getLastError(), true );
      return;
    }

    FolderItem* folder = getFolderItem(rec->values[0]);
    if (folder)
      station = getStationItem(folder, rec->values[1]);

    if (!station)
    {
      cerr << "cannot find folder,name: " << rec->values[0]
           << "" << rec->values[1] << endl;
      return;
    }

    if (folder->childCount() == 1)
        delete(folder);
      else
        delete(station);

      streamEditGroup->initFields(0);
}


//----------------------- GUI Callback ---------------------------

void StreamConfig::itemSelected(QListViewItem *item)
{
    reportMessage("OK", false);
    if (!item) return;
    streamEditGroup->initFields(item);
}



