/* ============================================================
 * File        : streamstorage.cpp
 * Author      : Eric Giesselbach <ericgies@kabelfoon.nl>
 * Version     : v0.17_2a
 * Description : stream repository access via db, file or web
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <iostream>

#include <qnetwork.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qdict.h>

#include <qapplication.h>
#include <qurl.h>
#include <qptrlist.h>
#include <qregexp.h>

#include "streamstorage.h"
#include "configelements.h"

#include <sys/stat.h>

using namespace std;


bool ReposStorage::filesDiffer(const QString& source, const QString& dest)
{
   char sdata[1024];
   char ddata[1024];
   int i;
   long nread = 0;
   
   QFile src( source );
   QFile dst( dest );

   if (src.size() != dst.size() )
     return false;
     
   if ( !src.open(IO_ReadOnly) )
     return false;
   if ( !dst.open(IO_ReadOnly) )
   {
     src.close();
     return false;
   }       

   src.open( IO_ReadOnly );
   dst.open( IO_ReadOnly );

   bool ohNoTheyDiffer = false;
   
   while ( ! src.atEnd() && nread > -1 )
   {
     nread = src.readBlock( sdata, sizeof(sdata) );
     dst.readBlock( ddata, nread );
     for (i=0; i<nread; i++)
       ohNoTheyDiffer = ohNoTheyDiffer || ( sdata[i] != ddata[i] );
   }

   dst.close();
   src.close();
   
   return ohNoTheyDiffer;
}


bool ReposStorage::copyFile(const QString& source, const QString& dest)
{
   char data[1024];
   long nread = 0;
   
   QFile src( source );
   QFile dst( dest );

   if ( !src.open(IO_ReadOnly) )
     return false;
   if ( !dst.open(IO_ReadWrite) )
   {
     src.close();
     return false;
   }       

   while ( ! src.atEnd() && nread > -1 )
   {
     nread = src.readBlock( data, sizeof(data) );
     if (nread > 0)
       dst.writeBlock( data, nread );
   }
   
   dst.close();
   src.close();

   struct stat buffer;
   struct stat *buf;
   buf = &buffer;
   
   if ( stat(source, buf) ==0 )
     chmod(dest, buf->st_mode);
     //chmod(dest, S_IRWXU | S_IRGRP | S_IROTH);

   return true;
}


bool ReposStorage::copyFiles(const QString& sourceDir, const QString& destDir)
{
    QDir source = sourceDir;
    QDir newDir;
    
    source.setFilter( QDir::Files | QDir::NoSymLinks | QDir::Dirs);
 
    const QFileInfoList *list = source.entryInfoList();
    QFileInfoListIterator it( *list );
    QFileInfo *fi;
    
    while ( (fi = it.current()) != 0 ) 
    {
        if ( fi->isDir() )
        {
          if ( fi->fileName() != "." && fi->fileName() != ".." )
          {
            newDir = destDir + "/" + fi->fileName();
            if ( !newDir.exists() )
              newDir.mkdir( destDir + "/" + fi->fileName(), true);
            
            copyFiles(fi->filePath(), destDir + "/" + fi->fileName() );
          }
        }
        else
          if ( fi->fileName() != "list.xml" ) // shouldn't be there
            copyFile(sourceDir + "/" + fi->fileName(), destDir + "/" + fi->fileName() );
        ++it;
    }
   
    return true;
}


bool ReposStorage::createStorageRepos(const QString& dest, const QString& path)
{
  QFile dst(dest);
  
  if ( !dst.open(IO_ReadWrite) )
    return false;

  QTextStream dstStream( &dst );

  dstStream << endl;
  dstStream << "[item]" << endl;
  if (QString(TARGET) == "mythstream") // todo: cleanup mythstream init mess
    dstStream << "-" << endl;
  else
    dstStream << "*" << endl;
  dstStream << "file" << endl;
  dstStream << "default" << endl;
  dstStream << path << "/."SUBPATH"/streams.res" << endl;
  dstStream << endl;
  
  dstStream << "[item]" << endl;
  dstStream << "-" << endl;
  dstStream << "web" << endl;
  dstStream << "Ross Campbell's streams" << endl;
  dstStream << "http://home.kabelfoon.nl/~moongies/harvester/streams_ross.txt" << endl;
  dstStream << "[emptystring]" << endl;
  dstStream << "[emptystring]" << endl;
  dstStream << endl;

  dst.close();

  return true;
}

bool ReposStorage::checkParsers()
{
    QString home = getenv("HOME");
    QDir dir( home + "/."SUBPATH"/parsers" );

    if ( !dir.exists() )
    {
      // create default path/file in homedir
      dir.mkdir(dir.homeDirPath() + "/."SUBPATH"/parsers", true);
    }
     
    // parser source and destination dirs
    QString source = QString(PREFIX"/share/"SUBPATH"/parsers");
    QString dest   = home + "/."SUBPATH"/parsers";

    // only update once, use file parserupdateVERSION_done as flag
    if ( !QFile::exists(home + "/."SUBPATH"/parserupdate"VERSION"_done") ) // parser update pending
    {
      if (!copyFiles(source, dest) )
      {
          cerr << "error: cannot copy files from" << source << " to " << dest << endl;
          exit(-1);
      }

      QFile checkFile(home + "/."SUBPATH"/parserupdate"VERSION"_done");
      if ( !checkFile.open(IO_ReadWrite) )
        cerr << "MythStream error: cannot create file " << home << "/."SUBPATH"/parserupdate"VERSION"_done" << endl;
      else
      {
        QString txt = "this file stops mythstream version "VERSION" from updating the ./parsers directory\n";
        checkFile.writeBlock( txt, txt.length() );
        checkFile.close();
      } 
    }

   // check download dir:
    dir = QDir( home + "/."SUBPATH"/downloads" );

    if ( !dir.exists() )
      // create default path/file in homedir
      dir.mkdir(dir.homeDirPath() + "/."SUBPATH"/downloads", true);

   // check cache dir:
    dir = QDir( home + "/."SUBPATH"/cache" );

    if ( !dir.exists() )
      // create default path/file in homedir
      dir.mkdir(dir.homeDirPath() + "/."SUBPATH"/cache", true);
     
    return true;
}


bool StreamStorage::selectStorage(int ident, ValueList &values)
{
    bool loaded = false;

    if (values[1] == "database")
    {
        bool converted;
        int port = values[4].toInt(&converted, 10); // already checked in getDefaultRepository
        loaded = selectDbStorage( ident, values[2], "QMYSQL3", values[3], port,
                                  values[5], values[6], values[7], values[8] );
    }

    if ( values[1] == "file" )
        loaded = selectFileStorage( ident, values[2], QString(values[3]) );

    if ( values[1] == "web" )
    {
        selectWebStorage ( ident, values[2], values[3], values[4], values[5] );
        loaded = true;
    }

    return loaded;
}

bool ReposStorage::openStorage(StreamStorage *streams, int ident, ValueList& values, QString& errorMessage)
{
    int port;
    bool converted;
        
    if ( values[r_type] == "database" )
    {
        if ( values.count() < r_count_db )
        {
          cerr << "Invalid data. Database storage item has "
                << values.count() << " out of " << r_count_db
                << " properties." << endl;
          errorMessage = "Invalid database reference"; 
          return false;
        }
        port = values[r_db_port].toInt(&converted, 10);
        if (!converted)
        {
          errorMessage = "invalid port " + values[r_db_port];
          return false;
        }
        streams->selectDbStorage( ident, values[r_name], "QMYSQL3",
                                  values[r_db_host], port, values[r_db_db],
                                  values[r_db_table], values[r_db_login], values[r_db_pass] );
    }
        else
    if ( values[r_type] == "file" )
    {
        streams->selectFileStorage( ident, values[r_name],
                                    QString(values[r_file_file]) );
    }
        else
    if ( values[r_type] == "web" )
    {
    
        if ( values.count() < r_count_web )
        {
          cerr << "Invalid data. Web storage item has "
                << values.count() << " out of " << r_count_web
                << " properties." << endl;
          errorMessage = "Invalid database reference"; 
          return false;
        }
        
        streams->selectWebStorage ( ident, values[r_name], values[r_web_url],
                                    values[r_web_login], values[r_web_pass]);
    }
    
    if (!streams) errorMessage = "could not open storage " + values[r_name];
    
    return (streams);
}

bool ReposStorage::getStorageValuesByName(ValueList& values, const QString& storageName)
{
    resetRecordList();
    while ( getNextRecord(values) &&
            values[r_name] != storageName ) {}
            
    //if (values.count() >= r_count_file && values[r_name] == storageName)
      //cerr << "found " << values[r_name] << endl;
      
    return (values.count() >= r_count_file && values[r_name] == storageName);
}

ReposStorage::ReposStorage() : Storage("", "", 2, 2, 2)
{
    QString error;
    QString source;
    QString dest;

    oldParserMoved = false; // parser to .preVERSION backup
    
    QString home = getenv("HOME");
    QFile file( home + "/."SUBPATH"/storages.res" );

    if ( !file.exists() )
    {
      // create default path/file in homedir
      QDir dir;
      dir.mkdir(dir.homeDirPath() + "/."SUBPATH, true);

      // create default storages file
      dest = home + "/."SUBPATH"/storages.res";
      if ( !createStorageRepos(dest, home) )
      {
          cerr << "error: cannot create " << dest << endl;
          exit(-1);
      }

      // if no streams.res in home path, copy default streams file
      source = QString(PREFIX"/share/"SUBPATH"/streams.res");
      dest   = home + "/."SUBPATH"/streams.res";

      file.setName( dest );
      if ( !file.exists() && !copyFile(source, dest) )
      {
          cerr << "error: cannot copy file " << source << " to " << dest << endl;
          exit(-1);
      }

    }

}

bool ReposStorage::openRepository()
{
    QString error;
    QString home = getenv("HOME");
    QFile file( home + "/."SUBPATH"/storages.res" );

    // load storages file and get selected stream storage
    if ( !selectFileStorage(0, "storage", home + "/."SUBPATH"/storages.res" ) )
    {
      cerr << TARGET": cannot open default storage file $HOME/."SUBPATH"/storages.res" << endl;
      exit( -1 );
    }
    
    if ( !loadList(0, error) )
    {
        cerr << error << endl;
        return false;
    }
      else 
    {
      resetRecordList();
      return true;
    }

}

bool ReposStorage::getDefaultRepository(ValueList& initValues)
{
    int port;
    bool converted;

    bool loaded = false;

    if ( !openRepository() ) 
      return false;
    
    ValueList values(9);
    while (getNextRecord(values) && values[0] != "*") {}

    if (values[0] == "*" && values.count() > 3)
    {
        loaded = true;

        if ( values[1] == "database" )
        {
            if ( values.count() < 9 )
            {
              cerr << "Invalid data. Database storage item has "
                  << values.count() << " out of 9 properties." << endl;
              loaded = false;
            }

            port = values[4].toInt(&converted, 10);
            if (!converted)
            {
              cerr << "invalid port " << values[4] << endl;
              loaded = false;
            }

            if (loaded)
              initValues = values;
        }

        else

        if ( values[1] == "file" )
           initValues = values;

        else

        if ( values[1] == "web" )
           initValues = values;
    }

    return loaded;
}


